from cStringIO import StringIO

from twisted.internet.defer import fail, succeed
from twisted.web.error import Error

from juju.errors import FileNotFound
from juju.lib.testing import TestCase
from juju.providers.orchestra import MachineProvider


def get_file_storage(custom_config=None):
    config = {"orchestra-server": "somewhereel.se",
              "orchestra-user": "user",
              "orchestra-pass": "pass",
              "acquired-mgmt-class": "acquired",
              "available-mgmt-class": "available"}
    if custom_config is None:
        config["storage-url"] = "http://somewhe.re/webdav"
    else:
        config.update(custom_config)
    provider = MachineProvider("blah", config)
    return provider.get_file_storage()


class FileStorageTest(TestCase):

    def test_get_works_no_storage_url(self):
        getPage = self.mocker.replace("twisted.web.client.getPage")
        getPage("http://somewhereel.se/webdav/rubber/chicken")
        self.mocker.result(succeed("pulley"))
        self.mocker.replay()

        fs = get_file_storage({})
        d = fs.get("rubber/chicken")

        def verify(result):
            self.assertEquals(result.read(), "pulley")
        d.addCallback(verify)
        return d

    def test_get_works(self):
        getPage = self.mocker.replace("twisted.web.client.getPage")
        getPage("http://somewhe.re/webdav/rubber/chicken")
        self.mocker.result(succeed("pulley"))
        self.mocker.replay()

        fs = get_file_storage()
        d = fs.get("rubber/chicken")

        def verify(result):
            self.assertEquals(result.read(), "pulley")
        d.addCallback(verify)
        return d

    def test_get_fails(self):
        getPage = self.mocker.replace("twisted.web.client.getPage")
        getPage("http://somewhe.re/webdav/rubber/chicken")
        self.mocker.result(fail(Error("404")))
        self.mocker.replay()

        fs = get_file_storage()
        d = fs.get("rubber/chicken")
        self.assertFailure(d, FileNotFound)
        return d

    def test_get_errors(self):
        getPage = self.mocker.replace("twisted.web.client.getPage")
        getPage("http://somewhe.re/webdav/rubber/chicken")
        self.mocker.result(fail(Error("500")))
        self.mocker.replay()

        fs = get_file_storage()
        d = fs.get("rubber/chicken")
        self.assertFailure(d, Error)
        return d

    def test_get_url(self):
        fs = get_file_storage()
        url = fs.get_url("rubber/chicken")
        self.assertEqual(url, "http://somewhe.re/webdav/rubber/chicken")

    def test_get_url_unicode(self):
        fs = get_file_storage({"storage-url": u"http://\u2666.co.\u2660"})
        url = fs.get_url(u"rubber/\u2665/chicken")
        self.assertEqual(
            url, "http://xn--h6h.co.xn--b6h/rubber/%E2%99%A5/chicken")
        self.assertInstance(url, str)

    def test_put_works(self):
        getPage = self.mocker.replace("twisted.web.client.getPage")
        getPage("http://somewhe.re/webdav/rubber/chicken",
                method="PUT", postdata="pulley")
        self.mocker.result(succeed(None))
        self.mocker.replay()

        fs = get_file_storage()
        d = fs.put("rubber/chicken", StringIO("pulley"))

        def verify(result):
            self.assertEquals(result, True)
        d.addCallback(verify)
        return d

    def test_put_works_no_storage_url(self):
        getPage = self.mocker.replace("twisted.web.client.getPage")
        getPage("http://somewhereel.se/webdav/rubber/chicken",
                method="PUT", postdata="pulley")
        self.mocker.result(succeed(None))
        self.mocker.replay()

        fs = get_file_storage({})
        d = fs.put("rubber/chicken", StringIO("pulley"))

        def verify(result):
            self.assertEquals(result, True)
        d.addCallback(verify)
        return d

    def test_put_handles_204(self):
        """If we're overwriting instead of creating, we get 204 instead of 200
        """
        getPage = self.mocker.replace("twisted.web.client.getPage")
        getPage("http://somewhe.re/webdav/rubber/chicken",
                method="PUT", postdata="pulley")
        self.mocker.result(fail(Error("204")))
        self.mocker.replay()

        fs = get_file_storage()
        d = fs.put("rubber/chicken", StringIO("pulley"))

        def verify(result):
            self.assertEquals(result, True)
        d.addCallback(verify)
        return d

    def test_put_errors(self):
        getPage = self.mocker.replace("twisted.web.client.getPage")
        getPage("http://somewhe.re/webdav/rubber/chicken",
                method="PUT", postdata="pulley")
        self.mocker.result(fail(Error("500")))
        self.mocker.replay()

        fs = get_file_storage()
        d = fs.put("rubber/chicken", StringIO("pulley"))
        self.assertFailure(d, Error)
        return d
