from base64 import b64decode
import os
import re
from xmlrpclib import Fault
from yaml import dump, load

from twisted.internet.defer import fail, succeed
from twisted.web.error import Error
from twisted.web.xmlrpc import Proxy

from juju.lib.mocker import MATCH
from juju.providers.orchestra import MachineProvider

DATA_DIR = os.path.join(os.path.abspath(os.path.dirname(__file__)), "data")

CONFIG = {"type": "orchestra",
          "juju-origin": "distro",
          "orchestra-server": "somewhe.re",
          "orchestra-user": "user",
          "orchestra-pass": "pass",
          "acquired-mgmt-class": "acquired",
          "available-mgmt-class": "available",
          "admin-secret": "SEEKRIT",
          "storage-url": "http://somewhe.re/webdav",
          "authorized-keys": "this-is-a-public-key"}


def _extract_user_data(ks_meta_leftover):
    encoded_re = re.compile('^USER_DATA_BASE64="([^ ]*)"$')
    encoded = encoded_re.match(ks_meta_leftover).group(1)
    return load(b64decode(encoded))


class OrchestraTestMixin(object):

    def get_provider(self):
        return MachineProvider("tetrascape", CONFIG)

    def setup_mocks(self):
        self.proxy_m = self.mocker.mock(Proxy)
        Proxy_m = self.mocker.replace(Proxy, spec=None)
        Proxy_m("http://somewhe.re/cobbler_api")
        self.mocker.result(self.proxy_m)
        self.getPage = self.mocker.replace("twisted.web.client.getPage")

    def mock_find_zookeepers(self, existing=None):
        self.getPage("http://somewhe.re/webdav/provider-state")

        if existing is None:
            self.mocker.result(fail(Error("404")))
        else:
            uid, name = existing
            self.mocker.result(succeed(dump(
                {"zookeeper-instances": [uid]})))
            self.mock_describe_systems(succeed([{
                "uid": uid, "name": name, "mgmt_classes": ["acquired"]}]))

    def mock_get_systems(self, acceptable=True, unacceptable=True):
        self.proxy_m.callRemote("find_system", {"netboot_enabled": "true",
                                                "mgmt_classes": "available"})
        systems = []
        if unacceptable:
            systems.append("bad-system")
        if acceptable:
            systems.append("good-system")
        self.mocker.result(systems)

        if unacceptable:
            self.proxy_m.callRemote("get_system", "bad-system")
            self.mocker.result({"mgmt_classes": ["available", "acquired"],
                                "uid": "bad-system-uid"})
        if acceptable:
            self.proxy_m.callRemote("get_system", "good-system")
            self.mocker.result({"mgmt_classes": ["preserve_me", "available"],
                                "uid": "winston-uid"})

    def mock_acquire_system(self, unexpected_auth_error=None):
        self.proxy_m.callRemote("find_system", {"uid": "winston-uid"})
        self.mocker.result(succeed(["winston"]))
        self.proxy_m.callRemote("get_system_handle", "winston", "")
        if unexpected_auth_error is not None:
            self.mocker.result(fail(unexpected_auth_error))
            return
        self.mocker.result(fail(Fault("blah", "invalid token")))
        self.proxy_m.callRemote("login", "user", "pass")
        self.mocker.result(succeed("TOKEN"))
        self.proxy_m.callRemote("get_system_handle", "winston", "TOKEN")
        self.mocker.result(succeed("smith"))
        self.proxy_m.callRemote(
            "modify_system", "smith", "mgmt_classes",
            ["preserve_me", "acquired"], "TOKEN")
        self.mocker.result(succeed(True))
        self.proxy_m.callRemote("save_system", "smith", "TOKEN")
        self.mocker.result(succeed(True))

    def get_verify_ks_meta(self, machine_id, user_data_filename):

        def verify(ks_meta):
            seek = '([ ]?MACHINE_ID="%s"[ ]?)' % machine_id
            machine_id_re = re.compile(seek)
            self.assertTrue(machine_id_re.search(ks_meta))
            leftover = machine_id_re.sub("", ks_meta)
            user_data = _extract_user_data(leftover)

            expect_path = os.path.join(DATA_DIR, user_data_filename)
            with open(expect_path) as f:
                expect_user_data = load(f.read())
            self.assertEquals(user_data, expect_user_data)
            return True
        return verify

    def mock_set_ks_meta(
            self, verify_ks_meta, fail_modify=False, fail_save=False):
        """Mock setting the kickstart data"""
        self.proxy_m.callRemote("find_system", {"uid": "winston-uid"})
        self.mocker.result(succeed(["winston"]))
        self.proxy_m.callRemote("get_system_handle", "winston", "TOKEN")
        self.mocker.result(succeed("smith"))

        match_ks_meta = MATCH(verify_ks_meta)
        self.proxy_m.callRemote(
            "modify_system", "smith", "ks_meta", match_ks_meta, "TOKEN")
        if fail_modify:
            self.mocker.result(succeed(False))
            return
        self.mocker.result(succeed(True))
        self.proxy_m.callRemote("save_system", "smith", "TOKEN")
        if fail_save:
            self.mocker.result(succeed(False))
            return
        self.mocker.result(succeed(True))

    def mock_start_system(self):
        self.proxy_m.callRemote("find_system", {"uid": "winston-uid"})
        self.mocker.result(succeed(["winston"]))
        self.proxy_m.callRemote("get_system_handle", "winston", "TOKEN")
        self.mocker.result(succeed("smith"))
        self.proxy_m.callRemote(
            "modify_system", "smith", "netboot_enabled", True, "TOKEN")
        self.mocker.result(succeed(True))
        self.proxy_m.callRemote("save_system", "smith", "TOKEN")
        self.mocker.result(succeed(True))
        self.proxy_m.callRemote(
            "background_power_system",
            {"power": "on", "systems": ["winston"]},
            "TOKEN")
        self.mocker.result(succeed("[some-timestamp]_power"))

    def mock_describe_systems(self, result):
        self.proxy_m.callRemote("get_systems")
        self.mocker.result(result)
