from cStringIO import StringIO
import urllib
import urlparse

from twisted.web.client import getPage
from twisted.web.error import Error

from juju.errors import FileNotFound


class FileStorage(object):
    """A WebDAV-backed :class:`FileStorage` abstraction"""

    def __init__(self, base_url):
        self._base_url = base_url

    def get_url(self, name):
        """Return a URL that can be used to access a stored file.

        :param unicode name: the file path for which to provide a URL

        :return: a URL
        :rtype: str
        """
        url = u"/".join((self._base_url, name))
        # query and fragment are irrelevant to our purposes
        scheme, netloc, path = urlparse.urlsplit(url)[:3]
        return urlparse.urlunsplit((
            str(scheme),
            netloc.encode("idna"),
            urllib.quote(path.encode("utf-8")),
            "", ""))

    def get(self, name):
        """Get a file object from the Orchestra WebDAV server.

        :param unicode name: path to for the desired file

        :return: an open file object
        :rtype: :class:`twisted.internet.defer.Deferred`

        :raises: :exc:`juju.errors.FileNotFound` if the file doesn't exist
        """
        url = self.get_url(name)
        d = getPage(url)
        d.addCallback(StringIO)

        def convert_404(failure):
            failure.trap(Error)
            if failure.value.status == "404":
                raise FileNotFound(url)
            return failure
        d.addErrback(convert_404)
        return d

    def put(self, remote_path, file_object):
        """Upload a file to S3.

        :param unicode remote_path: path on which to store the content

        :param file_object: open file object containing the content

        :rtype: :class:`twisted.internet.defer.Deferred`
        """
        url = self.get_url(remote_path)
        data = file_object.read()
        d = getPage(url, method="PUT", postdata=data)
        d.addCallback(lambda _: True)

        def accept_204(failure):
            # NOTE: webdav returns 204s when we overwrite
            failure.trap(Error)
            if failure.value.status == "204":
                return True
            return failure
        d.addErrback(accept_204)
        return d
