from yaml import dump

from twisted.internet.defer import inlineCallbacks, succeed, fail

import zookeeper

from txzookeeper import ZookeeperClient
from txzookeeper.client import ConnectionTimeoutException
from txzookeeper.tests.utils import deleteTree

from juju.lib.testing import TestCase

from juju.errors import EnvironmentPending, NoConnection
from juju.state.sshclient import SSHClient
from juju.providers.ec2.tests.common import EC2TestMixin
from juju.tests.common import get_test_zookeeper_address


class EC2ConnectTest(EC2TestMixin, TestCase):

    def mock_find_zookeepers(self, instance):
        self.s3.get_object(self.env_name, "provider-state")
        self.mocker.result(succeed(dump(
            {"zookeeper-instances": ["i-foobar"]})))
        self.ec2.describe_instances("i-foobar")
        self.mocker.result(succeed([instance]))

    def mock_connect(self, share, instance, result):
        self.mock_find_zookeepers(instance)
        client = self.mocker.patch(SSHClient)
        client.connect("foo.example.com:2181", timeout=30, share=share)
        self.mocker.result(result)

    def assert_connect_error(self, error, expect_type, expect_message):
        instance = self.get_instance("i-foobar", dns_name="foo.example.com")
        self.mock_connect(False, instance, fail(error))
        self.mocker.replay()

        provider = self.get_provider()
        d = provider.connect()

        def check_error(error):
            self.assertEqual(str(error), expect_message)
        self.assertFailure(d, expect_type)
        d.addCallback(check_error)
        return d

    def test_no_dns_name(self):
        """
        `EnvironmentPending` should be raised if no zookeeper nodes have dns
        names
        """
        instance = self.get_instance("i-foobar")
        self.mock_find_zookeepers(instance)
        self.mocker.replay()

        provider = self.get_provider()
        d = provider.connect()

        def check_error(error):
            self.assertEqual(
                str(error), "No machines have addresses assigned yet")

        self.assertFailure(d, NoConnection)
        d.addCallback(check_error)
        return d

    def test_provider_connect_forwards_share_option(self):
        """The `share` kwarg should be passed through to `SSHClient.connect`"""
        instance = self.get_instance("i-foobar", dns_name="foo.example.com")
        connected_client = self.mocker.mock(type=SSHClient)
        self.mock_connect(True, instance, succeed(connected_client))
        # We'll test the wait on initialization separately.
        connected_client.exists_and_watch("/initialized")
        self.mocker.result((succeed(True), None))
        self.mocker.replay()

        provider = self.get_provider()
        d = provider.connect(share=True)

        def verify_result(result):
            self.assertIdentical(connected_client, result)
        d.addCallback(verify_result)
        return d

    def test_no_connection(self):
        """`NoConnection` errors should become `EnvironmentPending`s"""
        return self.assert_connect_error(
            NoConnection("KABOOM!"), EnvironmentPending,
            "Cannot connect to machine i-foobar (perhaps still initializing): "
            "KABOOM!")

    def test_txzookeeper_error(self):
        """
        `ConnectionTimeoutException` errors should become `EnvironmentPending`s
        """
        return self.assert_connect_error(
            ConnectionTimeoutException("SPLAT!"), EnvironmentPending,
            "Cannot connect to machine i-foobar (perhaps still initializing): "
            "SPLAT!")

    def test_other_error(self):
        """Other errors should propagate"""
        return self.assert_connect_error(
            TypeError("THUD!"), TypeError, "THUD!")

    @inlineCallbacks
    def test_provider_connect_waits_on_initialization(self):
        """
        A connection to a zookeeper that is running, but whose juju state
        is not ready, should wait until that state is ready.
        """
        # Hand back a real connected client to test the wait on initialization.
        instance = self.get_instance("i-foobar", dns_name="foo.example.com")
        connected_client = ZookeeperClient()
        self.client = connected_client  # for poke_zk
        self.mock_connect(False, instance, succeed(connected_client))

        self.mocker.replay()

        zookeeper.set_debug_level(0)
        yield connected_client.connect(get_test_zookeeper_address())

        client_result = []

        provider = self.get_provider()
        client_deferred = provider.connect()
        client_deferred.addCallback(client_result.append)

        # Give it a chance to do it incorrectly.
        yield self.poke_zk()

        try:
            self.assertEquals(client_result, [])

            yield connected_client.create("/initialized")

            yield client_deferred
            self.assertTrue(client_result, client_result)
            self.assertIdentical(client_result[0], connected_client)
        finally:
            deleteTree("/", connected_client.handle)
            connected_client.close()
