from twisted.internet.defer import fail, inlineCallbacks, succeed

import zookeeper

from txzookeeper import ZookeeperClient
from txzookeeper.client import ConnectionTimeoutException
from txzookeeper.tests.utils import deleteTree

from juju.errors import EnvironmentPending, NoConnection
from juju.lib.testing import TestCase
from juju.machine import ProviderMachine
from juju.providers.common.base import MachineProviderBase
from juju.state.sshclient import SSHClient
from juju.tests.common import get_test_zookeeper_address


class DummyProvider(MachineProviderBase):

    def __init__(self, second_zookeeeper):
        self._second_zookeeper = second_zookeeeper

    def get_zookeeper_machines(self):
        """
        Return a pair of possible zookeepers, the first of which is invalid
        """
        return succeed([
            ProviderMachine("i-havenodns"),
            self._second_zookeeper])


class ConnectTest(TestCase):

    def mock_connect(self, share, result):
        client = self.mocker.patch(SSHClient)
        client.connect("foo.example.com:2181", timeout=30, share=share)
        self.mocker.result(result)

    def assert_connect_error(self, error, expect_type, expect_message):
        self.mock_connect(False, fail(error))
        self.mocker.replay()

        provider = DummyProvider(ProviderMachine("i-exist", "foo.example.com"))
        d = provider.connect()

        def check_error(error):
            self.assertEqual(str(error), expect_message)
        self.assertFailure(d, expect_type)
        d.addCallback(check_error)
        return d

    def test_none_have_dns(self):
        """
        `EnvironmentPending` should be raised if no zookeeper nodes have dns
        names
        """
        provider = DummyProvider(ProviderMachine("i-havenodnseither"))
        d = provider.connect()
        self.assertFailure(d, EnvironmentPending)

        def check(error):
            self.assertEquals(
                str(error), "No machines have addresses assigned yet")
        d.addCallback(check)
        return d

    def test_share_kwarg(self):
        """The `share` kwarg should be passed through to `SSHClient.connect`"""
        client = self.mocker.mock(type=SSHClient)
        self.mock_connect(True, succeed(client))
        client.exists_and_watch("/initialized")
        self.mocker.result((succeed(True), None))
        self.mocker.replay()

        provider = DummyProvider(ProviderMachine("i-amok", "foo.example.com"))
        d = provider.connect(share=True)

        def verify(result):
            self.assertIdentical(result, client)
        d.addCallback(verify)
        return d

    def test_no_connection(self):
        """`NoConnection` errors should become `EnvironmentPending`s"""
        return self.assert_connect_error(
            NoConnection("KABOOM!"), EnvironmentPending,
            "Cannot connect to machine i-exist (perhaps still initializing): "
            "KABOOM!")

    def test_txzookeeper_error(self):
        """
        `ConnectionTimeoutException` errors should become `EnvironmentPending`s
        """
        return self.assert_connect_error(
            ConnectionTimeoutException("SPLAT!"), EnvironmentPending,
            "Cannot connect to machine i-exist (perhaps still initializing): "
            "SPLAT!")

    def test_other_error(self):
        """Other errors should propagate"""
        return self.assert_connect_error(
            TypeError("THUD!"), TypeError, "THUD!")

    @inlineCallbacks
    def test_wait_for_initialize(self):
        """
        A connection to a zookeeper that is running, but whose juju state
        is not ready, should wait until that state is ready.
        """
        client = ZookeeperClient()
        self.client = client # for poke_zk
        self.mock_connect(False, succeed(client))
        self.mocker.replay()

        zookeeper.set_debug_level(0)
        yield client.connect(get_test_zookeeper_address())

        provider = DummyProvider(ProviderMachine("i-amok", "foo.example.com"))
        d = provider.connect()
        client_result = []
        d.addCallback(client_result.append)

        # Give it a chance to do it incorrectly.
        yield self.poke_zk()

        try:
            self.assertEquals(client_result, [])
            yield client.create("/initialized")
            yield d
            self.assertTrue(client_result, client_result)
            self.assertIdentical(client_result[0], client)
        finally:
            deleteTree("/", client.handle)
            client.close()
