#include <Python.h>
#include <datetime.h>

#include <time.h>
#include "pytype_appointment.h"
#include "pytype_basics.h"
#include "structmember.h"
#include "python_compatibility.h"

/*******************************************************
 * Create and delloc methods for objects 
 ******************************************************/

extern PyObject* PyPiAppointment_New(PyTypeObject *type, PyObject *args, PyObject *kwds) {
  int i;
  PyPiAppointment* self;

  /* Why do we have to do this here? The one in the swig init doesn't seem
     to work ?! */
  PyDateTime_IMPORT;
  
  AppointmentType.ob_type = &PyType_Type;
  self = (PyPiAppointment *)type->tp_alloc(type, 0);
  new_Appointment(&(self->a));
  SetBasicRecordObjectAttributeDefaults((PyObject*) self, pack_Appointment);  
  self->filters = NULL;

  return (PyObject*)self;
}

extern int PyPiAppointment_Init(PyObject *self, PyObject *args, PyObject *kwds) {
  PyPiAppointment* fromappt = NULL;
  PyPiAppointment* appt = NULL;
  PyObject* filters = NULL;
  int i;
  
  static char *kwlist[] = {"appt", "record_field_filters", NULL};
  if (!PyArg_ParseTupleAndKeywords(args, kwds, "|OO", kwlist, 
				   &fromappt, &filters)) {
    return -1;
  }
  
  appt = (PyPiAppointment*)self;
  /* we have to support calling __init__ more than once */
  if (appt->filters != NULL) {
    Py_DECREF(appt->filters);
    appt->filters = NULL;
  }
  if (filters != NULL) {
    appt->filters = filters;
    Py_INCREF(filters);
  }
  free_Appointment(&(appt->a));
  if (appt->saved_br.size > 0 && appt->saved_br.buf) {	
    free(appt->saved_br.buf);					
  }

  if ((fromappt == NULL) || ((PyObject *)fromappt == Py_None)) {
    /* Initialise attributes custom to this type of object */
    new_Appointment(&(appt->a));
    SetBasicRecordObjectAttributeDefaults((PyObject*) appt, pack_Appointment);    
  } else {
    if (!PyPiAppointment_Check(fromappt)) {
      PyErr_SetString(PyExc_TypeError,"Must provide an Appointment object to share");
      return -1;
    }
  
    /* copy all the database agnostic record attributes */
    appt->saved_br.size = fromappt->saved_br.size;
    appt->saved_br.attrib = fromappt->saved_br.attrib;
    appt->saved_br.rt = fromappt->saved_br.rt;
    appt->saved_br.unique_id = fromappt->saved_br.unique_id;

    appt->rt = fromappt->rt;
    appt->unique_id = fromappt->unique_id;
    
    appt->saved_br.buf = malloc(fromappt->saved_br.size);
    memcpy(appt->saved_br.buf,
	   fromappt->saved_br.buf,
	   fromappt->saved_br.size);
    
    appt->category = fromappt->category;
    appt->unsaved_changes = fromappt->unsaved_changes;
    appt->deleted = fromappt->deleted;
    appt->modified = fromappt->modified;
    appt->busy = fromappt->busy;
    appt->secret = fromappt->secret;
    
    /* copy appointment, begin, end, alarm, advance, advanceUnits,
       repeatType, repeatForever, repeatFrequency, repeatDay,
       repeatDays, repeatWeekstart and exceptions (note, not
       exception as that is a pointer) */
    memcpy(&(appt->a), &(fromappt->a), sizeof(struct Appointment));
    
    /* Now do the pointers */
    
    pyint_strcpy(appt->a.description, fromappt->a.description);
    pyint_strcpy(appt->a.note, fromappt->a.note);
    appt->a.exception = malloc(sizeof(struct tm) * fromappt->a.exceptions);
    for (i = 0; i < fromappt->a.exceptions; i++) {
      memcpy(&(appt->a.exception[i]), &(fromappt->a.exception[i]), sizeof(struct tm));
    }
  }

  return 0;
}

static PyObject * PyPiAppointment_Allocate(PyTypeObject *type, int nitems) {
  PyPiAppointment *appt;
  if (type == &AppointmentType) {
    appt = PyObject_New(PyPiAppointment, &AppointmentType);
    return (PyObject *) appt;
  } else {
    /* Is this for subclasses ? */
    appt = (PyPiAppointment *)PyType_GenericAlloc(type, nitems);
    return (PyObject*)appt;
  }
}

extern PyObject* PyPiAppointment_Wrap(Appointment_t* a, PCRecType rt, 
				 unsigned int unique_id, unsigned char attrib,
                                int size, void* buf, PyObject *filters) {
  PyPiAppointment* appt;
  int i;

 PyObject *python_mod, *python_mdict, *appt_class, *python_args, *python_kw;

  python_mod = PyImport_Import(PyString_FromString("jppy.jpilot.legacy"));
  if (python_mod == NULL) {
    PyErr_Print();
    return NULL;
  }
  python_mdict = PyModule_GetDict(python_mod);
  if (python_mdict == NULL) {
    PyErr_Print();
    Py_DECREF(python_mod);
    return NULL;
  }  
  Py_INCREF(python_mdict);
  Py_DECREF(python_mod);
  appt_class = PyDict_GetItemString(python_mdict, "Appointment"); /* borrowed reference */
  if (appt_class == NULL) {
    PyErr_Print();
    Py_DECREF(python_mdict);
    return NULL;
  }
  Py_INCREF(appt_class);
  python_args = Py_BuildValue("()");
  python_kw = Py_BuildValue("{s:O}","record_field_filters",filters);
  appt = (PyPiAppointment*) PyObject_Call(appt_class, python_args, python_kw);
  Py_DECREF(appt_class);
  Py_DECREF(python_args);
  Py_DECREF(python_kw);
  if (appt == NULL) {
    PyErr_Print();
    return NULL;
  }
  Py_INCREF(appt);
 
  /* copy indefinite, tm, priority and complete */
  memcpy(&(appt->a), a, sizeof(Appointment_t));


  pyp_strcpy(appt->a.description, a->description);
  pyp_strcpy(appt->a.note, a->note);
  appt->a.exception = malloc(sizeof(struct tm) * a->exceptions);
  for (i = 0; i < a->exceptions; i++) {
    memcpy(&(appt->a.exception[i]), &(a->exception[i]), sizeof(struct tm));
  }

  /* set saved_br stuff, and rt and unique_id, and attrib derived
     details for the current appt */
  SetSavedBrAndRTandUniqueIDandAttribs(rt, unique_id, attrib, size, buf, (PyObject *)appt);

  return (PyObject*)appt;
}

static void PyPiAppointment_Dealloc(PyPiAppointment* self) {
  free_Appointment(&(self->a));
  if (self->filters != NULL) {
    Py_DECREF(self->filters);
    self->filters = NULL;
  }
  if (self->saved_br.size > 0 && self->saved_br.buf) {
    free(self->saved_br.buf);
  }
  self->ob_type->tp_free((PyObject*)self);
}


static int PyPiAppointment_Compare(PyPiAppointment* self,PyPiAppointment *other) {
  time_t s, o;

  s = mktime(&(self->a.begin));
  o = mktime(&(other->a.begin));

  if (s == o) {
    return 0;
  } else if (s > 0) {
    return -1;
  } else {
    return 1;
  };
}

static char *PyPiAppointment_key_list[] = {
  "event", /* boolean */
  "begin",
  "end",
  "alarm",
  "repeatType", /* int */
  "repeatForever", /* boolean */
  "repeatEnd", /* datetime.date representing the end of the repeat */
  "repeatFrequency", /* int */
  "repeatDay", /* int */
  "repeatDays.sunday", /* boolean */
  "repeatDays.monday", /* boolean */
  "repeatDays.tuesday", /* boolean */
  "repeatDays.wednesday", /* boolean */
  "repeatDays.thursday", /* boolean */
  "repeatDays.friday", /* boolean */
  "repeatDays.saturday", /* boolean */
  "repeatWeekstart", /* int */
  "exceptions", /* datetime.date */
  "description",
  "note",
  "final", /* deprecated, calls repeatEnd */ 
  NULL};

static PyObject* PyPiAppointment_keys(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiAppointment_key_list[n]) {
    PyObject *value;
    value = PyString_FromString(PyPiAppointment_key_list[n++]);
    PyList_Append(list, value);
    Py_DECREF(value);
  }
  PyPi_extend_keys_from_filters((PyPiBase*)self, list);

  return list;
}

/* forward declaration */
PyObject *PyPiAppointment_GetItem(PyPiAppointment* self,  PyObject* key);

static PyObject* PyPiAppointment_values(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiAppointment_key_list[n]) {
    PyObject *key;
    PyObject *value;
    key   = PyString_FromString(PyPiAppointment_key_list[n++]);
    value = PyPiAppointment_GetItem((PyPiAppointment *)self, key);
    PyList_Append(list, value);
    Py_DECREF(key);
    Py_DECREF(value);
  }
  return list;
}

static PyObject* PyPiAppointment_items(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiAppointment_key_list[n]) {
    PyObject *key, *value, *tuple;
    key = PyString_FromString(PyPiAppointment_key_list[n++]);
    value = PyPiAppointment_GetItem((PyPiAppointment *)self, key);
    tuple = Py_BuildValue("(OO)", key, value);
    PyList_Append(list, tuple); /* get it's own ref */
    Py_DECREF(key);
    Py_DECREF(value);
    Py_DECREF(tuple);
  }
  return list;
}

/*******************************************************
 * 
 ******************************************************/

static PyMethodDef PyPiAppointment_Methods[] = {
  { "keys", (PyCFunction)PyPiAppointment_keys, METH_NOARGS, "Return a list of available keys"},
  { "items",(PyCFunction)PyPiAppointment_items, METH_NOARGS, "Return a list of available items"},
  { "values",(PyCFunction)PyPiAppointment_values, METH_NOARGS, "Return a list of available items"},
  {NULL,NULL,0,NULL} /* Sentinel */
};

static PyMemberDef PyPiAppointment_Members[] = {
  PYPI_MEMBERS_HEAD,
  {NULL}  /* Sentinel */
};

static PyGetSetDef PyPiAppointment_Getseters[] = {
  PYPI_GETSETERS_HEAD,
  {NULL}  /* Sentinel */
};

/**** mapping interface ****/
int PyPiAppointment_Len(PyObject* self) {
  int len=0;
  // quite expensive way to do it, but we need to get filters too
  PyObject *keys = PyPiAppointment_keys(self);
  len = PySequence_Size(keys);
  Py_DECREF(keys);
  return len;
}

PyObject *PyPiAppointment_GetItem(PyPiAppointment* self,  PyObject* key) {
  char *keystring;
  PyObject *result;

  if (!PyString_Check(key)) {
    Py_INCREF(Py_None);
    return Py_None;  
  }

  if ((result = PyPi_GetItem_from_filters((PyPiBase *)self, key)) != NULL)
    return result;
  else if (PyErr_Occurred() != NULL)
    return NULL;

  Py_INCREF(key);
  keystring = PyString_AsString(key);

  GET_BOOL_ATTR(keystring, "event", a.event);
  GET_INT_ATTR(keystring, "repeatType", a.repeatType);
  GET_BOOL_ATTR(keystring, "repeatForever", a.repeatForever);
  GET_INT_ATTR(keystring, "repeatFrequency", a.repeatFrequency);
  GET_INT_ATTR(keystring, "repeatDay", a.repeatDay);
  GET_BOOL_ATTR(keystring, "repeatDays.sunday", a.repeatDays[0]);
  GET_BOOL_ATTR(keystring, "repeatDays.monday", a.repeatDays[1]);
  GET_BOOL_ATTR(keystring, "repeatDays.tuesday", a.repeatDays[2]);
  GET_BOOL_ATTR(keystring, "repeatDays.wednesday", a.repeatDays[3]);
  GET_BOOL_ATTR(keystring, "repeatDays.thursday", a.repeatDays[4]);
  GET_BOOL_ATTR(keystring, "repeatDays.friday", a.repeatDays[5]);
  GET_BOOL_ATTR(keystring, "repeatDays.saturday", a.repeatDays[6]);
  GET_INT_ATTR(keystring, "repeatWeekstart", a.repeatWeekstart);
  GET_STRING_ATTR(keystring,"description", a.description);
  GET_STRING_ATTR(keystring,"note", a.note);

  if (strcasecmp(keystring,"alarm") == 0) {
    PyObject *units;
    Py_DECREF(key);
    if (!(self->a.alarm)) {
      Py_INCREF(Py_None);
      return Py_None;  
    }
    if (self->a.advanceUnits == advMinutes) {
      return PyDelta_FromDSU(0,self->a.advance * 60,0);
    } else if (self->a.advanceUnits == advHours) {
      return PyDelta_FromDSU(0,self->a.advance * 60 * 60,0);
    } else {
      return PyDelta_FromDSU(self->a.advance,0,0);
    }
  } else if (strcasecmp(keystring,"exceptions") == 0) {
    PyObject *list;
    int i;
    Py_DECREF(key);
    list = PyList_New(self->a.exceptions);    
    for (i = 0; i < self->a.exceptions; i++) {
       PyList_SetItem(list,i,PyDate_FromDate(self->a.exception[i].tm_year + 1900,
					     self->a.exception[i].tm_mon + 1,
					     self->a.exception[i].tm_mday));
    }
    return list;
  } else if (strcasecmp(keystring,"begin") == 0) {
    Py_DECREF(key);
    if (self->a.event) {
      return PyDate_FromDate(self->a.begin.tm_year + 1900,
			     self->a.begin.tm_mon + 1,
			     self->a.begin.tm_mday);
    } else {
      PyObject *pydatetime;
      pydatetime = PyDateTime_FromDateAndTime(self->a.begin.tm_year + 1900,
					      self->a.begin.tm_mon + 1,
					      self->a.begin.tm_mday,
					      self->a.begin.tm_hour,
					      self->a.begin.tm_min,
					      self->a.begin.tm_sec,
					      0);
      return pydatetime;
    }
  } else if (strcasecmp(keystring,"end") == 0) {
    Py_DECREF(key);
    if (self->a.event) {
      return PyDate_FromDate(self->a.end.tm_year + 1900,
			     self->a.end.tm_mon + 1,
			     self->a.end.tm_mday);
    } else {
      PyObject *pydatetime;
      pydatetime = PyDateTime_FromDateAndTime(self->a.end.tm_year + 1900,
					      self->a.end.tm_mon + 1,
					      self->a.end.tm_mday,
					      self->a.end.tm_hour,
					      self->a.end.tm_min,
					      self->a.end.tm_sec,
					      0);
      return pydatetime;
    }
  } else if (strcasecmp(keystring,"final") == 0
             || strcasecmp(keystring, "repeatEnd") == 0) {
    Py_DECREF(key);
    if (self->a.repeatForever) {
      Py_INCREF(Py_None);
      return Py_None;
    } else {
      return PyDate_FromDate(self->a.repeatEnd.tm_year + 1900,
			     self->a.repeatEnd.tm_mon + 1,
			     self->a.repeatEnd.tm_mday);
    }
  } else {
    PyErr_Format(PyExc_KeyError,"no such key '%s'", keystring);
    Py_DECREF(key);
    return NULL;
  }
}

int PyPiAppointment_SetItem(PyPiAppointment* self, PyObject* key, PyObject* value) {
  char buf[255];
  char *keystring;

  if (!PyString_Check(key)) {
    PyErr_SetString(PyExc_TypeError,"key must be a String");
    return -1;
  }
  
  if (PyPi_SetItem_from_filters((PyPiBase *)self, key, value) > 0)
    return 0;
  else if (PyErr_Occurred() != NULL)
    return -1;

  Py_INCREF(key);
  keystring = PyString_AsString(key);

  if (value == NULL) {
    PyErr_Format(PyExc_ValueError,"Can't delete value %s", keystring);
    return -1;
  }
  
  SET_STRING_ATTR(keystring,"description",a.description,value, 256);
  SET_STRING_ATTR(keystring,"note",a.note,value, 4096);

  PyErr_SetString(PyExc_KeyError,"no such key");
  Py_DECREF(key);
  return -1;
}

static PyMappingMethods PyPiAppointment_Mapping = {
  (lenfunc)PyPiAppointment_Len,
  (binaryfunc)PyPiAppointment_GetItem, 
  (objobjargproc)PyPiAppointment_SetItem,
};

/*******************************************************
 * Provide a repr method
 ******************************************************/
static PyObject *PyPiAppointment_Repr(PyPiAppointment* self) {
  static PyObject *format = NULL;
  PyObject *attrib, *args, *result;
  int len1;
  char time[100];

  if (format == NULL) {
    format = PyString_FromString("<%s '%s' %r %s>");
    if (format == NULL)
      return NULL;
  }

  if (self->a.description) {
    len1 = strlen(self->a.description) > 25 ? 25 : strlen(self->a.description);
  } else {
    len1 = 0;
  }
  
  if (self->a.event) {
    strftime(time, 99, "%x", &(self->a.begin));
  } else {
    strftime(time, 99, "%c", &(self->a.begin));
  }

  args = Py_BuildValue("ss#sO", 
		       (self->ob_type)->tp_name,
		       self->a.description,
		       len1,
		       time,
		       Attribute_Repr((PyObject *)self));

  if (args == NULL)
    return NULL;

  result = PyString_Format(format, args);
  Py_DECREF(args);
  return result;
}


/*******************************************************
 * Declare the type
 ******************************************************/


PyTypeObject AppointmentType = {
  PyObject_HEAD_INIT(NULL)
  0,
  "jppy._jpilot.__jpilot.Appointment",
  sizeof(PyPiAppointment),
  0,
  (destructor)PyPiAppointment_Dealloc,      /*tp_dealloc*/
  0,                                /*tp_print*/
  0, /*tp_getattr*/
  0, /*tp_setattr*/
  (cmpfunc)PyPiAppointment_Compare,     /*tp_compare*/
  (reprfunc)PyPiAppointment_Repr,       /*tp_repr*/
  0,                                /*tp_as_number*/
  0,                                /*tp_as_sequence*/
  &PyPiAppointment_Mapping,                                /*tp_as_mapping*/
  0,                                /*tp_hash */
  0,                         /*tp_call*/
  0,                         /*tp_str*/
  0,                         /*tp_getattro*/
  0,                         /*tp_setattro*/
  0,                         /*tp_as_buffer*/
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
  "Appt objects",           /* tp_doc */
  0,		               /* tp_traverse */
  0,		               /* tp_clear */
  0,		               /* tp_richcompare */
  0,		               /* tp_weaklistoffset */
  0,		               /* tp_iter */
  0,		               /* tp_iternext */
  PyPiAppointment_Methods,             /* tp_methods */
  PyPiAppointment_Members,            /* tp_members */
  PyPiAppointment_Getseters,          /* tp_getset */
  0,                         /* tp_base */
  0,                         /* tp_dict */
  0,                         /* tp_descr_get */
  0,                         /* tp_descr_set */
  0,                         /* tp_dictoffset */
  (initproc)PyPiAppointment_Init,      /* tp_init */
  (allocfunc)PyPiAppointment_Allocate,                 /* tp_alloc */
  (newfunc)PyPiAppointment_New,                 /* tp_new */
  0, /* Low-level free-memory routine */
};
