package org.openstreetmap.josm.plugins.measurement;

import static org.openstreetmap.josm.tools.I18n.tr;

import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.Collection;

import javax.swing.AbstractAction;
import javax.swing.JLabel;
import javax.swing.JPanel;

import org.openstreetmap.josm.data.SelectionChangedListener;
import org.openstreetmap.josm.data.osm.DataSet;
import org.openstreetmap.josm.data.osm.Node;
import org.openstreetmap.josm.data.osm.OsmPrimitive;
import org.openstreetmap.josm.data.osm.Way;
import org.openstreetmap.josm.gui.NavigatableComponent;
import org.openstreetmap.josm.gui.SideButton;
import org.openstreetmap.josm.gui.dialogs.ToggleDialog;
import org.openstreetmap.josm.gui.help.HelpUtil;
import org.openstreetmap.josm.tools.ImageProvider;
import org.openstreetmap.josm.tools.Shortcut;

/**
 * A small tool dialog for displaying the current measurement data.
 *
 * @author ramack
 */
public class MeasurementDialog extends ToggleDialog implements SelectionChangedListener {
    private static final long serialVersionUID = 4708541586297950021L;

    /**
     * The reset button
     */
    private SideButton resetButton;

    /**
     * The measurement label for the path length
     */
    protected JLabel pathLengthLabel;

    /**
     * The measurement label for the currently selected segments
     */
    protected JLabel selectLengthLabel;

    /**
     * The measurement label for area of the currently selected loop
     */
    protected JLabel selectAreaLabel;

    /**
     * The measurement label for the segment angle, actually updated, if 2 nodes are selected
     */
    protected JLabel segAngleLabel;
    
    /**
     * Constructor
     */
    public MeasurementDialog()
    {
        super(tr("Measured values"), "measure", tr("Open the measurement window."),
        Shortcut.registerShortcut("subwindow:measurement", tr("Toggle: {0}", tr("Measured values")),
        KeyEvent.VK_U, Shortcut.CTRL_SHIFT), 150);

        resetButton = new SideButton(new AbstractAction() {
        	{
        		putValue(NAME, tr("Reset"));
        		putValue(SMALL_ICON,ImageProvider.get("dialogs", "select"));
        		putValue(SHORT_DESCRIPTION, tr("Reset current measurement results and delete measurement path."));
        		putValue("help", HelpUtil.ht("/Dialog/Measurement#Reset"));
        	}
            @Override
            public void actionPerformed(ActionEvent e)
            {
            	resetValues();
            }
        });

        JPanel valuePanel = new JPanel(new GridLayout(0,2));

        valuePanel.add(new JLabel(tr("Path Length")));

        pathLengthLabel = new JLabel(NavigatableComponent.getDistText(0));
        valuePanel.add(pathLengthLabel);

        valuePanel.add(new JLabel(tr("Selection Length")));

        selectLengthLabel = new JLabel(NavigatableComponent.getDistText(0));
        valuePanel.add(selectLengthLabel);

        valuePanel.add(new JLabel(tr("Selection Area")));

        selectAreaLabel = new JLabel(NavigatableComponent.getAreaText(0));
        valuePanel.add(selectAreaLabel);

        JLabel angle = new JLabel(tr("Angle"));
        angle.setToolTipText(tr("Angle between two selected Nodes"));
        valuePanel.add(angle);

        segAngleLabel = new JLabel("- \u00b0");
        valuePanel.add(segAngleLabel);

        this.setPreferredSize(new Dimension(0, 92));

        createLayout(valuePanel, false, Arrays.asList(new SideButton[] {
            resetButton
        }));
        
        DataSet.addSelectionListener(this);
    }

    /**
     * Cleans the active Meausurement Layer
     */
    public void resetValues(){
        MeasurementPlugin.getCurrentLayer().reset();
    }

	@Override
	public void selectionChanged(Collection<? extends OsmPrimitive> newSelection) {
        double length = 0.0;
        double segAngle = 0.0;
        double area = 0.0;
        Node lastNode = null;
        for (OsmPrimitive p : newSelection) {
            // ignore incomplete nodes
            if (p instanceof Node && !((Node)p).isIncomplete()) {
                Node n =(Node)p;
                if (lastNode == null) {
                    lastNode = n;
                } else {
                    length += lastNode.getCoor().greatCircleDistance(n.getCoor());
                    segAngle = MeasurementLayer.angleBetween(lastNode.getCoor(), n.getCoor());
                    lastNode = n;
                }
            } else if (p instanceof Way) {
                Way w = (Way)p;
                Node lastN = null;
                for (Node n: w.getNodes()) {
                    if (lastN != null && lastN.getCoor() != null && n.getCoor() != null) {
                        length += lastN.getCoor().greatCircleDistance(n.getCoor());
                        //http://local.wasp.uwa.edu.au/~pbourke/geometry/polyarea/
                        area += (MeasurementLayer.calcX(n.getCoor()) * MeasurementLayer.calcY(lastN.getCoor()))
                        - (MeasurementLayer.calcY(n.getCoor()) * MeasurementLayer.calcX(lastN.getCoor()));
                        segAngle = MeasurementLayer.angleBetween(lastN.getCoor(), n.getCoor());
                    }
                    lastN = n;
                }
                if (lastN != null && lastN == w.getNodes().iterator().next())
                    area = Math.abs(area / 2);
                else
                    area = 0;
            }
        }
        selectLengthLabel.setText(NavigatableComponent.getDistText(length));
        segAngleLabel.setText(new DecimalFormat("#0.0").format(segAngle) + " \u00b0");
        selectAreaLabel.setText(NavigatableComponent.getAreaText(area));
	}

	/* (non-Javadoc)
	 * @see org.openstreetmap.josm.gui.dialogs.ToggleDialog#destroy()
	 */
	@Override
	public void destroy() {
		super.destroy();
		DataSet.removeSelectionListener(this);
	}
}
