// $Id: Command.java,v 1.2 2001/12/07 11:41:24 ramsdell Exp $

// A command specified by a rule.

/*
 * Copyright 1997 by John D. Ramsdell
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package edu.neu.ccs.jmk;

/**
 * A command specified by a rule.
 * Each command has an operator which is applied to an array of
 * strings generated by a function.  To produce the strings, the
 * function is applied to the variables set when a rule is invoked.
 * @version May 1999
 * @author John D. Ramsdell
 */
final class Command
{
  private Operator operator;
  private Function operands;	// A function that produces a string list
  private boolean ignore = false; // Ignore errors?

  /**
   * Creates a command from an operator and a function.
   */
  Command(Operator operator, Function operands) {
    this.operator = operator;
    this.operands = operands;
  }

  void setIgnore(boolean ignore) {
    this.ignore = ignore;
  }

  /**
   * Runs a command.
   * It produces an argument list from the operands and then applies
   * the operator to the arguments.  Each command variable is
   * replaced by its value.
   * @param target the origin for the value of '@' command variable
   * @param first the origin for the value of '&lt;' command variable
   * @param newer the origin for the value of '?' command variable
   * @param match the origin for the value of '%' command variable
   * @param justPrinting if true, print but do not run commands
   * @param out the place to write messages
   */
  void run(Rule target, Rule first, Rule[] newer, String match,
	   boolean justPrinting, java.io.PrintWriter out)
       throws CommandFailedException
  {
    // Construct the function's arguments
    StringList at = new StringList(target.getTarget());
    StringList lt = null;
    if (first != null)
      lt = new StringList(first.getTarget());
    StringList qm = null;
    for (int i = newer.length - 1; i >= 0; i--)
      qm = new StringList(newer[i].getTarget(), qm);
    StringList pc = null;
    if (match != null)
      pc = new StringList(match);
    // This must match Command.commandArgs
    Value[] params = new Value [] { at, lt, qm, pc };

    // Generate operands
    Value result = null;
    try {
      result = operands.invoke(params, null);
    }
    catch (Exception ex) {
      String msg = "Command failed: " + ex.getMessage();
      throw new CommandFailedException(msg);
    }

    if (!StringList.isStringList(result)) {
      String msg
	= "Command failed: Cannot convert operands to a list of strings";
      throw new CommandFailedException(msg);
    }

    // Convert operand into a localized array of strings
    StringList sl = (StringList)result;
    int len = StringList.length(sl);
    String[] args = new String[len];
    for (int i = 0; sl != null; sl = sl.getRest())
      args[i++] = StringUtils.localizePaths(sl.getString());

    // Print the command after command variable substitution.
    out.print(operator.getName());
    for (int i = 0; i < args.length; i++)
      out.print(" " + args[i]);
    out.println();

    // Run the command.
    try {
      if (!justPrinting)
	operator.exec(args, out);
    }
    catch (CommandFailedException ex) {
      if (ignore)
	out.println("Command failed: " + ex.getMessage() + " (ignored)");
      else
	throw ex;
    }
    catch (Throwable t) {
      throw new CommandFailedException(t.toString());
    }
  }

  /*
   * argument list used by the loader.
   */
  final static String[] commandArgs = {"@", "<", "?", "%"};
}
