/*************************************************************************
 *
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_mplayer.c,v $
 * $Id: mod_mplayer.c,v 1.18 2004/02/27 19:14:28 boucman Exp $
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include "config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3config.h"
#include "irmp3mod.h"
#include "mod_mplayer.h"

/*
 * todo
 * bugreport: position values are messed up when seeking in wma
 * add feature : mplayer can't report audio file length
 * */


/*************************************************************************
 * GLOBALS
 */
int mod_mplayer_pid    = 0;	// PID of mplayer
int mod_mplayer_fdsend = 0;	// fd to send commands to mplayer
int mod_mplayer_fdrecv = 0;	// fd to get responses from mplayer
int mod_mplayer_fderror = 0;	// fd to get responses from mplayer
static char *artist = NULL;
static char *title = NULL;
static char *album = NULL;
static char *year = NULL;
static char *genre = NULL;
static char *comment = NULL;
#define MPLAYER_STATUS_HALT	0
#define MPLAYER_STATUS_STOP	1
#define MPLAYER_STATUS_PLAY	2
#define MPLAYER_STATUS_PAUSE	3
int  mod_mplayer_status = MPLAYER_STATUS_HALT;
int mod_mplayer_song_length;
int mod_mplayer_song_pos;
fd_set mod_mplayer_fdset;

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_mplayer = {
	"mod_mplayer",
	mod_mplayer_deinit,	// deinit
	NULL,			// reload
	&mod_mplayer_fdset,	// watchfd_set
	NULL,			// poll
	NULL,			// update
	mod_mplayer_message,	// message
	mod_mplayer_chld,	// SIGCHLD handler
	mod_mplayer_init,
	NULL,			// avoid warning
};
/*************************************************************************
 * CATCH SIGNAL OF EXITING PLAYER
 */
void mod_mplayer_chld (pid_t pid, int status)
{
	if (pid == mod_mplayer_pid) {
		log_printf(LOG_DEBUG, "mod_mplayer_chld(): player %d  exited with code %d\n", pid,WEXITSTATUS(status));
		mod_mplayer_pid = 0;
		close(mod_mplayer_fdrecv);
		mod_mplayer_fdrecv=0;
		close(mod_mplayer_fdsend);
		mod_mplayer_fdsend=0;
		close(mod_mplayer_fderror);
		mod_mplayer_fderror=0;
	}
}


/*************************************************************************
 * ANSWER A HALT REQUEST
 *
 */
void mod_mplayer_halt (void)
{
	if (mod_mplayer_fdsend)
		close(mod_mplayer_fdsend);
	mod_mplayer_fdsend = 0;
	if (mod_mplayer_fdrecv)
		close(mod_mplayer_fdrecv);
	mod_mplayer_fdrecv = 0;
	if (mod_mplayer_fderror)
		close(mod_mplayer_fderror);
	mod_mplayer_fderror = 0;
	if (mod_mplayer_pid) {
		log_printf(LOG_DEBUG, "mod_mplayer_halt(): killed player pid %d\n", mod_mplayer_pid);
		kill(mod_mplayer_pid, SIGTERM);
		mod_mplayer_pid = 0;
	}
	FD_ZERO(&mod_mplayer_fdset);
	mod_mplayer.poll = NULL;
	if(album) {free(album); album = NULL;}
	if(title) {free(title); title = NULL;}
	if(comment) {free(comment); comment = NULL;}
	if(artist) {free(artist); artist = NULL;}
	if(year) {free(year); year = NULL;}
	if(genre) {free(genre); genre = NULL;}
	mod_mplayer_status = MPLAYER_STATUS_HALT;
}



/*************************************************************************
 * START PLAYER PROCESS
 */
int mod_mplayer_launch (char * song)
{

	// check if player is already running
	if (mod_mplayer_status != MPLAYER_STATUS_HALT) {
		log_printf(LOG_DEBUG, "mod_mplayer_start(): mod_mplayer_pid is %d; mplayer seems to be running!\n", mod_mplayer_pid);
		return 0;
	}


	mod_mplayer_pid = system_noblock(&mod_mplayer_fdsend,&mod_mplayer_fdrecv,&mod_mplayer_fderror,"%s %s -vo null -slave '%s'",config_getstr("mplayer_binary",MPLAYER_PATH),config_getstr("mplayer_param", ""),song);
		
	sleep(1);
	if (!mod_mplayer_pid || mod_mplayer_pid == -1) {
		//char s[512];
		mod_mplayer_pid = 0;
		log_printf(LOG_DEBUG, "mod_mplayer_start(): player process didn't start!\n");
		mod_mplayer_halt();
		//while(readline(mod_mplayer_fdrecv,s,512)){log_printf(LOG_ERROR,"%s\n",mod_mplayer_fdrecv);}
		//exit(0);
		return -1;
	}

	FD_SET(mod_mplayer_fdrecv,&mod_mplayer_fdset);
	FD_SET(mod_mplayer_fderror,&mod_mplayer_fdset);
	mod_mplayer.poll = mod_mplayer_poll;
	return 0;
}

/******************************************
 * ANSWER A STOP REQUEST
 *
 */

void mod_mplayer_stop()
{
	switch(mod_mplayer_status) {
		case MPLAYER_STATUS_HALT:
		case MPLAYER_STATUS_STOP:
			return;
		case MPLAYER_STATUS_PLAY:
		case MPLAYER_STATUS_PAUSE:
			sendtext(mod_mplayer_fdsend, "seek 0 1\n");
			sendtext(mod_mplayer_fdsend, "pause\n");
			mod_mplayer_status = MPLAYER_STATUS_STOP;
			log_printf(LOG_NORMAL, "Playing stopped\n");
			log_printf(LOG_NOISYDEBUG,"mod_mplayer : entered stop state\n");
			return;
		default:
			log_printf(LOG_ERROR,"mod_mplayer: unknown status %d\n",mod_mplayer_status);
			return;
	}
}
		

/******************************************
 * ANSWER A PLAY REQUEST
 *
 */
	
void mod_mplayer_play(char* song) {
	char *tmp;
	switch(mod_mplayer_status) {
		case MPLAYER_STATUS_HALT:
			if (mod_mplayer_launch(song)) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				return;
			}
			tmp = "get_time_length\n";
			sendtext(mod_mplayer_fdsend, tmp);
			log_printf(LOG_NORMAL, "Playing '%s'\n", song);
			mod_mplayer_status= MPLAYER_STATUS_PLAY;
			mod_mplayer_song_pos = -1;
			mod_mplayer_song_length = -1;
			log_printf(LOG_NOISYDEBUG,"mod_mplayer : entered play state\n");
			return;

		case MPLAYER_STATUS_PAUSE:
		case MPLAYER_STATUS_STOP:
			sendtext(mod_mplayer_fdsend, "pause\n");
		case MPLAYER_STATUS_PLAY:
			tmp = malloc(strlen(song) + 18);
			sprintf(tmp,"loadfile \"%s\"\n",song);
			sendtext(mod_mplayer_fdsend, tmp);
			sprintf(tmp,"get_time_length\n");
			sendtext(mod_mplayer_fdsend, tmp);
			log_printf(LOG_NORMAL, "Playing '%s'\n", song);
			if(album) {free(album); album = NULL;}
			if(title) {free(title); title = NULL;}
			if(comment) {free(comment); comment = NULL;}
			if(artist) {free(artist); artist = NULL;}
			if(year) {free(year); year = NULL;}
			if(genre) {free(genre); genre = NULL;}
			free(tmp);
			mod_mplayer_status= MPLAYER_STATUS_PLAY;
			mod_mplayer_song_pos = -1;
			mod_mplayer_song_length = -1;
			log_printf(LOG_NOISYDEBUG,"mod_mplayer : entered play state\n");
			return;
			
		default:
			log_printf(LOG_ERROR,"mod_mplayer: unknown status %d\n",mod_mplayer_status);
			return;
	}
}

/******************************************
 * ANSWER A PAUSE REQUEST
 *
 */
	
void mod_mplayer_pause() {
	switch(mod_mplayer_status) {
		case MPLAYER_STATUS_HALT:
		case MPLAYER_STATUS_STOP:
			return;
			
		case MPLAYER_STATUS_PAUSE:
			sendtext(mod_mplayer_fdsend, "pause\n");
			log_printf(LOG_NOISYDEBUG,"mod_mplayer : entered play state\n");
			mod_mplayer_status= MPLAYER_STATUS_PLAY;
			return;
			
		case MPLAYER_STATUS_PLAY:
			sendtext(mod_mplayer_fdsend, "pause\n");
			log_printf(LOG_NOISYDEBUG,"mod_mplayer : entered pause state\n");
			mod_mplayer_status= MPLAYER_STATUS_PAUSE;
			return;
			
		default:
			log_printf(LOG_ERROR,"mod_mplayer: unknown status %d\n",mod_mplayer_status);
			return;

	}
}
/******************************************
 * ANSWER A SEEK REQUEST
 *
 */
	
void mod_mplayer_seek(char* position) {
	switch(mod_mplayer_status) {
		case MPLAYER_STATUS_HALT:
		case MPLAYER_STATUS_STOP:
			return;

		case MPLAYER_STATUS_PAUSE:
			sendtext(mod_mplayer_fdsend, "pause\n");
			log_printf(LOG_NOISYDEBUG,"mod_mplayer : entered play state\n");
			mod_mplayer_status= MPLAYER_STATUS_PLAY;
		case MPLAYER_STATUS_PLAY:
			if (*position=='-' || *position=='+')
				sendtext(mod_mplayer_fdsend, "seek %c%d 0\n", *position, atoi(position+1));
			else
				sendtext(mod_mplayer_fdsend, "seek %d 2\n", atoi(position));
			return;

		default:
			log_printf(LOG_ERROR,"mod_mplayer: unknown status %d\n",mod_mplayer_status);
			return;
	}
}

/********************************************
 * TRY TO SEE If WE CAN PLAY THE FILE
 * */


int mod_mplayer_identify(char*song) {
	char out[512];
	int out_fd;
	system_block(0,&out_fd,NULL,"%s %s -quiet -frames 0 -vo null -ao null -identify \"%s\" | grep ID_AUDIO_CODEC",config_getstr("mplayer_param", ""),config_getstr("mplayer_binary", MPLAYER_PATH),song);
	while(readline(out_fd,out,sizeof(out)) >= 0) {
		if (!strncmp(out,"ID_AUDIO_CODEC=",15)){ //file can be played
			close(out_fd);
			return 1;
		}
	}
	close(out_fd);	
	return 0;
}
			
	



/*************************************************************************
 * POLL INPUT DATA
 */
void mod_mplayer_poll (int fd)
{
	char s[512], *c1, *c2, *c3;
	int rc, secpos;

	// read the response from the player
	rc = readline(fd, s, sizeof(s));
	if (rc < 0) {
		// player closed connection
		log_printf(LOG_DEBUG,"mod_mplayer_poll(): player closed connection with %s.\n",strerror(errno));
		mod_mplayer_halt();
		mod_sendmsg(MSGTYPE_PLAYER, "error");
		return;
	}

	// process player response
	if (!s || !*s)
		return;
	c1 = strtok(s, " \t");
	if(!c1) return;

	if(!strcasecmp(c1,"A:")) { // status
		c2 = strtok(NULL," \t");
		for(c3=c2; isdigit(*c3); c3++);
		if(*c3==':') {// we have mm:ss display
			secpos = atoi(c2) * 60 + atoi(c3+1);
		} else {// we have straight seconds
			secpos = atoi(c2);
		}
		if(secpos != mod_mplayer_song_pos) {
			if (mod_mplayer_song_pos == -1) {
				mod_sendmsgf(MSGTYPE_PLAYER,"playing");
			}
			mod_mplayer_song_pos = secpos;
			if(mod_mplayer_song_length != -1) {
				mod_sendmsgf(MSGTYPE_PLAYER, "time %d %d", secpos, mod_mplayer_song_length - secpos);
			} else {
				mod_sendmsgf(MSGTYPE_PLAYER, "time %d -1", secpos);
			}
		}
	} else if ( !strncasecmp(c1, "ANS_LENGTH=",11)) {
		secpos = atoi(c1+11);
		if(secpos) { // mplayer reports 0 for unknown
			mod_mplayer_song_length = secpos;
		}
	} else if ( !strncasecmp(c1, "title",5) || !strncasecmp(c1,"name",4)) {
		c2 = strtok(NULL,"");
		if(!c2) return;
		if(title) free(title);
		title = strdup(c2);
		mod_sendmsgf(MSGTYPE_EVENT,"update title %s",title);
	} else if ( !strncasecmp(c1, "artist",6)) {
		c2 = strtok(NULL,"");
		if(!c2) return;
		if(artist) free(artist);
		artist = strdup(c2);
		mod_sendmsgf(MSGTYPE_EVENT,"update artist %s",artist);
	} else if ( !strncasecmp(c1, "album",6)) {
		c2 = strtok(NULL,"");
		if(!c2) return;
		if(album) free(album);
		album = strdup(c2);
		mod_sendmsgf(MSGTYPE_EVENT,"update album %s",album);
	} else if ( !strncasecmp(c1, "year",4)) {
		c2 = strtok(NULL,"");
		if(!c2) return;
		if(year) free(year);
		year = strdup(c2);
		mod_sendmsgf(MSGTYPE_EVENT,"update year %s",year);
	} else if ( !strncasecmp(c1, "genre",6)) {
		c2 = strtok(NULL,"");
		if(!c2) return;
		if(genre) free(genre);
		genre = strdup(c2);
		mod_sendmsgf(MSGTYPE_EVENT,"update genre %s",genre);
	} else if ( !strncasecmp(c1, "comment",7)) {
		c2 = strtok(NULL,"");
		if(!c2) return;
		if(comment) free(comment);
		comment = strdup(c2);
		mod_sendmsgf(MSGTYPE_EVENT,"update comment %s",comment);
	} else {
		// unknown message, probably an error
		c2 = strtok(NULL, "");
		log_printf(LOG_VERBOSE, "mplayer reports: '%s %s'\n", c1, c2);
		if(!strcasecmp("yes",config_getstr("abort_on_unknown_mplayer_status","no"))) {
			log_printf(LOG_NORMAL, "Playing stopped\n");
			mod_sendmsg(MSGTYPE_PLAYER, "error");
			mod_mplayer_halt();
		}
	}

}


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_mplayer_message (int msgtype, char *msg,const char __attribute__((unused))*sender)
{
	char *c1, *c2, *c3;

	if (msgtype == MSGTYPE_PLAYER) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		if(!c1) return;
		// see if we can play the requested song
		if ( !strcasecmp(c1, "play")) {
			c2 = strtok(NULL," \t");
			if(!c2 )return;
			if (!strcasecmp(c2,"mod_mplayer") ) {
				c3 = strtok(NULL,"");
				if(!c3 )return;
				mod_mplayer_play(c3);
			} else {
					log_printf(LOG_DEBUG, "mod_mplayer_message(): not for me: '%s'\n", c2);
					mod_mplayer_halt();
			}
		} else if ( !strcasecmp(c1, "stop") && mod_mplayer_pid) {
			mod_mplayer_stop();
		} else if (!strcasecmp(c1, "pause") && mod_mplayer_pid) {
			mod_mplayer_pause();

		} else if ( !strcasecmp(c1, "seek") && mod_mplayer_pid) {
			c2 = strtok(NULL," \t");
			if(!c2 )return;
			log_printf(LOG_DEBUG, "mod_mplayer_message(): seeking %s\n", c2);
			mod_mplayer_seek(c2);
		}
	} else if (msgtype == MSGTYPE_QUERY) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		if(!c1) return;
		if (!strcasecmp(c1,"songtype")) {
			c2 = strtok(NULL,"");
			if(!c2 )return;
			if(mod_mplayer_identify(c2)){
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mplayer %s",c2);
			}
		} else if (!strcasecmp(c1,"whocanplay")) {
			c2 = strtok(NULL,"");
			if(!c2 )return;
			if( !strcasecmp(c2,"mpg")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mplayer %s",c2);
			} else if( !strcasecmp(c2,"mp2")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mplayer %s",c2);
			} else if(  !strcasecmp(c2,"mp3")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mplayer %s",c2);
			} else if(  !strcasecmp(c2,"asf")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mplayer %s",c2);
			} else if(  !strcasecmp(c2,"rm")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mplayer %s",c2);
			} else if(  !strcasecmp(c2,"ogg")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mplayer %s",c2);
			} else if(  !strcasecmp(c2,"avi")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mplayer %s",c2);
			} else if(  !strcasecmp(c2,"http")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mplayer %s",c2);
			}
		} else if (mod_mplayer_status != MPLAYER_STATUS_HALT) {
			if(artist  && !strcasecmp(c1,"artist"))  mod_sendmsgf(MSGTYPE_INFO,"artist %s",artist);
			if(album   && !strcasecmp(c1,"album"))   mod_sendmsgf(MSGTYPE_INFO,"album %s",album);
			if(title   && !strcasecmp(c1,"title"))   mod_sendmsgf(MSGTYPE_INFO,"title %s",title);
			if(year    && !strcasecmp(c1,"year"))    mod_sendmsgf(MSGTYPE_INFO,"year %s",year);
			if(genre   && !strcasecmp(c1,"genre"))   mod_sendmsgf(MSGTYPE_INFO,"genre %s",genre);
			if(comment && !strcasecmp(c1,"comment")) mod_sendmsgf(MSGTYPE_INFO,"comment %s",comment);
			if(!strcasecmp(c1,"time"))    mod_sendmsgf( MSGTYPE_INFO,"time %d %d", mod_mplayer_song_pos, mod_mplayer_song_length -mod_mplayer_song_pos);
		}
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_mplayer_init (void)
{
	FD_ZERO(&mod_mplayer_fdset);


	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_mplayer_deinit (void)
{
	mod_mplayer_halt();
	log_printf(LOG_DEBUG, "mod_mplayer_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
