/***************************************************************************
 *   Copyright (C) 2004 by Andrew de Quincey                               *
 *   adq@lidskialf.net                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "ipodsysinfo.h"

#include <qdir.h>
#include <qfile.h>
#include <qregexp.h>
#include <qtextstream.h>

#include <sys/statfs.h>

#include <kdebug.h>

const QString IPodSysInfo::iPodControlDir("/iPod_Control");

IPodSysInfo::IPodSysInfo(const QString& ipod_Base) : kbytesTotal(0), kbytesAvail(0) {
    this->ipod_base = ipod_Base;
}

IPodSysInfo::~IPodSysInfo() {
}

bool IPodSysInfo::load() {
    bool result = true;

    QDir dir(ipod_base + iPodControlDir);
    QStringList musicDirNames = dir.entryList( "Music;music", QDir::Dirs );
    if ( musicDirNames.size() < 1 ) {
        return false;
    }

    bool musicDirFound = false;
    for ( uint i = 0; i < musicDirNames.size() && !musicDirFound; ++i ) {
        if ( dir.cd ( musicDirNames[ i ] ) ) {
            dir.setFilter( QDir::Dirs );
            dir.setNameFilter( "F??;f??" );
            if ( dir.count() > 0 ) {
                musicDirFound = true;
            } else {
                dir.cdUp();
            }
        }
    }
    if ( !musicDirFound ) {
        return false;
    }

    musicDir = dir;

    // refresh disk usage statistics
    refreshDiskUsageStats();

    // figure out the number of directories where Tracks reside (file wise)
    this->numTrackFileDirs = musicDir.count();

    // load the sysinfo file
    QFile sysInfoFile(ipod_base + iPodControlDir + "/Device/SysInfo");
    if (sysInfoFile.exists() && sysInfoFile.open(IO_ReadOnly)) {
        QRegExp re("^(.+):(.+)$");
        QTextIStream stringStream(sysInfoFile.readAll());
        stringStream.setEncoding(QTextStream::UnicodeUTF8); // guess

        while(!stringStream.atEnd()) {
        QString line = stringStream.readLine();
            if (!re.search(line)) {
                QString key = re.cap(1).stripWhiteSpace();
                QString value = re.cap(2).stripWhiteSpace();
                if (key.length()) {
                details[key] = value;
                }
            }
        }

        sysInfoFile.close();
        result = true;
    }

    return result;
}

QValueList<QString> IPodSysInfo::getKeys() {
    return details.keys();
}

QString IPodSysInfo::getValue(const QString& key) {
    return details[key];
}

void IPodSysInfo::refreshDiskUsageStats() {
    this->kbytesTotal = 0;
}

unsigned long IPodSysInfo::getTotalDiskSpaceKB() {
    ensureDiskUsageStats();
    return kbytesTotal;
}

unsigned long IPodSysInfo::getAvailableDiskSpaceKB() {
    ensureDiskUsageStats();
    return kbytesAvail;
}

unsigned long IPodSysInfo::getUsedDiskSpaceKB() {
    ensureDiskUsageStats();
    return kbytesTotal - kbytesAvail;
}

void IPodSysInfo::ensureDiskUsageStats() {
    if ( ! this->kbytesTotal ) {
        struct statfs statistics;
        if ( statfs( (ipod_base + iPodControlDir).ascii(), &statistics ) == 0 ) {
            this->kbytesTotal = long(statistics.f_bsize * (statistics.f_blocks >> 10));
            this->kbytesAvail = long(statistics.f_bsize * (statistics.f_bavail >> 10));
        }
    }
}

