// -*- C++ -*-
// --------------------------------------------------------------------
// Ipe object attributes (not just color!)
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPECOLOR_H
#define IPECOLOR_H

#include "ipebase.h"
#include "ipegeo.h"

// --------------------------------------------------------------------

class IpeAttribute {
  enum { ESymbolic = 0x80000000, EValue = 0x40000000, ENumeric = 0x20000000,
	 EKindShift = 24, EKindMask = 0x1f000000,
	 ENameMask = 0x00ffffff };
public:
  //! The different kinds of attributes.
  // Attributes 0 .. ETemplate are stored in style sheet,
  enum TKind { ELineWidth = 0, EMarkSize, EArrowSize,
	       EGridSize, EAngleSize, ETextStretch,
	       EDashStyle, ETextSize,
	       EColor, ETextStyle, ETemplate,
	       ELineCap, ELineJoin, EWindRule,
	       EMaxKind };

  //! Default constructor creates null object.
  explicit IpeAttribute() : iName(0) { /* nothing */ }

  explicit IpeAttribute(TKind kind, bool symbolic, int index);
  explicit IpeAttribute(TKind kind, IpeFixed value);

  //! Is it symbolic?
  inline int IsSymbolic() const { return iName & ESymbolic; }
  //! In bool expressions, attribute is false iff it is null.
  inline operator bool() const { return !IsNull(); }
  //! Is it null?
  inline bool IsNull() const { return iName == 0; }
  /*! Is it absolute (including the special values \c void, \c solid,
    \c black, or \c white)? */
  inline bool IsAbsolute() const { return iName && !(iName & ESymbolic); }
  //! Is it an (absolute) value (either numeric or in IpeRepository)?
  inline int IsValue() const { return (iName & EValue); }
  //! Is it a numeric value?
  inline int IsNumeric() const { return (iName & ENumeric); }
  //! Is it void?
  inline bool IsVoid() const { return (iName == 1); }
  //! Is it null or void (null is interpreted as void when drawing)?
  inline bool IsNullOrVoid() const { return IsNull() || IsVoid(); }
  //! Is it solid?
  inline bool IsSolid() const { return (iName == 2); }
  //! Is it null or solid (null is interpreted as solid when drawing)?
  inline bool IsNullOrSolid() const { return IsNull() || IsSolid(); }
  //! Is it black?
  inline bool IsBlack() const { return (iName == 3); }
  //! Is it white?
  inline bool IsWhite() const { return (iName == 4); }
  //! Return index into IpeRepository.
  inline int Index() const { return iName & ENameMask; }
  //! Return value of absolute numeric argument.
  inline IpeFixed Number() const {
    return IpeFixed::FromInternal(iName & ENameMask); }

  TKind Kind() const;

  //! Are two values equal (only compares index!)
  inline bool operator==(const IpeAttribute &rhs) const {
    return iName == rhs.iName; }

  //! Are two values different (only compares index!)
  inline bool operator!=(const IpeAttribute &rhs) const {
    return iName != rhs.iName; }

  //! Create void color or pen.
  inline static IpeAttribute Void() { return IpeAttribute(1); }
  //! Create solid dash style.
  inline static IpeAttribute Solid() { return IpeAttribute(2); }
  //! Create black color.
  inline static IpeAttribute Black() { return IpeAttribute(3); }
  //! Create white color.
  inline static IpeAttribute White() { return IpeAttribute(4); }
private:
  inline IpeAttribute(int index) : iName(index) { /* nothing */ }
private:
  int iName;

  friend class IpeStyleSheet;
};

/*! \var IpeKind
  \ingroup attr
  \brief Shortcut for IpeAttribute::TKind.
*/
typedef IpeAttribute::TKind IpeKind;

/*! \var IpeAttributeSeq
  \ingroup attr
  \brief A sequence of attribute values.
*/
typedef std::vector<IpeAttribute> IpeAttributeSeq;

class IpeColor {
public:
  //! Default constructor.
  IpeColor() { /* nothing */ }
  explicit IpeColor(double red, double green, double blue);
  void Save(IpeStream &stream) const;
  bool IsGray() const;
  bool operator==(const IpeColor &rhs) const;
public:
  double iRed, iGreen, iBlue;
};

//! Layout of an IpePage.
struct IpeLayout {
  //! Create null layout.
  IpeLayout() { iPaperSize.iX = -1.0; }
  //! Is this an undefined (null) layout?
  bool isNull() const { return iPaperSize.iX < 0.0; }
  //! Return rectangle describing the paper.
  IpeRect paper() const { return IpeRect(-iOrigin, iPaperSize - iOrigin); }
  //! Dimensions of the media.
  IpeVector iPaperSize;
  //! Origin of the Ipe coordinate system relative to the paper.
  IpeVector iOrigin;
  //! Size of the frame (the drawing area).
  IpeVector iFrameSize;
  //! Paragraph skip (between textboxes).
  double iParagraphSkip;
};

//! A shading pattern for the paper background.
struct IpeShading {
  //! There are two types of shadings, along an axis or between two circles.
  enum TType { ENone = 0, EAxial = 2, ERadial = 3 };

  //! The type of shading: axial or radial.
  TType iType;
  //! The extreme colors of the shading.
  IpeColor iColor[2];
  //! The coordinates of the axis endpoints, or the two circle centers.
  IpeVector iV[2];
  //! The radii of the two circles (not used for axial shadings).
  IpeScalar iRadius[2];
  //! Whether to extend the shading beyond the endpoints.
  bool iExtend[2];
};

class IpeDashStyle {
public:
  //! Default constructor creates solid style.
  IpeDashStyle() { /* nothing */ }
  //! Construct dash style with given PDF representation.
  explicit IpeDashStyle(IpeString str);
  bool operator==(const IpeDashStyle &rhs) const;
public:
  //! The PDF representation of this dash style.
  IpeString iDash;
};

class IpeStrokeStyle {
public:
  //! Line join style.
  enum { EMiterJoin = 0, ERoundJoin, EBevelJoin };
  //! Line cap style.
  enum { EButtCap = 0, ERoundCap, ESquareCap };
  //! Constructor creates all null attributes.
  IpeStrokeStyle() : iBits(0) { /* nothing */ }
  IpeAttribute Join() const;
  IpeAttribute Cap() const;
  IpeAttribute WindRule() const;
  void SetJoin(int val);
  void SetCap(int val);
  void SetWindRule(int val);
  void SetJoin(IpeAttribute val);
  void SetCap(IpeAttribute val);
  void SetWindRule(IpeAttribute val);
private:
  int iBits;
};

class IpeRepository {
public:
  IpeRepository();

  IpeString ToString(IpeAttribute index) const;
  IpeColor ToColor(IpeAttribute index) const;
  IpeFixed ToScalar(IpeAttribute index) const;

  IpeAttribute ToAttribute(const IpeColor &color);
  IpeAttribute ToAttribute(IpeKind kind, IpeFixed value);

  IpeAttribute MakeColor(IpeString str);
  IpeAttribute MakeString(IpeKind kind, IpeString str);
  IpeAttribute MakeDashStyle(IpeString str);
  IpeAttribute MakeTextSize(IpeString str);
  IpeAttribute MakeScalar(IpeKind kind, IpeString str);
  IpeAttribute MakeSymbol(IpeKind kind, IpeString str);
  IpeAttribute GetSymbol(IpeKind kind, IpeString str) const;

  IpeString String(IpeAttribute attr) const;

private:
  IpeAttribute ToSymbolic(IpeKind kind, IpeString name);

private:
  std::vector<IpeString> iStrings;
  std::vector<IpeColor> iColors;
};

// --------------------------------------------------------------------

inline IpeStream &operator<<(IpeStream &stream, const IpeColor &attr)
{
  attr.Save(stream); return stream;
}

// --------------------------------------------------------------------
#endif
