#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import thread

from twisted.internet import reactor
from twisted.internet.defer import Deferred

from imagestore.lib.tests import TestCase
from imagestore.lib.twistedutil import mergeDeferreds

from imagestore.model import Image, ImageState
from imagestore.sqlitestorage import SQLiteStorage
from imagestore.storageservice import (
    StorageService, StoreImagesTask, GetStoredImagesTask,
    GetStoredImageStatesTask, SetUpgradeImagesTask, SetImageRegistrationTask,
    SetErrorMessageTask, ClearErrorMessageTask, GetInstalledImageUrisTask)
from imagestore.tests.helpers import ServiceTestCase


class StorageServiceTest(ServiceTestCase):

    def setUp(self):
        self.filename = self.makeFile()
        self.storage = SQLiteStorage(self.filename)
        def storageFactory():
            return SQLiteStorage(self.filename)
        self.service = StorageService(reactor, storageFactory)

    def testStorageConstructedInDifferentThread(self):
        filename = self.makeFile()
        factoryThread = []
        def storageFactory():
            factoryThread.append(thread.get_ident())
            return SQLiteStorage(filename)
        service = StorageService(reactor, storageFactory)
        service.start()

        def callback(result):
            self.assertEquals(len(factoryThread), 1)
            self.assertNotEquals(factoryThread, [thread.get_ident()])

        return service.stop().addCallback(callback)

    def testStoreImages(self):
        image1 = self.createImage(1)
        image2 = self.createImage(2)

        task = StoreImagesTask([image1, image2])
        self.service.addTask(task)

        self.startStopService(self.service)

        self.assertEquals(self.storage.getImage(image1["uri"]), image1)
        self.assertEquals(self.storage.getImage(image2["uri"]), image2)

    def testGetStoredImages(self):
        image1 = self.createImage(1)
        image2 = self.createImage(2)

        self.storage.addImage(image1)
        self.storage.addImage(image2)
        self.storage.commitAndClose()

        task = GetStoredImagesTask([image1["uri"], image2["uri"]])
        deferred = self.service.addTask(task)

        def callback(result):
            self.assertEquals(len(result), 2)
            self.assertEquals(result[0], image1)
            self.assertEquals(result[1], image2)

        deferred.addCallback(callback)

        return self.runServicesAndWaitForDeferred([self.service], deferred)

    def testGetStoredImageStates(self):
        self.storage.commitAndClose()

        task = GetStoredImageStatesTask(["http://uri1", "http://uri2"])
        deferred = self.service.addTask(task)

        def callback(result):
            self.assertEquals(len(result), 2)
            self.assertEquals(result[0],
                              ImageState({
                                  "image-uri": "http://uri1",
                                  "status": "uninstalled",
                              }))
            self.assertEquals(result[1],
                              ImageState({
                                  "image-uri": "http://uri2",
                                  "status": "uninstalled",
                              }))

        deferred.addCallback(callback)

        return self.runServicesAndWaitForDeferred([self.service], deferred)

    def testSetUpgradeImages(self):
        image1 = self.createImage(1)
        image2 = self.createImage(2)
        image3 = self.createImage(3)
        uri1 = image1["uri"]
        uri2 = image2["uri"]
        uri3 = image3["uri"]

        self.storage.addImage(image1)
        self.storage.setUpgrades([uri2, uri3])
        self.storage.commitAndClose()

        task = SetUpgradeImagesTask([image1, image2])
        deferred1 = self.service.addTask(task)
        def callback(result):
            storage = SQLiteStorage(self.filename)
            try:
                uris = storage.getUpgrades()
                self.assertEquals(set(uris), set([uri1, uri2]))
            finally:
                storage.close()
        deferred1.addCallback(callback)

        return self.runServicesAndWaitForDeferred([self.service], deferred1)

    def testSetImageRegistration(self):
        imageRegistration = self.createImageRegistration(1)
        task = SetImageRegistrationTask(imageRegistration)
        deferred1 = self.service.addTask(task)
        def callback(result):
            storage = SQLiteStorage(self.filename)
            try:
                imageUri = imageRegistration.image["uri"]
                state = storage.getImageState(imageUri)
                self.assertEquals(state,
                                  ImageState({
                                      "image-uri": imageUri,
                                      "status": "installed",
                                      "eki": imageRegistration.eki.eid,
                                      "eri": imageRegistration.eri.eid,
                                      "emi": imageRegistration.emi.eid,
                                  }))
            finally:
                storage.close()
        deferred1.addCallback(callback)

        return self.runServicesAndWaitForDeferred([self.service], deferred1)

    def testSetErrorMessage(self):
        task = SetErrorMessageTask("http://uri", "Some error!")
        deferred1 = self.service.addTask(task)
        def callback(result):
            storage = SQLiteStorage(self.filename)
            try:
                self.assertEquals(storage.getErrorMessage("http://uri"),
                                  "Some error!")
            finally:
                storage.close()
        deferred1.addCallback(callback)
        return self.runServicesAndWaitForDeferred([self.service], deferred1)

    def testSetErrorMessage(self):
        self.storage.setErrorMessage("http://uri", "Some error!")
        self.storage.commitAndClose()

        task = ClearErrorMessageTask("http://uri")
        deferred1 = self.service.addTask(task)
        def callback(result):
            storage = SQLiteStorage(self.filename)
            try:
                self.assertEquals(storage.getErrorMessage("http://uri"), None)
            finally:
                storage.close()
        deferred1.addCallback(callback)
        return self.runServicesAndWaitForDeferred([self.service], deferred1)

    def testGetInstalledImageUris(self):
        imageRegistration1 = self.createImageRegistration(1)
        imageRegistration2 = self.createImageRegistration(2)
        image3 = self.createImage(3)
        image4 = self.createImage(4)
        deferred1 = self.service.addTask(
                        SetImageRegistrationTask(imageRegistration1))
        deferred2 = self.service.addTask(
                        SetImageRegistrationTask(imageRegistration2))
        deferred3 = self.service.addTask(StoreImagesTask([image3, image4]))

        def setupDone(result):
            deferred = self.service.addTask(GetInstalledImageUrisTask())

            def checkResult(uris):
                self.assertEquals(set(uris),
                                  set([imageRegistration1.image["uri"],
                                       imageRegistration2.image["uri"]]))

            deferred.addCallback(checkResult)
            deferred.addCallback(testFinished.callback)
            deferred.addErrback(testFinished.errback)

        testFinished = Deferred()

        setupDeferred = mergeDeferreds([deferred1, deferred2, deferred3])
        setupDeferred.addCallback(setupDone)
        setupDeferred.addErrback(testFinished.errback)

        return self.runServicesAndWaitForDeferred([self.service], testFinished)
