/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


//
//  A front-end for portable inheritance
//
#ifndef IGENERICFILTER_H
#define IGENERICFILTER_H


#include "iviewsubjectcomponent.h"


#include "ierror.h"

#include <vtkType.h>


class vtkDataObject;
class vtkDataSetAttributes;
class vtkFloatArray;
class vtkInformation;
class vtkInformationVector;
class vtkPolyData;


template<class Source, class OutputType>
class iGenericSource : public Source, public iViewSubjectComponent
{

public:

	virtual float GetMemorySize();
	virtual int GetNumberOfOutputs();  

protected:

	iGenericSource(iViewSubject *vo, bool usesLimits);
	virtual ~iGenericSource();

	virtual int RequestData(vtkInformation *request, vtkInformationVector **inInfo, vtkInformationVector *outInfo);

	void SaveRequest(vtkInformation *request, vtkInformationVector **inInfo, vtkInformationVector *outInfo);

	virtual void ExecuteData(vtkDataObject *output);
	void ExecuteParent();

	virtual void SyncWithData(const iDataSyncRequest &r);

	virtual void ProduceOutput() = 0; // called to recompute the output

	const bool mUsesLimits;
	vtkInformation *mSavedRequest;
	vtkInformationVector **mSavedInputVector;
	vtkInformationVector *mSavedOutputVector;
	int mReturnCode;

private:

	int mExecuteMethod;
};


template<class Filter, class InputType, class OutputType>
class iGenericFilter : public Filter, public iViewSubjectComponent
{

public:

	virtual float GetMemorySize();
	virtual int GetNumberOfOutputs();  

protected:

	iGenericFilter(iViewSubject *vo, int numInputs, bool ownsOutput, bool usesLimits);
	virtual ~iGenericFilter();

	virtual int ProcessRequest(vtkInformation *request, vtkInformationVector **inInfo, vtkInformationVector *outInfo);
	virtual int RequestData(vtkInformation *request, vtkInformationVector **inInfo, vtkInformationVector *outInfo);

	void SaveRequest(vtkInformation *request, vtkInformationVector **inInfo, vtkInformationVector *outInfo);

	virtual void ExecuteData(vtkDataObject *output);
	void ExecuteParent();

	virtual void SyncWithData(const iDataSyncRequest &r);

	virtual void InitExecution();		// called if recomputing is needed
	virtual void ProduceOutput() = 0;	// called to recompute the output
	virtual void VerifyResults();		// called on every pipeline pass after ProduceOutput

	//
	//  Helper members for transforming scalars
	//
	bool ScalarsInit(vtkDataSetAttributes *data, vtkIdType nSizeOut = 0, int nDimOut = 0);
	void ScalarsDone(vtkDataSetAttributes *data);

	const bool mOwnsOutput, mUsesLimits;
	vtkInformation *mSavedRequest;
	vtkInformationVector **mSavedInputVector;
	vtkInformationVector *mSavedOutputVector;
	int mReturnCode;

	vtkFloatArray *wScalarArrIn, *wScalarArrOut;
	float *wScalarPtrIn, *wScalarPtrOut;
	int wScalarDimIn, wScalarDimOut;

private:

	int mExecuteMethod;
};


template<class Filter>
class iGenericPolyDataToPolyDataFilter : public iGenericFilter<Filter,vtkPolyData,vtkPolyData>
{

protected:
	
	iGenericPolyDataToPolyDataFilter(iViewSubject *vo, int numInputs, bool ownsOutput, bool usesLimits) : iGenericFilter<Filter,vtkPolyData,vtkPolyData>(vo,numInputs,ownsOutput,usesLimits){}

	virtual void VerifyResults();
};


//
//  Useful macro
//
#define IGENERICFILTER_DECLARE(_name_,_parent_) \
	public: \
		vtkTypeMacro(_name_,_parent_); \
		static _name_* New(iViewSubject *vo = 0) { IERROR_ASSERT(vo); return new _name_(vo); } \
	protected: \
		_name_(iViewSubject *vo);

#endif // IGENERICFILTER_H
 
