/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggpagedata.h"


#include "ibasicdatasubjects.h"
#include "icalculator.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatastretch.h"
#include "ierror.h"
#include "iimagefactory.h"
#include "ishell.h"
#include "iviewmodule.h"

#include "iggdatatypeprovider.h"
#include "iggdialog.h"
#include "iggextensionwindow.h"
#include "iggframeboxsize.h"
#include "iggframedatatypeselector.h"
#include "iggframedatatypewidgetlist.h"
#include "iggframedatavariablelist.h"
#include "iggmainwindow.h"
#include "iggreloading.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeybutton.h"
#include "iggwidgetkeylineedit.h"
#include "iggwidgetkeyselectionbox.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetotherbutton.h"

#include "ibgwidgetareasubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetentrysubject.h"
#include "ibgwidgethelper.h"

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;
using namespace iParameter;

//
//  Templates
//
#include "iarraytemplate.h"
#include "icalculatortemplate.h"
#include "iggreloadingtemplate.h"
#include "iggwidgetkeylineedittemplate.h"
#include "iggwidgetkeyslidertemplate.h"


namespace iggPageData_Private
{
	//
	//  Helper widgets
	//
	class LimitsExpandButton : public iggWidgetSimpleButton
	{
		
	public:
		
		LimitsExpandButton(bool min, bool out, const iggPageData *page, iggFrame *parent, int index) : iggWidgetSimpleButton("",parent,true), mPage(page)
		{
			mIsMin = min;
			mOut = out;
			mIndex = index;

			if(min)
			{
				if(out)
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("range_ol.png"));
				}
				else
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("range_il.png"));
				}
			}
			else
			{
                if(out)
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("range_or.png"));
				}
				else
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("range_ir.png"));
				}
			}

			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void Execute()
		{
			iString ws;
			
			if(mPage->GetProvider(0)->IsThereDataType())
			{
				int n, s;
				float fmin, fmax;

				const iObjectKey &keyNumVars = mPage->GetProvider(0)->Translate(iDataSubject::KeyNumVars());
				const iObjectKey &keyStretch = mPage->GetProvider(0)->Translate(iDataSubject::KeyStretch());
				const iObjectKey &keyName = mPage->GetProvider(0)->Translate(iDataSubject::KeyName());
				const iObjectKey &keyMin = mPage->GetProvider(0)->Translate(iDataSubject::KeyMin());
				const iObjectKey &keyMax = mPage->GetProvider(0)->Translate(iDataSubject::KeyMax());

				if(this->GetShell()->GetControlModule()->QueryValue(keyNumVars,n) && n>mIndex && this->GetShell()->GetControlModule()->QueryValue(keyStretch,mIndex,s,n,-1,-1) && this->GetShell()->GetControlModule()->QueryValue(keyMin,mIndex,fmin,n) && this->GetShell()->GetControlModule()->QueryValue(keyMax,mIndex,fmax,n))
				{
					iString ws;
					float d;

					fmin = iDataStretch::ApplyStretch(fmin,s,false);
					fmax = iDataStretch::ApplyStretch(fmax,s,true);

					if(mOut)
					{
						if(s==_StretchLog && fmax-fmin>2.0f) d = 1.0f; else d = 0.5f*(fmax-fmin);
					}
					else
					{

						if(s==_StretchLog && fmax-fmin>4.0f) d = -1.0f; else d = -0.25f*(fmax-fmin);

						const iObjectKey &keyLL = mPage->GetProvider(0)->Translate(iDataSubject::KeyLowerLimit());
						const iObjectKey &keyUL = mPage->GetProvider(0)->Translate(iDataSubject::KeyUpperLimit());

						float fmid, fll, ful;
						if(this->GetShell()->GetControlModule()->QueryValue(keyLL,mIndex,fll,n) && this->GetShell()->GetControlModule()->QueryValue(keyMax,mIndex,ful,n))
						{
							fmid = 0.5f*(fll+ful);
						}
						else
						{
							fmid = 0.5f*(fmin+fmax);
						}
					}

					if(mIsMin)
					{
						fmin = (fmin>-iMath::_LargeFloat || d<0.0f) ? iDataStretch::ResetStretch(fmin-d,s) : iDataStretch::ResetStretch(fmin,s);
						this->GetShell()->GetControlModule()->PackCommand(ws,keyMin,mIndex,fmin);
					}
					else
					{
						fmax = (fmax< iMath::_LargeFloat || d<0.0f) ? iDataStretch::ResetStretch(fmax+d,s) : iDataStretch::ResetStretch(fmax,s);
						this->GetShell()->GetControlModule()->PackCommand(ws,keyMax,mIndex,fmax);
					}

					this->GetShell()->GetControlModule()->Execute(ws,!mOut && this->GetMainWindow()->IsAutoRender(),this->GetExecuteFlags());
					mParent->UpdateWidget();
				}
			}
		}

		bool mIsMin, mOut;
		int mIndex;
		const iggPageData *mPage;
	};


	class LimitsStretchComboBox : public iggWidgetKeyStretchComboBox
	{
		
	public:
		
		LimitsStretchComboBox(const iggPageData *page, iggFrame *parent, int index) : iggWidgetKeyStretchComboBox(iDataSubject::KeyStretch(),parent,index,0), mPage(page)
		{
			this->AttachDataTypeProvider(mPage->GetProvider(0));
			this->SetExecuteFlags(_ObjectOptionOne|_ModuleOptionOne|_RenderOptionClones);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			//
			//  Optimization - direct access
			//
			iDataLimits *lim = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLimits(mPage->GetProvider(0)->GetActiveDataType());
			if(mIndex < lim->GetNumVars()) this->iggWidgetKeyComboBox::UpdateWidgetBody();
		}

		virtual void OnInt1Body(int i)
		{
			if(mPage->GetProvider(0)->IsThereDataType())
			{
				iggWidgetKeyComboBox::OnInt1Body(i);
				mParent->UpdateWidget();
			}
		}

		const iggPageData *mPage;
	};


	class LimitsNameEdit : public iggWidgetKeyTextLineEdit
	{
		
	public:
		
		LimitsNameEdit(const iggPageData *page, iggFrame *parent, int index) : iggWidgetKeyTextLineEdit(false,"Name",iDataSubject::KeyName(),_RenderModeUseGlobal,parent,index,0), mPage(page)
		{
			this->AttachDataTypeProvider(mPage->GetProvider(0));
			this->SetExecuteFlags(_ObjectOptionOne|_ModuleOptionOne|_RenderOptionClones);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			iDataLimits *lim = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLimits(mPage->GetProvider(0)->GetActiveDataType());
			if(lim != 0) this->Show(!lim->AreNamesFixed()); else this->Show(false);
			//
			//  Optimization - direct access
			//
			mSubject->SetText(lim->GetName(mIndex));
			this->iggWidgetKeyTextLineEdit::UpdateWidgetBody();
		}

		virtual void OnVoid1Body()
		{
			if(mPage->GetProvider(0)->IsThereDataType())
			{
				iggWidgetKeyTextLineEdit::OnVoid1Body();
				mParent->UpdateWidget();
			}
		}

		const iggPageData *mPage;
	};


	class LimitsRangeSlider : public iggWidgetKeyVariableLimitsSlider
	{
		
	public:
		
		LimitsRangeSlider(bool low, const iggPageData *page, iggFrame *parent, int index) : iggWidgetKeyVariableLimitsSlider(page->GetProvider(0),7,low?"Min":"Max",low?iDataSubject::KeyLowerLimit():iDataSubject::KeyUpperLimit(),0,0,_RenderModeUseGlobal,parent,index), mPage(page)
		{
			this->SetExecuteFlags(_ObjectOptionOne|_ModuleOptionOne|_RenderOptionClones);
			mSubject->FreezeNumberOfDigits(true);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			//
			//  Optimization - direct access
			//
			iDataLimits *lim = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLimits(mPage->GetProvider(0)->GetActiveDataType());
			if(mIndex < lim->GetNumVars())
			{
				mMin = lim->GetMin(mIndex);
				mMax = lim->GetMax(mIndex);
				mStretch = lim->GetStretch(mIndex);
				if(lim->GetClass(mIndex) > 999) this->SetStep(1.0f);
				mResolution = this->ComputeResolution();
				this->SetRange(0,mResolution);
				this->iggWidgetKeyFloatSlider::UpdateWidgetBody();
			}
		}

		const iggPageData *mPage;
	};


	class LimitsFrameBox : public iggFrame
	{
		
	public:
		
		LimitsFrameBox(const iggPageData *page, iggFrame *parent, int index) : iggFrame("",parent,4), mPage(page)
		{
			mIndex = index;
			mNeedsBaloonHelp = false;

			LimitsRangeSlider *vls1 = new LimitsRangeSlider(true,mPage,this,index);
			LimitsRangeSlider *vls2 = new LimitsRangeSlider(false,mPage,this,index);
			vls1->AddBuddy(vls2);
			this->AddLine(vls1,2,new LimitsExpandButton(true,true,mPage,this,index),1,new LimitsExpandButton(true,false,mPage,this,index),1);
			this->AddLine(vls2,2,new LimitsExpandButton(false,false,mPage,this,index),1,new LimitsExpandButton(false,true,mPage,this,index),1);
			mStretchBox = new LimitsStretchComboBox(mPage,this,index);
			this->AddLine(mStretchBox,1,new LimitsNameEdit(mPage,this,index),3);
			this->SetColStretch(1,10);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
//			int n;
//			iString s;
			
			if(mPage->GetProvider(0)->IsThereDataType())
			{
				const iObjectKey &keyNumVars = mPage->GetProvider(0)->Translate(iDataSubject::KeyNumVars());
				const iObjectKey &keyName = mPage->GetProvider(0)->Translate(iDataSubject::KeyName());

				//
				//  Optimization - direct access
				//
				iDataLimits *lim = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLimits(mPage->GetProvider(0)->GetActiveDataType());
				if(mIndex < lim->GetNumVars())
//				if(this->GetShell()->GetControlModule()->QueryValue(keyNumVars,n) && mIndex<n && this->GetShell()->GetControlModule()->QueryValue(keyName,mIndex,s,n))
				{
//					this->SetTitle(s);
					this->SetTitle(lim->GetName(mIndex));
					mStretchBox->Show(!lim->GetFixedStretch(mIndex));
					this->Show(true);
					this->iggFrame::UpdateWidgetBody();
				}
				else
				{
					this->Show(false);
				}
			}
			else
			{
				this->Show(false);
			}
		}

		int mIndex;
		const iggPageData *mPage;
		LimitsStretchComboBox *mStretchBox;
	};


	class LimitsFrameSet : public iggFrameScroll
	{
		
	public:
		
		LimitsFrameSet(const iggPageData *page, iggFrame *parent) : iggFrameScroll(parent,false), mPage(page)
		{
			mNeedsBaloonHelp = false;
		}

		virtual void UpdateChildren()
		{
			int i, n;

			//
			//  This is expensive to update, make sure it does not update needlessly
			//
			if(!mNeedsUpdate) return;

			const iObjectKey &keyNumVars = mPage->GetProvider(0)->Translate(iDataSubject::KeyNumVars());
			if(this->GetShell()->GetControlModule()->QueryValue(keyNumVars,n) && n>mBoxes.Size())
			{
				for(i=mBoxes.Size(); i<n; i++)
				{
					mBoxes.Add(new LimitsFrameBox(mPage,this->GetContents(),i));
					this->GetContents()->AddLine(mBoxes[i]);
					this->GetContents()->AddSpace(5);
				}
			}
			for(i=n-1; i>=0; i--) mBoxes[i]->Show(false);
			iggFrameScroll::UpdateChildren();
			for(i=0; i<n; i++) mBoxes[i]->Show(true);
		}

	protected:

		const iggPageData *mPage;
		iArray<LimitsFrameBox*> mBoxes;
	};


	class FixedLimitsFrame : public iggFrameDataTypeWidgetList
	{

	public:

		FixedLimitsFrame(iggFrame *parent) : iggFrameDataTypeWidgetList("Fix ranges for data types:",parent)
		{
			this->UpdateList();
		}

	protected:

		virtual iggWidget* CreateEntry(const iDataType &type) const
		{
			return new iggWidgetKeyCheckBox(type.GetTextName(),this->Translate(iDataSubject::KeyFixedLimits(),type),this->EntryParent());
		}
	};


	//
	//  Shift slider
	//
	class ShiftingSlider : public iggWidgetKeyDoubleSlider
	{

	public:

		ShiftingSlider(const iString &title, const iObjectKey &key, iggFrame *parent, int index) : iggWidgetKeyDoubleSlider(-1.0,1.0,200,0,5,title,key,_RenderModeUseGlobal,parent,index)
		{
			this->SetExecuteFlags(_ObjectOptionOne|_ModuleOptionOne|_RenderOptionClones);
		}

	protected:

		virtual bool ExecuteControl(bool final)
		{
			this->QueryValue(mValue);
			this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->SetDataShift(mIndex,mValue);
			
			if(final || this->GetRenderMode()==_RenderModeImmediate)
			{
				this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->ShiftData();
				this->GetShell()->GetControlModule()->Execute("",this->GetMainWindow()->IsAutoRender(),this->GetExecuteFlags());
			}
			return true;
		}
	};


	//
	//  Erase widgets
	//
	class EraseCheckBox : public iggWidget
	{

	public:

		EraseCheckBox(const iDataType &type, const iString &title, iggFrame *parent) : iggWidget(parent), mType(type)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,_ButtonTypeCheckBox,title,1);

			iString tt = "Check this box if you would like to erase these data.";
			this->SetBaloonHelp(tt);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mSubject->SetDown(false);
			mWidgetHelper->Enable(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(mType));
		}

		virtual void OnVoid1Body()
		{
			if(mSubject->IsDown())
			{
				this->GetMainWindow()->AddEraseDataType(mType);
			}
			else
			{
				this->GetMainWindow()->RemoveEraseDataType(mType);
			}
		}

		ibgWidgetButtonSubject *mSubject;
		const iDataType &mType;
	};


	class ErasePushButton : public iggWidgetSimpleButton
	{

	public:

		ErasePushButton(const iString &title, iggFrame *parent) : iggWidgetSimpleButton(title,parent)
		{
			iString tt = "Click this button to erase selected data.";
			this->SetBaloonHelp(tt);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->Enable(this->GetMainWindow()->GetEraseDataInfo().Count()>0 && this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(this->GetMainWindow()->GetEraseDataInfo()));
		}

		virtual void Execute()
		{
			if(this->GetMainWindow()->AskForConfirmation("Are you sure you want to erase data?","Erase"))
			{
				this->GetMainWindow()->EraseData();
				this->Enable(false);
			}
		}
	};


	class EraseFrame : public iggFrameDataTypeWidgetList
	{

	public:

		EraseFrame(ErasePushButton *button, iggFrame *parent) : iggFrameDataTypeWidgetList("Select the data to erase:",parent)
		{
			mButton = button;
			this->UpdateList();
		}

	protected:

		virtual iggWidget* CreateEntry(const iDataType &type) const
		{
			iggWidget *w = new EraseCheckBox(type,type.GetTextName(),this->EntryParent());
			w->AddBuddy(mButton);
			return w;
		}

		ErasePushButton *mButton;
	};


	//
	//  Function calculator
	//
	class CalculatorLineEdit : public iggWidget
	{

	public:

		CalculatorLineEdit(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetEntrySubject(this,false,0,"%bFunction:",0);
			mSubject->SetEditable(false);

			mNeedsBaloonHelp = false;
		}

		void SelectText(int start, int len = -1)
		{
			mSubject->SelectText(start,len);
		}

		const iString GetText() const
		{
			return mSubject->GetText();
		}

		void SetText(const iString &text)
		{
			mSubject->SetText(text);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
		}

		ibgWidgetEntrySubject *mSubject;
	};


	class CalculatorButton : public iggWidgetSimpleButton
	{

	public:

		CalculatorButton(const iString &title, const iString &text, CalculatorLineEdit *lineedit, iggFrame *parent) : iggWidgetSimpleButton(title,parent)
		{
			mText = text;
			mLineEdit = lineedit;
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void Execute()
		{
			iString line = mLineEdit->GetText();

			if(mText.IsEmpty())
			{
				if(line.IsEmpty()) return;

				if(line.Length()>2 && line.Part(line.Length()-3)=="Vec")
				{
					//
					//  Remove 'Vec'
					//
					line = line.Part(0,line.Length()-3);
				}
				else if(line.Length()>3 && line.Part(line.Length()-4)=="Var[")
				{
					//
					//  Remove 'Var['
					//
					line = line.Part(0,line.Length()-4);
				}
				else if(line[line.Length()-1] == '(')
				{
					//
					//  Remove 'fun('
					//
					int i = line.Length() - 2;
					while(i>=0 && line[i]>='a' && line[i]<='z') i--;
					line = line.Part(0,i+1);
				}
				else
				{
					//
					//  Remove one symbol
					//
					line = line.Part(0,line.Length()-1);
				}
			}
			else
			{
				line += mText;
			}
			mLineEdit->SetText(line);
		}

		iString mText;
		CalculatorLineEdit *mLineEdit;
	};


	class CalculatorDialog : public iggDialog
	{

	public:

		CalculatorDialog(const iDataInfo &info, iggWidgetKeyTextLineEdit *le, iggMainWindow *mw) : iggDialog(mw,_DialogModal,0,"Array Calculator","sr.gg.da",2), mInfo(info)
		{
			mLineEdit = le;

			mCalculatorLineEdit = new CalculatorLineEdit(mFrame);
			mFrame->AddLine(mCalculatorLineEdit,2);

			iggFrame *f1 = new iggFrame(mFrame,6);
			f1->AddLine(
                new CalculatorButton("+","+",mCalculatorLineEdit,f1),
				new CalculatorButton("-","-",mCalculatorLineEdit,f1),
				new CalculatorButton("x","*",mCalculatorLineEdit,f1),
				new CalculatorButton("/","/",mCalculatorLineEdit,f1),
				new CalculatorButton("^","^",mCalculatorLineEdit,f1),0);
			f1->SetColStretch(5,10);
			mFrame->AddLine(f1,2);

			iggFrame *f2a = new iggFrame("Variables",mFrame,3);
			f2a->AddLine(
				new CalculatorButton("Scalar var[#]","Var[",mCalculatorLineEdit,f2a),
				new CalculatorButton("]","]",mCalculatorLineEdit,f2a),
				new CalculatorButton("Vector field","Vec",mCalculatorLineEdit,f2a));
			iggFrame *f2b = new iggFrame("Vector operations",mFrame,3);
			f2b->AddLine(0,
				new CalculatorButton(".",".",mCalculatorLineEdit,f2b),
				new CalculatorButton("X","><",mCalculatorLineEdit,f2b));
			f2b->SetColStretch(0,10);
			mFrame->AddLine(f2a,f2b);

			iggFrame *f3 = new iggFrame("",mFrame,3);
			f3->AddLine(
                new CalculatorButton("abs","abs(",mCalculatorLineEdit,f3),
				new CalculatorButton("min","min(",mCalculatorLineEdit,f3),
				new CalculatorButton("max","max(",mCalculatorLineEdit,f3));
			f3->AddLine(
				new CalculatorButton("log","log(",mCalculatorLineEdit,f3),
				new CalculatorButton("exp","exp(",mCalculatorLineEdit,f3),
				new CalculatorButton("sqrt","sqrt(",mCalculatorLineEdit,f3));
			f3->AddLine(
				new CalculatorButton("cos","cos(",mCalculatorLineEdit,f3),
				new CalculatorButton("sin","sin(",mCalculatorLineEdit,f3),
				new CalculatorButton("tan","tan(",mCalculatorLineEdit,f3));
			f3->AddLine(
				new CalculatorButton("arccos","acos(",mCalculatorLineEdit,f3),
				new CalculatorButton("arcsin","asin(",mCalculatorLineEdit,f3),
				new CalculatorButton("arctan","atan(",mCalculatorLineEdit,f3));
			f3->AddLine(
				new CalculatorButton("cosh","cosh(",mCalculatorLineEdit,f3),
				new CalculatorButton("sinh","sinh(",mCalculatorLineEdit,f3),
				new CalculatorButton("tanh","tanh(",mCalculatorLineEdit,f3));

			iggFrame *f4 = new iggFrame("",mFrame,3);
			f4->AddLine(
				new CalculatorButton("7","7",mCalculatorLineEdit,f4),
				new CalculatorButton("8","8",mCalculatorLineEdit,f4),
				new CalculatorButton("9","9",mCalculatorLineEdit,f4));
			f4->AddLine(
				new CalculatorButton("4","4",mCalculatorLineEdit,f4),
				new CalculatorButton("5","5",mCalculatorLineEdit,f4),
				new CalculatorButton("6","6",mCalculatorLineEdit,f4));
			f4->AddLine(
				new CalculatorButton("1","1",mCalculatorLineEdit,f4),
				new CalculatorButton("2","2",mCalculatorLineEdit,f4),
				new CalculatorButton("3","3",mCalculatorLineEdit,f4));
			f4->AddLine(
				new CalculatorButton("0","0",mCalculatorLineEdit,f4),
				new CalculatorButton(".",".",mCalculatorLineEdit,f4),
				new CalculatorButton("E","e",mCalculatorLineEdit,f4));
			f4->AddLine(
				new CalculatorButton("(",")",mCalculatorLineEdit,f4),
				new CalculatorButton(")",")",mCalculatorLineEdit,f4),
				new CalculatorButton("<-","",mCalculatorLineEdit,f4));
			mFrame->AddLine(f3,f4);
		}

		virtual void Show(bool s)
		{
			if(s)
			{
				iString line;
				mLineEdit->QueryValue(line);
				mCalculatorLineEdit->SetText(line);
			}
			
			iggDialog::Show(s);
		}

		virtual const iString& GetToolTip() const
		{
			static const iString tt = "Calculator allows you to do mathematical operations on scalar data.";
			return tt;
		}

	protected:

		virtual bool CanBeClosed()
		{
			iString line = mCalculatorLineEdit->GetText();
			if(line.IsEmpty()) return true;

			iCalculator<float> calc;
			iCalculator<float>::real_var var("Var",3,&calc);
			iCalculator<float>::real_var vec("Vec",3,&calc);
			if(calc.Evaluate(line))
			{
				if(calc.GetResultSize() == 1)
				{
					iString oldLine;
					mLineEdit->QueryValue(oldLine);
					if(oldLine != line)
					{
						mLineEdit->SetText(line);
						mLineEdit->ExecuteControl(true);
						this->GetMainWindow()->RequestReloadData(mInfo);
						this->GetMainWindow()->UpdateAll();
					}
					return true;
				}
				else
				{
					mFrame->GetMainWindow()->PopupWindow("Result is not a scalar",_PopupWindowError);
					return false;
				}
			}
			else
			{
				mCalculatorLineEdit->SelectText(calc.GetErrorPosition());
				mFrame->GetMainWindow()->PopupWindow(calc.GetErrorMessage(),_PopupWindowError);
				return false;
			}
		}

		iString mLine;
		const iDataInfo &mInfo;
		iggWidgetKeyTextLineEdit *mLineEdit;
		CalculatorLineEdit *mCalculatorLineEdit;
	};


	class ReloadingPushButton : public iggWidgetSimpleButton
	{

	public:

		ReloadingPushButton(const iString &title, iggFrame *parent) : iggWidgetSimpleButton(title,parent)
		{
			iString tt = "Click this button to reload the current data set.";
			this->SetBaloonHelp(tt);
		}

	protected:


		virtual void UpdateWidgetBody()
		{
			this->Enable(this->GetMainWindow()->GetReloadDataInfo().Count()>0 && this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(this->GetMainWindow()->GetReloadDataInfo()));
		}

		virtual void Execute()
		{
			this->GetMainWindow()->ReloadData();
			this->Enable(false);
		}
	};
};


using namespace iggPageData_Private;


iggPageData::iggPageData(iggFrameBase *parent) : iggPageMain(parent,true,true)
{
	const iImage *icon = iImageFactory::FindIcon("data.png");

	mInfoBuffer = new iDataInfo[__NumDataWith]; IERROR_ASSERT(mInfoBuffer);
	mProvider = new iggCompositeDataTypeProvider(this); IERROR_ASSERT(mProvider);
	mProvider->AddProvider(new iggKeywordDataTypeProvider("scalars",this));
	mProvider->AddProvider(new iggKeywordDataTypeProvider("particles",this));

	iDataType::FindTypesByKeywords(mInfoBuffer[_DataWithParticleDownsampling],"particles");
	mInfoBuffer[_DataWithBoundaryCondition] = mInfoBuffer[_DataWithParticleDownsampling] + iUniformScalarsDataSubject::DataType() + iUniformVectorsDataSubject::DataType() + iUniformTensorsDataSubject::DataType();
	mInfoBuffer[_DataWithVariableCalculation] = iUniformScalarsDataSubject::DataType();

	this->GetMainWindow()->GetExtensionWindow()->AddReloadingDataTypes(this);

	//
	//  Common page
	// ************************************************
	//
	iggFrame *page0 = new iggFrame(mBook,1);
	mBook->AddPage("Common",icon,page0);
	{
		//
		//  Book
		//
		iggFrameBook *sb = new iggFrameBook(page0);
		page0->AddLine(sb);
		{
			//
			//  Global page
			//
			iggFrame *sbpage0 = new iggFrame(sb,2);
			sb->AddPage("Global",icon,sbpage0);
			{
				sbpage0->AddLine(new iggFrameBoxSize(sbpage0));
				sbpage0->AddSpace(2);

				iggWidgetKeyRadioBox *orb = new iggWidgetKeyRadioBox(1,"Optimize for...",0,iControlModule::KeyOptimizationMode(),sbpage0);
				orb->InsertItem("Speed");
				orb->InsertItem("Memory");
				orb->InsertItem("Quality");
				sbpage0->AddLine(orb);
				sbpage0->AddSpace(2);

				iggWidgetKeyRadioBox *bc = new iggReloading<iggWidgetKeyRadioBox>(mInfoBuffer[_DataWithBoundaryCondition],0,1,"Boundary conditions",0,iDataReader::KeyBoundaryConditions(),sbpage0);
				bc->InsertItem("Periodic");
				bc->InsertItem("Wall");
				sbpage0->AddLine(bc);
				sbpage0->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",sbpage0));
				sbpage0->AddSpace(2);

				ReloadingPushButton *rb = new ReloadingPushButton("Reload data",sbpage0);
				rb->AddDependent(bc);
				sbpage0->AddLine(rb);
				sbpage0->AddSpace(10);
				sbpage0->SetColStretch(1,10);
			}
			//
			//  Shift page
			//
			iggFrame *sbpage1 = new iggFrame(sb,2);
			sb->AddPage("Shift data",icon,sbpage1);
			{
				iggFrame *sh = new iggFrame(sbpage1);
				sh->AddLine(new ShiftingSlider("%bX",iDataReader::KeyShiftData(),sh,0));
				sh->AddLine(new ShiftingSlider("%bY",iDataReader::KeyShiftData(),sh,1));
				sh->AddLine(new ShiftingSlider("%bZ",iDataReader::KeyShiftData(),sh,2));
				sbpage1->AddLine(sh);
				sbpage1->AddSpace(10);
				sbpage1->SetColStretch(0,10);
				sbpage1->SetColStretch(1,3);
			}		
			//
			//  Erase page
			//
			iggFrame *sbpage2 = new iggFrame(sb,2);
			sb->AddPage("Erase",icon,sbpage2);
			{
				ErasePushButton *eb = new ErasePushButton("Erase selected data",sbpage2);
				sbpage2->AddLine(new EraseFrame(eb,sbpage2));
				sbpage2->AddLine(eb);
				sbpage2->AddSpace(2);
				sbpage2->SetRowStretch(0,10);
				sbpage2->SetColStretch(0,10);
				sbpage2->SetColStretch(1,10);
			}
			//
			//  Fixed/adjustable page
			//
			iggFrame *sbpage3 = new iggFrame(sb,2);
			sb->AddPage("Ranges",icon,sbpage3);
			{
				sbpage3->AddLine(new FixedLimitsFrame(sbpage3));
				sbpage3->AddSpace(10);
				sbpage3->SetRowStretch(0,8);
				sbpage3->SetColStretch(1,10);
			}
		}
	}

	//
	//  Field data page
	// ************************************************
	//
	iggFrame *page1 = new iggFrame(mBook,1);
	mBook->AddPage("Field data",icon,page1);
	{
		//
		//  Book
		//
		iggFrameBook *fb = new iggFrameBook(page1);
		page1->AddLine(fb);
		{
			//
			//  Limits page
			//
			iggFrame *fbpage1 = new iggFrame(fb,1);
			fb->AddPage("Limits",icon,fbpage1);
			{
				fbpage1->AddLine(new LimitsFrameSet(this,fbpage1));
			}
			//
			//  Placement page
			//
			iggFrame *fbpage2 = new iggFrame(fb,2);
			fb->AddPage("Placement",icon,fbpage2);
			{
				iggWidgetKeyRadioBox *vl = new iggWidgetKeyRadioBox(1,"Voxel location",0,iDataReader::KeyVoxelLocation(),fbpage2);
				vl->InsertItem("Vertex of a cell (\"point data\")");
				vl->InsertItem("Center of a cell (\"cell data\")");
				fbpage2->AddLine(vl);

				iggWidgetTextArea *tmp = new iggWidgetTextArea("  Dimension to fit",fbpage2);
				fbpage2->AddLine(tmp);
				iggWidgetKeyRadioBox *fb = new iggWidgetKeyRadioBox(1,"into bounding box",2,iDataReader::KeyScaledDimension(),fbpage2);
				fb->InsertItem("Longest");
				fb->InsertItem("Shortest");
				fb->InsertItem("X");
				fb->InsertItem("Y");
				fb->InsertItem("Z");
				fbpage2->AddLine(fb);
				fbpage2->AddSpace(10);

				fbpage2->SetColStretch(1,10);
			}		
			//
			//  Operate page
			//
			iggFrame *fbpage3 = new iggReloading<iggFrame>(mInfoBuffer[_DataWithVariableCalculation],mProvider->GetProvider(0),fb,2);
			fb->AddPage("Operate",icon,fbpage3);
			{
				iggFrameDataVariableList *ol = new iggFrameDataVariableList(mInfoBuffer[_DataWithVariableCalculation],mProvider->GetProvider(0),"Assign result to...",iUniformScalarsDataSubject::KeyVariableCalculatorOutput(),0,fbpage3);
				ol->Complete();
				iggFrame *of = new iggFrame(fbpage3,1);
				of->AddSpace(10);
				iggWidgetKeyTextLineEdit *oe = new iggWidgetKeyTextLineEdit(true,"= ",iUniformScalarsDataSubject::KeyVariableCalculatorFunction(),_RenderModeNoRender,of);
				of->AddLine(oe);
				of->AddSpace(10);
				fbpage3->AddLine(ol,of);
				fbpage3->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",fbpage3));
				fbpage3->AddSpace(2);

				CalculatorDialog *cd = new CalculatorDialog(mInfoBuffer[_DataWithVariableCalculation],oe,this->GetMainWindow());
				fbpage3->AddLine(new iggWidgetLaunchButton(cd,"Launch calculator",fbpage3,true));
				fbpage3->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",fbpage3));

				fbpage3->AddSpace(10);
				fbpage3->SetColStretch(1,10);
				fbpage3->SetColStretch(2,3);
			}
		}
	}

	//
	//  Particle data page
	// ************************************************
	//
	iggFrame *page2 = new iggFrame(mBook,1);
	mBook->AddPage("Particle data",icon,page2);
	{
		//
		//  Book
		//
		iggFrameBook *pb = new iggFrameBook(page2);
		page2->AddLine(pb);
		{
			//
			//  Downsample page
			//
			iggFrame *pbpage0 = new iggFrame(pb,2);
			pb->AddPage("Downsample",icon,pbpage0);
			{
				iggWidgetKeyIntSlider *dl = new iggReloading<iggWidgetKeyIntSlider>(mInfoBuffer[_DataWithParticleDownsampling],mProvider->GetProvider(1),1,100,3,"Factor",iParticleDataSubject::KeyDownsampleFactor(),_RenderModeNoRender,pbpage0);
				dl->AttachDataTypeProvider(mProvider->GetProvider(1));

				pbpage0->AddLine(dl,2);
				pbpage0->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",pbpage0));

				iggWidgetKeyRadioBox *db = new iggReloading<iggWidgetKeyRadioBox>(mInfoBuffer[_DataWithParticleDownsampling],mProvider->GetProvider(1),1,"Downsample mask",0,iParticleDataSubject::KeyDownsampleMode(),pbpage0);
				db->InsertItem("Regular");
				db->InsertItem("Regular 2D");
				db->InsertItem("Regular 3D");
				db->InsertItem("Random");
				db->InsertItem("Head of file");
				db->InsertItem("Tail of file");
				db->AttachDataTypeProvider(mProvider->GetProvider(1));
				pbpage0->AddLine(db);
				pbpage0->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",pbpage0));
				pbpage0->SetColStretch(1,10);

				pbpage0->AddSpace(10);
				pbpage0->SetColStretch(1,10);
			}
			//
			//  Operate page
			//
			iggFrame *pbpage1 = new iggFrame(pb,2);
			pb->AddPage("Operate",icon,pbpage1);
			{
				iggWidgetKeyCheckBox *oa = new iggReloading<iggWidgetKeyCheckBox>(mInfoBuffer[_DataWithParticleDownsampling],mProvider->GetProvider(1),"Include the particle number as an attribute",iParticleDataSubject::KeyOrderIsAttribute(),pbpage1);
				oa->AttachDataTypeProvider(mProvider->GetProvider(1));
				pbpage1->AddLine(oa);
				pbpage1->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",pbpage1));

				pbpage1->AddSpace(2);

				iggFrameDataVariableList *al = new iggFrameDataVariableList(mInfoBuffer[_DataWithParticleDownsampling],mProvider->GetProvider(1),"Compute density of...",iParticleDataSubject::KeyDensityAttribute(),0,pbpage1);
				al->InsertItem("Do not compute");
				al->InsertItem("Particle number");
				al->Complete();

				pbpage1->AddLine(al);
				pbpage1->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",pbpage1));

				pbpage1->AddSpace(10);
				pbpage1->SetColStretch(1,10);
			}
		}
	}
}


iggPageData::~iggPageData()
{
	delete [] mInfoBuffer;
	delete mProvider;
}


void iggPageData::UpdateChidren()
{
	mBook->EnablePage(3,mProvider->GetActiveDataTypeIndex()==0U);
	iggPageMain::UpdateChildren();
}


iggFrame* iggPageData::CreateFlipFrame(iggFrameBase *parent) const
{
	return new iggFrameDataTypeSelector(mProvider,false,"Current data",parent);
}


void iggPageData::UpdateOnDataLoad()
{
	this->GetProvider()->FindAvailableData();
}


void iggPageData::AddReloadingType(int key, const iDataType &type)
{
	if(key>=0 && key<__NumDataWith)
	{
		mInfoBuffer[key] += type;
	}
}


iggKeywordDataTypeProvider* iggPageData::GetProvider(int n) const
{
	if(n>=0 && n<2) return mProvider->GetProvider(n); else
	{
		if(mProvider->GetCurrentProvider() == 0)
		{
			IERROR_FATAL("iggPageData is configured incorrectly.");
			return 0;
		}
		else return mProvider->GetCurrentProvider();
	}
}


void iggPageData::OnInt1Body(int i)
{
	i--;
	if(i != mProvider->GetCurrentIndex())
	{
		mProvider->SetCurrentIndex(i);
		if(mFlipFrame != 0) mFlipFrame->UpdateWidget();
	}
}

#endif
