/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggframepaletteselectionbase.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "ipalette.h"
#include "ishell.h"

#include "iggdialogpaletteeditor.h"
#include "iggmainwindow.h"
#include "iggwidgetarea.h"
#include "iggwidgetmisc.h"
#include "iggwidgetotherbutton.h"

#include "ibgwidgetselectionboxsubject.h"

#include "iggsubjectfactory.h"

//
//  Templates
//
#include "iarraytemplate.h"


namespace iggFramePaletteSelectionBase_Private
{
	class PaletteView : public iggWidgetImageArea
	{

	public:

		PaletteView(iggFramePaletteSelectionBase *parent) : iggWidgetImageArea(parent)
		{
			mFrame = parent;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			static const iImage dummy;
			if(mFrame->GetCurrentPalette() != 0)
			{
				this->SetImage(*mFrame->GetCurrentPalette()->GetImage(),true);
			}
			else
			{
				this->SetImage(dummy,true);
			}
			this->iggWidgetImageArea::UpdateWidgetBody();
		}

		iggFramePaletteSelectionBase *mFrame;
	};

	
	class IsReversedCheckBox : public iggWidgetSimpleCheckBox
	{

	public:

		IsReversedCheckBox(iggFramePaletteSelectionBase *parent) : iggWidgetSimpleCheckBox("Reversed",parent)
		{
			mFrame = parent;
			this->SetBaloonHelp("Reverse the current palette","Use this check box to flip the order of colors in the current palette.");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			if(mFrame->GetCurrentPaletteIndex() != 0)
			{
				this->Enable(true);
				this->SetChecked(mFrame->GetCurrentPaletteIndex() < 0);
			}
			else this->Enable(false);
		}

		virtual void OnChecked(bool s)
		{
			mFrame->SetCurrentPaletteIndex(iMath::Abs(mFrame->GetCurrentPaletteIndex())*(s?-1:1));
		}

		iggFramePaletteSelectionBase *mFrame;
	};


	class PaletteList : public iggWidget
	{

	public:

		PaletteList(IsReversedCheckBox *rb, bool withbrightness, iggFramePaletteSelectionBase *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"Palette");
			mFrame = parent;
			mReverseBox = rb;
			mOffset = (withbrightness ? 0 : 1);

			this->SetBaloonHelp("Choose palette from the list","Choose a palette from the list of all available palettes. Use Palette Editor from the dialog menus to create new or change existing palettes.");
		}

		virtual ~PaletteList()
		{
		}

		int GetValue() const
		{
			return mSubject->GetValue();
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			int i, np = this->GetShell()->GetControlModule()->GetNumberOfPalettes();

			mSubject->Clear();
			if(mOffset == 0) mSubject->InsertItem("Brightness");
			for(i=0; i<np; i++)
			{
				mSubject->InsertItem(this->GetShell()->GetControlModule()->GetPalette(i)->GetName());
			}

			mSubject->SetValue(iMath::Abs(mFrame->GetCurrentPaletteIndex())-mOffset);
		}

		virtual void OnInt1Body(int n)
		{
			mFrame->SetCurrentPaletteIndex((mOffset+n)*(mReverseBox->IsChecked()?-1:1));
		}

		int mOffset;
		IsReversedCheckBox *mReverseBox;
		iggFramePaletteSelectionBase *mFrame;
		ibgWidgetComboBoxSubject *mSubject;
	};


	class LaunchButton : public iggWidgetLaunchButton
	{

	public:

		LaunchButton(PaletteList *list, iggFrame *parent) : iggWidgetLaunchButton(parent->GetMainWindow()->GetDialogPaletteEditor(),"Launch palette editor",parent)
		{
			mList = list;
		}

	protected:

		virtual void OnVoid1Body()
		{
			if(mDialog!=0 && mList!=0)
			{
				iDynamicCast<iggDialogPaletteEditor,iggDialog>(INFO,mDialog)->SetCurrentPaletteIndex(mList->GetValue());
			}
			iggWidgetLaunchButton::OnVoid1Body();
		}

		PaletteList *mList;
	};
};


using namespace iggFramePaletteSelectionBase_Private;


iggFramePaletteSelectionBase::iggFramePaletteSelectionBase(bool witheditor, bool withbrightness, iggFrame *parent) : iggFrame("Current palette",parent,3)
{
	PaletteView *view = new PaletteView(this);
	IsReversedCheckBox *rb = new IsReversedCheckBox(this);
	PaletteList *list = new PaletteList(rb,withbrightness,this);
	list->AddDependent(rb);
	this->AddLine(view,3);
	this->AddLine(list,2,rb,1);

	if(witheditor)
	{
		this->AddLine(new LaunchButton(list,this));
	}

	this->SetColStretch(1,10);

	list->AddDependent(view);
	List().AddUnique(this);
}


iggFramePaletteSelectionBase::~iggFramePaletteSelectionBase()
{
	List().Remove(this);
}


void iggFramePaletteSelectionBase::SetCurrentPaletteIndex(int n)
{
	if(n>=-this->GetShell()->GetControlModule()->GetNumberOfPalettes() && n<=this->GetShell()->GetControlModule()->GetNumberOfPalettes())
	{
		this->SetCurrentPaletteIndexBody(n);
	}
	else IERROR_LOW("Incorrect palette number.");
}


void iggFramePaletteSelectionBase::UpdateAll()
{
	iPointerArray<iggFramePaletteSelectionBase> &list = List();
	int i, n = list.Size();

	for(i=0; i<n; i++) list[i]->UpdateWidget();
}


iPointerArray<iggFramePaletteSelectionBase>& iggFramePaletteSelectionBase::List()
{
	static iPointerArray<iggFramePaletteSelectionBase> list;
	return list;
}


iPalette* iggFramePaletteSelectionBase::GetCurrentPalette() const
{
	return this->GetShell()->GetControlModule()->GetPalette(iMath::Abs(this->GetCurrentPaletteIndex())-1);
}

#endif
