/*
 *  Copyright (C) 2010 Tuomo Penttinen, all rights reserved.
 *
 *  Author: Tuomo Penttinen <tp@herqq.org>
 *
 *  This file is part of Herqq UPnP (HUPnP) library.
 *
 *  Herqq UPnP is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Herqq UPnP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Herqq UPnP. If not, see <http://www.gnu.org/licenses/>.
 */

#include "hclientdevice_adapter.h"
#include "hclientdevice_adapter_p.h"
#include "hclientservice_adapter.h"
#include "hclientdevice.h"

#include "../hdevicemodel_validator.h"
#include "../hdevicemodel_infoprovider.h"

#include "../../dataelements/hdeviceinfo.h"

namespace Herqq
{

namespace Upnp
{

/*******************************************************************************
 * HAdapterOp
 ******************************************************************************/
HAdapterOp::HAdapterOp(const HAsyncOp& op) :
    m_op(op)
{
    Q_ASSERT(!op.isNull());
}

HAdapterOp::~HAdapterOp()
{
}

/*******************************************************************************
 * HClientDeviceAdapterPrivate
 ******************************************************************************/
HClientDeviceAdapterPrivate::HClientDeviceAdapterPrivate() :
    m_device(0), q_ptr(0), m_ongoingOperations()
{
}

HClientDeviceAdapterPrivate::~HClientDeviceAdapterPrivate()
{
}

void HClientDeviceAdapterPrivate::removeOp(const HAsyncOp& op)
{
    HAdapterOp* adapterOp = m_ongoingOperations.value(op);
    if (adapterOp)
    {
        delete adapterOp;
        m_ongoingOperations.remove(op);
    }
}

bool HClientDeviceAdapterPrivate::validate(HClientDevice* device) const
{
    HDeviceModelValidator validator;
    validator.setInfoProvider(*m_infoProvider.data());
    return validator.validateDevice(device) == HDeviceModelValidator::ValidationSucceeded;
}

/*******************************************************************************
 * HClientDeviceAdapter
 ******************************************************************************/
HClientDeviceAdapter::HClientDeviceAdapter(
    const HResourceType& deviceType, HClientDeviceAdapterPrivate& dd,
    QObject* parent) :
        QObject(parent),
            h_ptr(&dd)
{
    h_ptr->q_ptr = this;
    Q_ASSERT(deviceType.isValid());
}

HClientDeviceAdapter::HClientDeviceAdapter(
    const HResourceType& deviceType, QObject* parent) :
        QObject(parent),
            h_ptr(new HClientDeviceAdapterPrivate())
{
    h_ptr->q_ptr = this;
    Q_ASSERT(deviceType.isValid());
}

HClientDeviceAdapter::~HClientDeviceAdapter()
{
    delete h_ptr;
}

bool HClientDeviceAdapter::prepareDevice(HClientDevice*)
{
    return true;
}

void HClientDeviceAdapter::setDeviceModelInfoProvider(
    const HDeviceModelInfoProvider& infoProvider)
{
    h_ptr->m_infoProvider.reset(infoProvider.clone());
}

const HDeviceModelInfoProvider* HClientDeviceAdapter::deviceModelInfoProvider() const
{
    return h_ptr->m_infoProvider.data();
}

bool HClientDeviceAdapter::setDevice(HClientDevice* device, ValidationType vt)
{
    if (!device ||
        !deviceType().compare(
            device->info().deviceType(), HResourceType::Inclusive))
    {
        return false;
    }

    if (vt == MinimalValidation)
    {
        return true;
    }

    if (deviceModelInfoProvider())
    {
        if (!h_ptr->validate(device))
        {
            return false;
        }
    }

    if (!prepareDevice(device))
    {
        return false;
    }

    h_ptr->m_device = device;

    return true;
}

HClientDevice* HClientDeviceAdapter::device() const
{
    return h_ptr->m_device;
}

bool HClientDeviceAdapter::isReady() const
{
    return h_ptr->m_device;
}

const HResourceType& HClientDeviceAdapter::deviceType() const
{
    return h_ptr->m_deviceType;
}

}
}
