#!/usr/bin/env python
# Copyright (C) 2005-2006 Sivan Greenberg. <sivan@ubuntu.com>
# 
# This software is distributed under the terms and conditions of the GNU General
# Public License. See http://www.gnu.org/copyleft/gpl.html for details.

import sys
import os
import time
import subprocess
import re
from pexpect import *
from DeviceInfo import *

DEBUG_MODE = True


class IsoBuildError(Exception):
    pass

class NotMultiSessoinCD(Exception):
    pass

class NonExistantDevice(Exception):
    pass

class CannotUnMountError(Exception):
    pass



class ISOBuilder:
    def __init__(self, sourcepath, targetpath, multisession=False, msdevice=None, vlabel=""):
        self.multi_session = multisession
        self.sourcepath = sourcepath
        self.targetpath = targetpath
        self.curPrecent = self.prevPrecent = 0

        devlist = HUBDeviceDict(False,True)
        
        try:
            self.properties = devlist[msdevice]
        except KeyError:
            raise NonExistantDevice
        
        if self.properties["Type"] != "cdrom":
            raise NotBurnDeviceError

        # check if this device is already mounted and thus inaccessible and
        # automatically unmount it.
        try:
            aVolumeIsMounted = self.properties['Volumes'][msdevice]['VolumeIsMounted']
        except KeyError:
            # no volumes on this target so it cannot be mounted, do nothing
            aVolumeIsMounted = False

        if aVolumeIsMounted:
            if DEBUG_MODE: print "* Attempting to unmount %s" % msdevice
            retcode = subprocess.call(["pumount",msdevice])
            if retcode!=0:
                raise CannotUnMountError
        
        
        # enable rock ridge and other useful stuff, also set up the volume label.
        self.TOOL_CMDLINE = " -gui -r -J -T"
        if vlabel:
            self.TOOL_CMDLINE = self.TOOL_CMDLINE + " -V " + vlabel

        # multi session mode specifics
        if self.multi_session:
            # mkisofs command line required to retrive multisession information.
            self.CDRECORD_CMDLINE = "cdrecord -dev=%s -msinfo" % msdevice
            if DEBUG_MODE: print "self.CDRECORD_CMDLINE = %s" % self.CDRECORD_CMDLINE
            
            #get multi session info
            proc = subprocess.Popen(self.CDRECORD_CMDLINE.split(),stderr=subprocess.PIPE,stdout=subprocess.PIPE)
            
            self.msinfo = proc.stdout.readline()
            
            if DEBUG_MODE: print "* MultiSession Info = %s" % self.msinfo
            if self.msinfo=="":
                raise NotMultiSessoinCD
            
            self.TOOL_CMDLINE = self.TOOL_CMDLINE + " -C %s -M %s" 
            self.TOOL_CMDLINE = self.TOOL_CMDLINE % (self.msinfo, msdevice)
            
        self.TOOL_CMDLINE = self.TOOL_CMDLINE + " -o " + targetpath
        self.TOOL_CMDLINE = self.TOOL_CMDLINE + " " + sourcepath
        if DEBUG_MODE: print "self.TOOL_CMDLINE: %s" % self.TOOL_CMDLINE
        self.curOutputLine = ''
        self.oper_log = []

    def pre_build_kick(self): # returns the spawned object reference for further manipulation by GUI code.
        if DEBUG_MODE: print "Executing mkisofs to build image file."
        self.proc = spawn("mkisofs",self.TOOL_CMDLINE.split())
        return self.proc
        

    def build(self):
        outputLine = ''
        proc_isatty = True
        while proc_isatty:
            try:
                outputLine = self.proc.readline()
            except TIMEOUT:
                yield True
            if outputLine!='':
                self.curOutputLine = outputLine
                self.oper_log.append(self.curOutputLine)
            proc_isatty = self.proc.isalive()
            yield True
        yield False


    def reportLine(self):
        return self.curOutputLine

    def reportPrecent(self):
        regexp = re.compile(r'\ *(\d*)\.\d*\%\ *done',re.IGNORECASE)
        curLine = self.curOutputLine        
        match = regexp.match(curLine)
        if match:
            try:
                self.curPrecent = int(match.group(1))
            except ValueError:
                self.curPrecent = self.prevPrecent
        else:
            self.curPrecent = self.prevPrecent
        self.prevPrecent = self.curPrecent
        return self.curPrecent


if __name__ == "__main__":
    vol_lable = "UPBSCD_" + time.strftime("%d-%b-%Y_%H:%M")
    try:
        src = sys.argv[1]
        target = sys.argv[2]
    except IndexError:
        print "Please provide source path and target path where to create the iso file."
        sys.exit(1)
    Builder = ISOBuilder(src,target,False,'/dev/hdc',vol_lable)
    print Builder
    procObj = Builder.pre_build_kick()
    g = Builder.build()
    while g.next():
        print Builder.reportLine()
    if DEBUG_MODE:
        print "log:"
        print Builder.oper_log

    
