/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2007, 2008, 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */


/**
 * SECTION:hrn-texture-frame
 * @short_description: Stretch a texture to fit the entire allocation
 *
 * #HrnTextureFrame
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cogl/cogl.h>

#include "hrn-texture-frame.h"
#include "hrn.h"

enum
{
  PROP_0,

  PROP_PARENT_TEXTURE,

  PROP_LEFT,
  PROP_TOP,
  PROP_RIGHT,
  PROP_BOTTOM,

  PROP_SCALE_INVARIANT
};

G_DEFINE_TYPE (HrnTextureFrame, hrn_texture_frame, CLUTTER_TYPE_ACTOR);

#define HRN_TEXTURE_FRAME_GET_PRIVATE(obj) \
   (G_TYPE_INSTANCE_GET_PRIVATE (( obj), HRN_TYPE_TEXTURE_FRAME, \
                                 HrnTextureFramePrivate))

struct _HrnTextureFramePrivate
{
  ClutterTexture *parent_texture;

  gfloat     left;
  gfloat     top;
  gfloat     right;
  gfloat     bottom;
  gboolean   scale_invariant;
  gboolean   draw_middle;
  gboolean   can_pinch;
  CoglHandle material;
};

static void
hrn_texture_frame_get_preferred_width (
  ClutterActor *self, gfloat for_height, gfloat       *min_width_p,
  gfloat       *natural_width_p)
{
  HrnTextureFramePrivate *priv = HRN_TEXTURE_FRAME (self)->priv;

  if (G_UNLIKELY (priv->parent_texture == NULL))
    {
      if (min_width_p)
        *min_width_p = 0;

      if (natural_width_p)
        *natural_width_p = 0;
    }
  else
    {
      ClutterActorClass *klass;

      /* by directly querying the parent texture's class implementation
       * we are going around any override mechanism the parent texture
       * might have in place, and we ask directly for the original
       * preferred width
       */
      klass = CLUTTER_ACTOR_GET_CLASS (priv->parent_texture);
      klass->get_preferred_width (CLUTTER_ACTOR (priv->parent_texture),
                                  for_height,
                                  min_width_p,
                                  natural_width_p);
    }
}

static void
hrn_texture_frame_get_preferred_height (
  ClutterActor *self, gfloat for_width, gfloat       *min_height_p,
  gfloat       *natural_height_p)
{
  HrnTextureFramePrivate *priv = HRN_TEXTURE_FRAME (self)->priv;

  if (G_UNLIKELY (priv->parent_texture == NULL))
    {
      if (min_height_p)
        *min_height_p = 0;

      if (natural_height_p)
        *natural_height_p = 0;
    }
  else
    {
      ClutterActorClass *klass;

      /* by directly querying the parent texture's class implementation
       * we are going around any override mechanism the parent texture
       * might have in place, and we ask directly for the original
       * preferred height
       */
      klass = CLUTTER_ACTOR_GET_CLASS (priv->parent_texture);
      klass->get_preferred_height (CLUTTER_ACTOR (priv->parent_texture),
                                   for_width,
                                   min_height_p,
                                   natural_height_p);
    }
}

static void
hrn_texture_frame_realize (ClutterActor *self)
{
  HrnTextureFramePrivate *priv = HRN_TEXTURE_FRAME (self)->priv;

  if (priv->material != COGL_INVALID_HANDLE)
    return;

  priv->material = cogl_material_new ();

  CLUTTER_ACTOR_SET_FLAGS (self, CLUTTER_ACTOR_REALIZED);
}

static void
hrn_texture_frame_unrealize (ClutterActor *self)
{
  HrnTextureFramePrivate *priv = HRN_TEXTURE_FRAME (self)->priv;

  if (priv->material == COGL_INVALID_HANDLE)
    return;

  cogl_material_unref (priv->material);
  priv->material = COGL_INVALID_HANDLE;

  CLUTTER_ACTOR_UNSET_FLAGS (self, CLUTTER_ACTOR_REALIZED);
}

/* utility function to retrieve the absolute scale (including
 * scale factors of all ancestors), does not take depth into
 * account
 */
static gdouble
actor_get_abs_scale (ClutterActor *actor)
{
  ClutterActor *iter  = actor;
  gdouble       scale = 1.0;

  while (iter)
    {
      gdouble tscale;
      clutter_actor_get_scale (iter, &tscale, NULL);
      scale *= tscale;
      iter   = clutter_actor_get_parent (iter);
    }
  return scale;
}

static void
hrn_texture_frame_paint (ClutterActor *self)
{
  HrnTextureFramePrivate *priv         = HRN_TEXTURE_FRAME (self)->priv;
  CoglHandle              cogl_texture = COGL_INVALID_HANDLE;
  ClutterActorBox         box          = { 0, };
  gfloat                  width, height;
  gfloat                  tex_width, tex_height;
  gfloat                  ex, ey;
  gfloat                  tx1, ty1, tx2, ty2;
  guint8                  opacity;
  gfloat                  factor = 1.0;

  gdouble                 pinch = hrn_get_pinch ();

  if (!priv->can_pinch)
    pinch = 0.0;

  /* no need to paint stuff if we don't have a texture */
  if (G_UNLIKELY (priv->parent_texture == NULL))
    return;

  /* parent texture may have been hidden, so need to make sure it gets
   * realized
   */
  if (!CLUTTER_ACTOR_IS_REALIZED (priv->parent_texture))
    clutter_actor_realize (CLUTTER_ACTOR (priv->parent_texture));

  cogl_texture = clutter_texture_get_cogl_texture (priv->parent_texture);
  if (cogl_texture == COGL_INVALID_HANDLE)
    return;

  if (priv->scale_invariant)
    {
      factor = 1.0 / actor_get_abs_scale (self);
    }

  tex_width  = cogl_texture_get_width (cogl_texture);
  tex_height = cogl_texture_get_height (cogl_texture);

  clutter_actor_get_allocation_box (self, &box);
  width  = box.x2 - box.x1;
  height = box.y2 - box.y1;

  tx1 = (priv->left) / tex_width;
  tx2 = (tex_width - priv->right) / tex_width;
  ty1 = priv->top / tex_height;
  ty2 = (tex_height - priv->bottom) / tex_height;

  ex = width - priv->right * factor;
  if (ex < 0)
    ex = priv->right * factor;          /* FIXME ? */

  ey = height - priv->bottom * factor;
  if (ey < 0)
    ey = priv->bottom * factor;         /* FIXME ? */

  opacity = clutter_actor_get_paint_opacity (self);

  g_assert (priv->material != COGL_INVALID_HANDLE);

  /* set the source material using the parent texture's COGL handle */
  cogl_material_set_color4ub (priv->material,
                              opacity, opacity, opacity, opacity);
  cogl_material_set_layer (priv->material, 0, cogl_texture);
  cogl_set_source (priv->material);

  if (pinch <= 0.1)
    {
      GLfloat rectangles[] =
      {
        /* top left corner */
        0,                   0,                                    priv->left *
        factor,
        priv->top * factor,
        0.0,                 0.0,
        tx1,                 ty1,

        /* top middle */
        priv->left * factor, 0,                                    ex,
        priv->top * factor,
        tx1,                 0.0,
        tx2,                 ty1,

        /* top right */
        ex,                  0,                                    width,
        priv->top * factor,
        tx2,                 0.0,
        1.0,                 ty1,

        /* mid left */
        0,                   priv->top * factor,                   priv->left *
        factor,                ey,
        0.0,                 ty1,
        tx1,                 ty2,



        /* mid right */
        ex,                  priv->top * factor,                   width,
        ey,
        tx2,                 ty1,
        1.0,                 ty2,

        /* bottom left */
        0,                   ey,                                   priv->left *
        factor,                height,
        0.0,                 ty2,
        tx1,                 1.0,

        /* bottom center */
        priv->left * factor, ey,                                   ex,
        height,
        tx1,                 ty2,
        tx2,                 1.0,

        /* bottom right */
        ex,                  ey,                                   width,
        height,
        tx2,                 ty2,
        1.0,                 1.0,

        /* center */
        priv->left * factor, priv->top * factor,
        ex,                  ey,
        tx1,                 ty1,                                  tx2,
        ty2,
      };

      if (priv->draw_middle)
        cogl_rectangles_with_texture_coords (rectangles, 9);
      else
        cogl_rectangles_with_texture_coords (rectangles, 8);
    }
  else
    {
      /* we cheat and draw the full polygon, for the use case
         in hornsey that seems sufficient */
      tx1 = ty1 = 0; tx2 = ty2 = 1;
      CoglTextureVertex polygon[4] = {
        { pinch * 0.5,         pinch * 2.5,         0.0,            tx1,    ty1, },
        { width - pinch * 0.5, pinch * 2.5,         0.0,            tx2,    ty1, },
        { width,         height,            0.0,            tx2,    ty2, },
        { 0.0,           height,            0.0,            tx1,    ty2, },
      };
      cogl_polygon (polygon, 4, FALSE);
    }
}

static inline void
hrn_texture_frame_set_frame_internal (HrnTextureFrame *frame, gfloat left,
                                      gfloat top, gfloat right,
                                      gfloat bottom)
{
  HrnTextureFramePrivate *priv    = frame->priv;
  GObject                *gobject = G_OBJECT (frame);
  gboolean                changed = FALSE;

  g_object_freeze_notify (gobject);

  if (priv->top != top)
    {
      priv->top = top;
      g_object_notify (gobject, "top");
      changed = TRUE;
    }

  if (priv->right != right)
    {
      priv->right = right;
      g_object_notify (gobject, "right");
      changed = TRUE;
    }

  if (priv->bottom != bottom)
    {
      priv->bottom = bottom;
      g_object_notify (gobject, "bottom");
      changed = TRUE;
    }

  if (priv->left != left)
    {
      priv->left = left;
      g_object_notify (gobject, "left");
      changed = TRUE;
    }

  if (changed && CLUTTER_ACTOR_IS_VISIBLE (frame))
    clutter_actor_queue_redraw (CLUTTER_ACTOR (frame));

  g_object_thaw_notify (gobject);
}

static void
hrn_texture_frame_set_property (GObject *gobject, guint prop_id,
                                const GValue *value,
                                GParamSpec   *pspec)
{
  HrnTextureFrame        *frame = HRN_TEXTURE_FRAME (gobject);
  HrnTextureFramePrivate *priv  = frame->priv;

  switch (prop_id)
    {
      case PROP_PARENT_TEXTURE:
        hrn_texture_frame_set_parent_texture (frame,
                                              g_value_get_object (value));
        break;

      case PROP_TOP:
        hrn_texture_frame_set_frame_internal (frame,
                                              priv->left,
                                              g_value_get_float (value),
                                              priv->right,
                                              priv->bottom);
        break;

      case PROP_RIGHT:
        hrn_texture_frame_set_frame_internal (frame,
                                              priv->top,
                                              g_value_get_float (value),
                                              priv->bottom,
                                              priv->left);
        break;

      case PROP_BOTTOM:
        hrn_texture_frame_set_frame_internal (frame,
                                              priv->top,
                                              priv->right,
                                              g_value_get_float (value),
                                              priv->left);
        break;

      case PROP_LEFT:
        hrn_texture_frame_set_frame_internal (frame,
                                              priv->top,
                                              priv->right,
                                              priv->bottom,
                                              g_value_get_float (value));
        break;

      case PROP_SCALE_INVARIANT:
        hrn_texture_frame_set_scale_invariant (frame,
                                               g_value_get_boolean (value));
        break;

      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
        break;
    }
}

static void
hrn_texture_frame_get_property (GObject *gobject, guint prop_id,
                                GValue     *value,
                                GParamSpec *pspec)
{
  HrnTextureFramePrivate *priv = HRN_TEXTURE_FRAME (gobject)->priv;

  switch (prop_id)
    {
      case PROP_PARENT_TEXTURE:
        g_value_set_object (value, priv->parent_texture);
        break;

      case PROP_LEFT:
        g_value_set_float (value, priv->left);
        break;

      case PROP_TOP:
        g_value_set_float (value, priv->top);
        break;

      case PROP_RIGHT:
        g_value_set_float (value, priv->right);
        break;

      case PROP_BOTTOM:
        g_value_set_float (value, priv->bottom);
        break;

      case PROP_SCALE_INVARIANT:
        g_value_set_boolean (value, priv->scale_invariant);
        break;

      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
        break;
    }
}

static void
hrn_texture_frame_dispose (GObject *gobject)
{
  HrnTextureFramePrivate *priv = HRN_TEXTURE_FRAME (gobject)->priv;

  if (priv->parent_texture)
    {
      g_object_unref (priv->parent_texture);
      priv->parent_texture = NULL;
    }

  if (priv->material)
    {
      cogl_material_unref (priv->material);
      priv->material = COGL_INVALID_HANDLE;
    }

  G_OBJECT_CLASS (hrn_texture_frame_parent_class)->dispose (gobject);
}

static void
hrn_texture_frame_class_init (HrnTextureFrameClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);
  GParamSpec        *pspec;

  g_type_class_add_private (gobject_class, sizeof (HrnTextureFramePrivate));

  actor_class->get_preferred_width =
    hrn_texture_frame_get_preferred_width;
  actor_class->get_preferred_height =
    hrn_texture_frame_get_preferred_height;
  actor_class->realize   = hrn_texture_frame_realize;
  actor_class->unrealize = hrn_texture_frame_unrealize;
  actor_class->paint     = hrn_texture_frame_paint;

  gobject_class->set_property = hrn_texture_frame_set_property;
  gobject_class->get_property = hrn_texture_frame_get_property;
  gobject_class->dispose      = hrn_texture_frame_dispose;

  pspec = g_param_spec_object ("parent-texture",
                               "Parent Texture",
                               "The parent ClutterTexture",
                               CLUTTER_TYPE_TEXTURE,
                               G_PARAM_READWRITE |
                               G_PARAM_CONSTRUCT);
  g_object_class_install_property (gobject_class, PROP_PARENT_TEXTURE, pspec);

  pspec = g_param_spec_float ("left",
                              "Left",
                              "Left offset",
                              0, G_MAXFLOAT,
                              0,
                              G_PARAM_READWRITE);
  g_object_class_install_property (gobject_class, PROP_LEFT, pspec);

  pspec = g_param_spec_float ("top",
                              "Top",
                              "Top offset",
                              0, G_MAXFLOAT,
                              0,
                              G_PARAM_READWRITE);
  g_object_class_install_property (gobject_class, PROP_TOP, pspec);

  pspec = g_param_spec_float ("bottom",
                              "Bottom",
                              "Bottom offset",
                              0, G_MAXFLOAT,
                              0,
                              G_PARAM_READWRITE);
  g_object_class_install_property (gobject_class, PROP_BOTTOM, pspec);

  pspec = g_param_spec_float ("right",
                              "Right",
                              "Right offset",
                              0, G_MAXFLOAT,
                              0,
                              G_PARAM_READWRITE);
  g_object_class_install_property (gobject_class, PROP_RIGHT, pspec);

  pspec = g_param_spec_boolean ("scale-invariant",
                                "Scale Invariant",
                                "",
                                FALSE,
                                G_PARAM_READWRITE);
  g_object_class_install_property (gobject_class, PROP_SCALE_INVARIANT, pspec);
}

static void
hrn_texture_frame_init (HrnTextureFrame *self)
{
  HrnTextureFramePrivate *priv;

  self->priv = priv = HRN_TEXTURE_FRAME_GET_PRIVATE (self);

  priv->material = COGL_INVALID_HANDLE;
}

/**
 * hrn_texture_frame_new:
 * @texture: a #ClutterTexture or %NULL
 * @left: left margin preserving its content
 * @top: top margin preserving its content
 * @right: right margin preserving its content
 * @bottom: bottom margin preserving its content
 *
 * A #HrnTextureFrame is a specialized texture that efficiently clones
 * an area of the given @texture while keeping preserving portions of the
 * same texture.
 *
 * A #HrnTextureFrame can be used to make a rectangular texture fit a
 * given size without stretching its borders.
 *
 * Return value: the newly created #HrnTextureFrame
 */
ClutterActor*
hrn_texture_frame_new (ClutterTexture *texture, gfloat left, gfloat top,
                       gfloat right,
                       gfloat bottom)
{
  g_return_val_if_fail (texture == NULL || CLUTTER_IS_TEXTURE (texture), NULL);

  return g_object_new (HRN_TYPE_TEXTURE_FRAME,
                       "parent-texture", texture,
                       "left", left,
                       "top", top,
                       "right", right,
                       "bottom", bottom,
                       NULL);
}

ClutterTexture *
hrn_texture_frame_get_parent_texture (HrnTextureFrame *frame)
{
  g_return_val_if_fail (HRN_IS_TEXTURE_FRAME (frame), NULL);

  return frame->priv->parent_texture;
}

void
hrn_texture_frame_set_parent_texture (HrnTextureFrame *frame,
                                      ClutterTexture  *texture)
{
  HrnTextureFramePrivate *priv;
  gboolean                was_visible;

  g_return_if_fail (HRN_IS_TEXTURE_FRAME (frame));
  g_return_if_fail (texture == NULL || CLUTTER_IS_TEXTURE (texture));

  priv = frame->priv;

  was_visible = CLUTTER_ACTOR_IS_VISIBLE (frame);

  if (priv->parent_texture == texture)
    return;

  if (priv->parent_texture)
    {
      g_object_unref (priv->parent_texture);
      priv->parent_texture = NULL;

      if (was_visible)
        clutter_actor_hide (CLUTTER_ACTOR (frame));
    }

  if (texture)
    {
      priv->parent_texture = g_object_ref (texture);

      /*  if (was_visible && CLUTTER_ACTOR_IS_VISIBLE (priv->parent_texture))*/
      clutter_actor_show (CLUTTER_ACTOR (frame));
    }

  clutter_actor_queue_relayout (CLUTTER_ACTOR (frame));

  g_object_notify (G_OBJECT (frame), "parent-texture");
}

void
hrn_texture_frame_set_frame (HrnTextureFrame *frame, gfloat top, gfloat right,
                             gfloat bottom,
                             gfloat left)
{
  g_return_if_fail (HRN_IS_TEXTURE_FRAME (frame));

  hrn_texture_frame_set_frame_internal (frame, top, right, bottom, left);
}

void
hrn_texture_frame_get_frame (HrnTextureFrame *frame, gfloat           *top,
                             gfloat           *right, gfloat           *bottom,
                             gfloat           *left)
{
  HrnTextureFramePrivate *priv;

  g_return_if_fail (HRN_IS_TEXTURE_FRAME (frame));

  priv = frame->priv;

  if (top)
    *top = priv->top;

  if (right)
    *right = priv->right;

  if (bottom)
    *bottom = priv->bottom;

  if (left)
    *left = priv->left;
}

gboolean
hrn_texture_frame_get_scale_invariant (HrnTextureFrame *frame)
{
  HrnTextureFramePrivate *priv;

  g_return_val_if_fail (HRN_IS_TEXTURE_FRAME (frame), FALSE);

  priv = frame->priv;
  return priv->scale_invariant;
}

void
hrn_texture_frame_set_draw_middle (HrnTextureFrame *frame, gboolean value)
{
  HrnTextureFramePrivate *priv;

  g_return_if_fail (HRN_IS_TEXTURE_FRAME (frame));

  priv = frame->priv;
  if (priv->draw_middle != value)
    {
      priv->draw_middle = value;
      if (CLUTTER_ACTOR_IS_VISIBLE (frame))
        clutter_actor_queue_redraw (CLUTTER_ACTOR (frame));
    }
}

void
hrn_texture_frame_set_scale_invariant (HrnTextureFrame *frame, gboolean value)
{
  HrnTextureFramePrivate *priv;

  g_return_if_fail (HRN_IS_TEXTURE_FRAME (frame));

  priv = frame->priv;
  if (priv->scale_invariant != value)
    {
      priv->scale_invariant = value;
      if (CLUTTER_ACTOR_IS_VISIBLE (frame))
        clutter_actor_queue_redraw (CLUTTER_ACTOR (frame));
    }
}

void
hrn_texture_frame_set_can_pinch (HrnTextureFrame *frame, gboolean value)
{
  HrnTextureFramePrivate *priv;

  g_return_if_fail (HRN_IS_TEXTURE_FRAME (frame));

  priv = frame->priv;
  if (priv->can_pinch != value)
    {
      priv->can_pinch = value;
      if (CLUTTER_ACTOR_IS_VISIBLE (frame))
        clutter_actor_queue_redraw (CLUTTER_ACTOR (frame));
    }
}
