#include "hrn.h"

#include "hrn-cluster-tree.h"
#include "hrn-source.h"

enum
{
  PROP_0,
};

enum
{
  ITEM_CHANGED,
  ITEM_ADDED,
  ITEM_REMOVED,
  STARTED,
  FINISHED,
  LAST_SIGNAL
};

struct _HrnSourcePrivate
{
  BklSourceClient *client;
  BklDB           *db;
  GSequence       *index;

  HrnClusterTree *tree;
  GHashTable     *uri_to_item;

  gboolean is_indexing;

  guint32 changed_id;
  guint32 added_id;
  guint32 removed_id;
  guint32 index_changed_id;
  guint32 started_id;
  guint32 finished_id;
};

#define GET_PRIVATE(obj)    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                                          HRN_TYPE_SOURCE, \
                                                          HrnSourcePrivate))
G_DEFINE_TYPE (HrnSource, hrn_source, G_TYPE_OBJECT);
static guint32 signals[LAST_SIGNAL] = { 0, };

#if 0
static void
free_items (GPtrArray *items)
{
  if (items == NULL) {
    return;
  }

  if (items) {
    int i;
    for (i = 0; i < items->len; i++) {
      g_object_unref (items->pdata[i]);
    }

    g_ptr_array_free (items, TRUE);
  }
}
#endif

static void
hrn_source_finalize (GObject *object)
{
  HrnSource        *self = (HrnSource *) object;
  HrnSourcePrivate *priv = self->priv;

  if (priv->index) {
      g_sequence_free (priv->index);
      priv->index = NULL;
  }

  if (priv->tree) {
      hrn_cluster_tree_free (priv->tree);
      priv->tree = NULL;
  }

  if (priv->uri_to_item) {
      g_hash_table_destroy (priv->uri_to_item);
      priv->uri_to_item = NULL;
  }

  G_OBJECT_CLASS (hrn_source_parent_class)->finalize (object);
}

static void
hrn_source_dispose (GObject *object)
{
  HrnSource        *self = (HrnSource *) object;
  HrnSourcePrivate *priv = self->priv;

  if (priv->client)
    {
      g_object_unref (priv->client);
      priv->client = NULL;
    }

  G_OBJECT_CLASS (hrn_source_parent_class)->dispose (object);
}

static void
hrn_source_set_property (GObject *object, guint prop_id, const GValue *value,
                         GParamSpec   *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_source_get_property (GObject *object, guint prop_id, GValue     *value,
                         GParamSpec *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_source_class_init (HrnSourceClass *klass)
{
  GObjectClass *o_class = (GObjectClass *) klass;

  o_class->dispose      = hrn_source_dispose;
  o_class->finalize     = hrn_source_finalize;
  o_class->set_property = hrn_source_set_property;
  o_class->get_property = hrn_source_get_property;

  g_type_class_add_private (klass, sizeof (HrnSourcePrivate));

  signals[ITEM_CHANGED] = g_signal_new ("item-changed",
                                        G_TYPE_FROM_CLASS (klass),
                                        G_SIGNAL_RUN_FIRST |
                                        G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                        g_cclosure_marshal_VOID__OBJECT,
                                        G_TYPE_NONE, 1, BKL_TYPE_ITEM);
  signals[ITEM_ADDED] = g_signal_new ("item-added",
                                      G_TYPE_FROM_CLASS (klass),
                                      G_SIGNAL_RUN_FIRST |
                                      G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                      g_cclosure_marshal_VOID__OBJECT,
                                      G_TYPE_NONE, 1, BKL_TYPE_ITEM);
  signals[ITEM_REMOVED] = g_signal_new ("item-removed",
                                        G_TYPE_FROM_CLASS (klass),
                                        G_SIGNAL_RUN_FIRST |
                                        G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                        g_cclosure_marshal_VOID__OBJECT,
                                        G_TYPE_NONE, 1, BKL_TYPE_ITEM);
  signals[STARTED] = g_signal_new ("index-started",
                                   G_TYPE_FROM_CLASS (klass),
                                   G_SIGNAL_RUN_FIRST |
                                   G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                   g_cclosure_marshal_VOID__VOID,
                                   G_TYPE_NONE, 0);
  signals[FINISHED] = g_signal_new ("index-finished",
                                    G_TYPE_FROM_CLASS (klass),
                                    G_SIGNAL_RUN_FIRST |
                                    G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                    g_cclosure_marshal_VOID__VOID,
                                    G_TYPE_NONE, 0);
}

static void
hrn_source_init (HrnSource *self)
{
  HrnSourcePrivate *priv;

  self->priv = GET_PRIVATE (self);
  priv       = self->priv;

  priv->uri_to_item = g_hash_table_new_full (g_str_hash, g_str_equal,
                                             NULL, g_object_unref);
}

static void
uri_changed_cb (BklSourceClient *client, const char      *uri,
                HrnSource       *source)
{
  HrnSourcePrivate *priv = source->priv;
  BklItem          *item;

  item = g_hash_table_lookup (priv->uri_to_item, uri);

  /* FIXME: This doesn't update the item.
     Bickley needs a bkl_item_update_from_db method */
  g_signal_emit (source, signals[ITEM_CHANGED], 0, item);
}

static void
insert_item (HrnSource *source,
             BklItem   *item)
{
  HrnSourcePrivate *priv = source->priv;

  hrn_cluster_tree_add_item (priv->tree, item);
  g_hash_table_insert (priv->uri_to_item,
                       (char *) bkl_item_get_uri (item), item);
}

static void
uri_added_cb (BklSourceClient *client, const char      *uri,
              HrnSource       *source)
{
  HrnSourcePrivate *priv = source->priv;
  BklItem          *item;
  GError           *error = NULL;

  item = bkl_db_get_item (priv->db, uri, &error);
  if (error != NULL)
    {
      g_warning ("Error getting item for %s: %s", uri, error->message);
      g_error_free (error);
      return;
    }

  insert_item (source, item);

  /* FIXME: Do we need this emitted now,
     as the signals are really emitted
     on the tree */
  g_signal_emit (source, signals[ITEM_ADDED], 0, item);
}

static void
uri_removed_cb (BklSourceClient *client,
                const char      *uri,
                HrnSource       *source)
{
  HrnSourcePrivate *priv = source->priv;
  BklItem          *item;

  item = g_hash_table_lookup (priv->uri_to_item, uri);
  if (item == NULL) {
      return;
  }

  hrn_cluster_tree_remove_item (priv->tree, item);
  /* FIXME: Do we need this signal emitted?
     Emit the signal before removing the item from the hash table
     as the hash table will unref the item */
  g_signal_emit (source, signals[ITEM_REMOVED], 0, item);
  g_hash_table_remove (priv->uri_to_item, uri);
}

static int
compare_words (gconstpointer a,
               gconstpointer b,
               gpointer      userdata)
{
    return strcmp (a, b);
}

static void
index_changed_cb (BklSourceClient *client,
                  const char     **added,
                  const char     **removed,
                  HrnSource       *source)
{
    GSequenceIter *iter;
    int i;

    /* Remove the words from the index first */
    for (i = 0; removed[i]; i++)
      {
        char *word;

        iter = g_sequence_search (source->priv->index, (char *) removed[i],
                                  (GCompareDataFunc) compare_words, NULL);
        word = g_sequence_get (iter);
        if (word && g_str_equal (word, removed[i]))
          {
            g_sequence_remove (iter);
          }
      }

    /* Add the new words */
    for (i = 0; added[i]; i++) {
        g_sequence_insert_sorted (source->priv->index, g_strdup (added[i]),
                                  (GCompareDataFunc) compare_words, NULL);
    }

    /* FIXME: Should update the search here */
}

static void
index_started_cb (BklSourceClient *client,
                  HrnSource       *source)
{
  g_signal_emit (source, signals[STARTED], 0);
}

static void
index_finished_cb (BklSourceClient *client,
                   HrnSource       *source)
{
  g_signal_emit (source, signals[FINISHED], 0);
}

HrnSource *
hrn_source_new (BklSourceClient *client)
{
  HrnSource        *source;
  HrnSourcePrivate *priv;
  GPtrArray        *items;
  GError           *error = NULL;
  int i;

  source = g_object_new (HRN_TYPE_SOURCE, NULL);
  priv   = source->priv;

  priv->client     = g_object_ref (client);
  priv->changed_id = g_signal_connect (client, "uri-changed",
                                       G_CALLBACK (uri_changed_cb), source);
  priv->added_id = g_signal_connect (client, "uri-added",
                                     G_CALLBACK (uri_added_cb), source);
  priv->removed_id = g_signal_connect (client, "uri-deleted",
                                       G_CALLBACK (uri_removed_cb), source);
  priv->index_changed_id = g_signal_connect (client, "index-changed",
                                       G_CALLBACK (index_changed_cb), source);
  priv->started_id = g_signal_connect (client, "in-progress",
                                       G_CALLBACK (index_started_cb), source);
  priv->finished_id = g_signal_connect (client, "complete",
                                        G_CALLBACK (index_finished_cb), source);

  priv->db = bkl_source_client_get_db (client);
  if (priv->db == NULL)
    {
      g_warning ("Error loading DB for %s",
                 bkl_source_client_get_path (client));
      g_object_unref (source);
      return NULL;
    }

  priv->index = bkl_db_get_index_words (priv->db);

  items = bkl_db_get_items (priv->db, FALSE, &error);
  if (items == NULL)
    {
      g_warning ("Error getting items: %s", error->message);
      g_error_free (error);

      g_object_unref (source);
      return NULL;
    }

  priv->tree = hrn_cluster_tree_new ();
  for (i = 0; i < items->len; i++) {
      BklItem *item = items->pdata[i];

      g_hash_table_insert (priv->uri_to_item,
                           (char *) bkl_item_get_uri (item), item);
      hrn_cluster_tree_add_item (priv->tree, item);
  }

  /* hrn_cluster_tree_dump_items (priv->tree); */
  g_ptr_array_free (items, TRUE);

  return source;
}

const char *
hrn_source_get_object_path (HrnSource *source)
{
  HrnSourcePrivate *priv = source->priv;

  return bkl_source_client_get_path (priv->client);
}

BklDB *
hrn_source_get_db (HrnSource *source)
{
  HrnSourcePrivate *priv = source->priv;

  return priv->db;
}

GSequence *
hrn_source_get_index (HrnSource *source)
{
  HrnSourcePrivate *priv = source->priv;

  return priv->index;
}

const char *
hrn_source_get_name (HrnSource *source)
{
  HrnSourcePrivate *priv = source->priv;

  return kozo_db_get_name (priv->db->db);
}

GPtrArray *
hrn_source_get_audio_items (HrnSource *source)
{
#if 0
  HrnSourcePrivate *priv = source->priv;

  return priv->audio_items;
#endif
  return NULL;
}

GPtrArray *
hrn_source_get_image_items (HrnSource *source)
{
#if 0
  HrnSourcePrivate *priv = source->priv;

  return priv->image_items;
#endif
  return NULL;
}

GPtrArray *
hrn_source_get_video_items (HrnSource *source)
{
#if 0
  HrnSourcePrivate *priv = source->priv;

  return priv->video_items;
#endif
  return NULL;
}

BklItem *
hrn_source_get_item (HrnSource *source, const char *uri)
{
  HrnSourcePrivate *priv = source->priv;

  return g_hash_table_lookup (priv->uri_to_item, uri);
}

/** query **/

struct _Bartowski
{
    int    word_count;
    GHashTable *chuck;
};

static void
flash_intersect (gpointer key, gpointer value, gpointer userdata)
{
  struct _Bartowski *bart  = (struct _Bartowski *) userdata;
  int                count = GPOINTER_TO_INT (value);

  /* We generate the intersection of the sets by including only
     uris that have a word count equal to the number of sets */
  if (count == bart->word_count) {
      char *data = g_strdup (key);

      g_hash_table_insert (bart->chuck, data, data);
  }
}


static GHashTable *
find_intersect (GPtrArray *sets)
{
  struct _Bartowski *bart;
  GHashTable        *set;
  int                i;

  set = g_hash_table_new (g_str_hash, g_str_equal);
  for (i = 0; i < sets->len; i++)
    {
      GList *results, *r;

      results = sets->pdata[i];

      /* Put each result into the hashtable */
      for (r = results; r; r = r->next)
        {
          gpointer key, value;

          /* If the result is already in the hash table
             increment its word count */
          if (g_hash_table_lookup_extended (set, r->data, &key, &value))
            {
              guint count = GPOINTER_TO_INT (value);
              g_hash_table_insert (set, r->data, GINT_TO_POINTER (count + 1));
            }
          else
            {
              g_hash_table_insert (set, r->data, GINT_TO_POINTER (1));
            }
        }
    }

  bart             = g_newa (struct _Bartowski, 1);
  bart->word_count = sets->len;
  bart->chuck      = g_hash_table_new_full (g_str_hash, g_str_equal,
                                            g_free, NULL);

  g_hash_table_foreach (set, flash_intersect, bart);
  g_hash_table_destroy (set);

  return bart->chuck;
}


static GPtrArray *
search_index_for_words (HrnSource *source, const char *text)
{
  char      *search_text;
  char     **search_terms;
  GPtrArray *words;
  int        k;
  GSequence *idx;


  search_text  = g_ascii_strup (text, -1);
  /* FIXME: Should we just remove all punctuation? */
  search_terms = g_strsplit_set (search_text, " :➜", 0);
  g_free (search_text);

  idx = hrn_source_get_index (source);

  /* Search each index for possible substrings in search_words */
  words = g_ptr_array_new ();
  for (k = 0; search_terms[k]; k++)
    {
      GPtrArray     *possible;
      GSequenceIter *iter;

      /* If @text ends in a space, then g_strsplit will insert a blank
         string in @search_terms */
      if (*(search_terms[k]) == '\0')
        {
          continue;
        }

      possible = g_ptr_array_new ();

      g_ptr_array_add (words, possible);

      iter = g_sequence_get_begin_iter (idx);

      while (!g_sequence_iter_is_end (iter))
        {
          char *word = g_sequence_get (iter);

          if (word)
            {
              char *new_word = g_filename_from_uri (word, NULL, NULL);

              /* The word may be an encoded uri, so we need to unencode it
                 before searching */
              if (new_word == NULL) {
                new_word = g_strdup (word);
              }

              if (strstr (new_word, search_terms[k]))
                {
                  g_ptr_array_add (possible, word);
                }

              g_free (new_word);
            }
          iter = g_sequence_iter_next (iter);
        }
    }

  g_strfreev (search_terms);

  return words;
}

static void
steal_keys (gpointer key, gpointer value, gpointer userdata)
{
  GList **results = (GList **) userdata;

  *results = g_list_append (*results, g_strdup ((char *) key));
}

static GPtrArray *
search_for_uris (HrnSource *source, GPtrArray *possible_words)
{
  GPtrArray *possible_uris;
  int        i;

  possible_uris = g_ptr_array_sized_new (possible_words->len);
  for (i = 0; i < possible_words->len; i++)
    {
      GPtrArray  *words   = possible_words->pdata[i];
      GList      *results = NULL;
      GHashTable *set;
      int         j;

      set = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
      for (j = 0; j < words->len; j++)
        {
          char   *word = words->pdata[j];
          GList  *r, *rr;
          GError *error = NULL;

          r = kozo_db_index_lookup (hrn_source_get_db (
                                      source)->db, word, &error);
          if (error != NULL)
            {
              g_warning ("Error getting results for %s: %s", word,
                         error->message);
              g_error_free (error);
              r = NULL;
            }

          /* Put uris into the hashtable so we can get only unique
             entries */
          for (rr = r; rr; rr = rr->next)
            {
              if (g_hash_table_lookup (set, rr->data) == NULL)
                {
                  g_hash_table_insert (set, rr->data, rr->data);
                }
              else
                {
                  g_free (rr->data);   /* Free data that doesn't go in @set */
                }
            }

          /* The contents of the list are freed either above, or when
             @set is destroyed */
          g_list_free (r);
        }

      /* Can't use g_hash_table_get_keys as it doesn't copy the data */
      g_hash_table_foreach (set, steal_keys, &results);
      g_hash_table_destroy (set);

      g_ptr_array_add (possible_uris, results);
    }

  return possible_uris;
}

void
hrn_source_filter (HrnSource   *source,
                   const gchar *text)
{
  HrnSourcePrivate *priv = source->priv;
  GHashTable *results;
  GPtrArray *index_words;
  GPtrArray *possible_results;
  int i;

  if (text == NULL || *text == '\0') {
      hrn_cluster_tree_filter_items (priv->tree, NULL);
      return;
  }

  /* @index_words is a GPtrArray containing GPtrArrays
     that contain all the possible words for each word in @text */
  index_words = search_index_for_words (source, text);

  /* @possible_results is a GPtrArray containing GLists
     that contain all the possible uris for each array of words
     in @index_words */
  possible_results = search_for_uris (source, index_words);

  for (i = 0; i < index_words->len; i++) {
      GPtrArray *words = index_words->pdata[i];

      /* We don't need to free the contents of words here because the words
         are owned by the GSequence */
      g_ptr_array_free (words, TRUE);
  }
  g_ptr_array_free (index_words, TRUE);

  results = find_intersect (possible_results);

  for (i = 0; i < possible_results->len; i++) {
      GList *res = possible_results->pdata[i];
      GList *iter;

      for (iter = res; iter; iter = iter->next) {
          g_free (iter->data);
      }
      g_list_free (res);
  }
  g_ptr_array_free (possible_results, TRUE);

  hrn_cluster_tree_filter_items (priv->tree, results);
  g_hash_table_destroy (results);

  hrn_cluster_tree_filter_nodes (priv->tree, text);
}

HrnClusterTree *
hrn_source_get_cluster_tree (HrnSource *source)
{
    HrnSourcePrivate *priv = source->priv;

    return priv->tree;
}
