/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: mailcap.cpp,v 1.1.2.1 2004/07/22 22:54:51 rggammon Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include "mailcap.h"

#include <glib.h>

#include <unistd.h>
#include <errno.h>
#include <stdio.h>

#include "hlxclib/string.h"

#include "commonapp.h"

/* Taken from our .keys file */
static const gchar* g_mime_types[] =
{
    "application/x-ogg",
    "application/ogg",
    "text/vnd.rn-realtext",        
    "image/vnd.rn-realpix",
    "application/smil",

#ifdef HELIX_FEATURE_REAL_BRANDING
    "audio/mp3",
    "audio/x-mp3",
    "audio/mpeg",
    "audio/mpg",
    "audio/x-mpeg",
    "audio/x-mpg",
    "audio/mpegurl",
    "audio/x-mpegurl",        
    "audio/wav",
    "audio/x-wav",
    "audio/x-pn-wav",
    "audio/x-pn-windows-acm",
    "audio/x-pn-windows-pcm",
    "application/vnd.rn-realmedia-secure",
    "application/vnd.rn-realaudio-secure",
    "audio/x-realaudio-secure",
    "video/vnd.rn-realvideo-secure",
    "audio/vnd.rn-realaudio",
    "audio/x-realaudio",
    "application/vnd.rn-realmedia",
    "application/vnd.rn-realmedia-vbr",
    "image/vnd.rn-realpix",
    "audio/x-pn-realaudio",
    "video/vnd.rn-realvideo",
    "application/vnd.rn-realsystem-rmj",
    "application/vnd.rn-realsystem-rmx",
    "audio/aac",
    "audio/m4a",
    "audio/mp2",
    "audio/mp1",
    "audio/rn-mpeg",
    "audio/scpls",
    "audio/x-scpls",
    "application/streamingmedia",
    "application/sdp",
    "application/x-sdp",
    "audio/basic",
    "audio/x-pn-au",
    "audio/aiff",
    "audio/x-aiff",
    "audio/x-pn-aiff",
    "video/3gpp",
    "video/3gpp-encrypted",
    "audio/3gpp",
    "audio/3gpp-encrypted",
    "audio/amr",
    "audio/amr-encrypted",
    "audio/amr-wb",
    "audio/amr-wb-encrypted",
    "audio/x-rn-3gpp-amr",
    "audio/x-rn-3gpp-amr-encrypted",
    "audio/x-rn-3gpp-amr-wb",
    "audio/x-rn-3gpp-amr-wb-encrypted",
    "video/3gpp2",
    "audio/3gpp2"
#endif
};

static void
append_mime_types_to_be_claimed(GList* exceptions_list)
{
    GError* error = NULL;
    GIOChannel* chan = NULL;
    gboolean result = FALSE;
    GIOStatus status;
    GList* iter;
    guint next_exception;
    guint i;

    const gchar* home;
    gchar* mailcap_path;

    /* Append the mime types listed above to ~/.mailcap */
    home = g_get_home_dir();
    mailcap_path = g_build_path(G_DIR_SEPARATOR_S, home, ".mailcap", NULL);

    chan = g_io_channel_new_file(mailcap_path, "a", &error);    

    g_free(mailcap_path);

    if(error)
    {
        g_warning(error->message);
        g_error_free(error);
        error = NULL;
        result = FALSE;
    }

    if(!chan)
    {
        result = FALSE;
    }

    iter = g_list_first(exceptions_list);
    if(iter)
    {
        next_exception = (gint) iter->data;
    }

    gchar* app_cmd = hxcommon_get_rel_app_path(APP_NAME_SHORT);

    for(i = 0; i < sizeof(g_mime_types) / sizeof(*g_mime_types); i++)
    {
        gsize length;
        gsize bytes_written;
        gchar* line;

        /* See if this type is an exception */
        if(iter)
        {
            if(i == next_exception)
            {
                iter = g_list_next(iter);
                if(iter)
                {
                    next_exception = (gint) iter->data;
                }
                continue;
            }
        }

        line = g_strdup_printf("%s;%s %%s\n", g_mime_types[i], app_cmd);

        length = strlen(line);
        
        do
        {
            status = g_io_channel_write_chars(chan, line, length, &bytes_written, &error);
        } while(status == G_IO_STATUS_AGAIN);

        if(error)
        {
            g_warning(error->message);
            g_error_free(error);
            error = NULL;
            result = FALSE;
            break;
        }
            
        if(length != bytes_written || status != G_IO_STATUS_NORMAL)
        {
            break;
        }
    }

    g_free(app_cmd);
}

static gboolean
remove_mime_types_to_be_claimed(GList** exceptions_list_ptr)
{
    GError* error = NULL;
    GIOChannel* orig_chan = NULL;
    GIOChannel* working_chan = NULL;
    gboolean result = TRUE;
    GIOStatus status;
    guint i;
    gchar* line = NULL;
    GList* exceptions_list = *exceptions_list_ptr;

    const gchar* home;
    gchar* mailcap_path;
    
    home = g_get_home_dir();
    mailcap_path = g_build_path(G_DIR_SEPARATOR_S, home, ".mailcap", NULL);

    if(!g_file_test(mailcap_path, G_FILE_TEST_EXISTS))
    {
        /* No ~/.mailcap, nothing to do */
        g_free(mailcap_path);
        return TRUE;
    }
    
    const gchar* tmpdir;
    gchar* working_mailcap_path;
    gchar* working_mailcap_name;

    working_mailcap_name = g_strdup_printf("mailcap.%d", getpid());
    tmpdir = g_get_tmp_dir();
    working_mailcap_path = g_build_path(G_DIR_SEPARATOR_S, tmpdir, working_mailcap_name, NULL);
    g_free(working_mailcap_name);

    /* Build a mailcap without lines that assign mime types we want to claim,
       and replace ~/.mailcap with this file. */

    orig_chan = g_io_channel_new_file(mailcap_path, "r", &error);
    if(error)
    {
        g_warning(error->message);
        g_error_free(error);
        error = NULL;
        result = FALSE;
    }
    
    if(result)
    {
        working_chan = g_io_channel_new_file(working_mailcap_path, "w", &error);

        if(error)
        {
            g_warning(error->message);
            g_error_free(error);
            error = NULL;
            result = FALSE;
        }
    }
    
    if(!working_chan || !orig_chan)
    {
        result = FALSE;
    }
    
    for(;;)
    {
        gsize length = 0;
        gboolean suppress_line = FALSE;

        g_free(line);
        
        do
        {
            status = g_io_channel_read_line(orig_chan, &line, &length, NULL, &error);
        } while(status == G_IO_STATUS_AGAIN);

        if(error)
        {
            g_warning(error->message);
            g_error_free(error);
            error = NULL;
            result = FALSE;
            break;
        }
        
        if(status == G_IO_STATUS_EOF)
        {
            break;
        }
        
        if(!line || status != G_IO_STATUS_NORMAL)
        {
            result = FALSE;
            break;
        }

        /* process line */
        for(i = 0; i < sizeof(g_mime_types) / sizeof(*g_mime_types); i++)
        {
            if(strncasecmp(line, g_mime_types[i], strlen(g_mime_types[i])) == 0)
            {
#ifdef HELIX_FEATURE_REAL_BRANDING
                suppress_line = TRUE;
#else
                /* We're Helix Player. Don't claim if the type is
                   assigned to the realplayer */
                if(strstr(line, "realplay"))
                {
                    /* Add this to the exceptions list so Helix Player
                       doesn't claim it. */
                    exceptions_list = g_list_append(exceptions_list, (gpointer) i);
                }
                else
                {
                    suppress_line = TRUE;
                }
#endif
            }
        }

        if(!suppress_line)
        {
            gsize bytes_written = 0;
    
            /* copy line over to the new mailcap as is */
            do
            {
                status = g_io_channel_write_chars(working_chan, line, length, &bytes_written, &error);
            } while(status == G_IO_STATUS_AGAIN);

            if(error)
            {
                g_warning(error->message);
                g_error_free(error);
                error = NULL;
                result = FALSE;
                break;
            }
            
            if(length != bytes_written || status != G_IO_STATUS_NORMAL)
            {
                result = FALSE;
                break;
            }
        }
    }

    if(working_chan)
    {
        g_io_channel_shutdown(working_chan, TRUE, &error);
        if(error)
        {
            g_warning(error->message);
            g_error_free(error);
            error = NULL;
            result = FALSE;
        }
    }

    if(orig_chan)
    {
        g_io_channel_shutdown(orig_chan, TRUE, &error);
        if(error)
        {
            g_warning(error->message);
            g_error_free(error);
            error = NULL;
            result = FALSE;
        }
    }

    if(result)
    {
        /* Move our modified .mailcap over the old one */
        int err;

        /* poor man's gnome_vfs_move() */
        err = rename(working_mailcap_path, mailcap_path);
        if(err < 0)
        {
            g_warning(strerror(errno));
            result = FALSE;
            unlink(working_mailcap_path);
        }
    }
    
    g_free(working_mailcap_path);
    g_free(mailcap_path);

    *exceptions_list_ptr = exceptions_list;

    return result;
}

void
mailcap_update(void)
{
    gboolean result;
    GList* exceptions_list = NULL;
    
    result = remove_mime_types_to_be_claimed(&exceptions_list);
    if(result)
    {
        append_mime_types_to_be_claimed(exceptions_list);
    }

    g_list_free(exceptions_list);
}
