{-# OPTIONS -fglasgow-exts -cpp #-}
module HAppS.Protocols.HTTP.Handler(request,presult) where

import Control.Exception as E
import Control.Monad
import Data.List(foldl')
import Data.Char(toLower)
import Data.Maybe ( fromMaybe, fromJust )
import qualified Data.List as List
import qualified Data.ByteString.Char8 as P
import qualified Data.ByteString.Lazy.Char8 as L
import qualified Data.Map as M
import System.IO
import System.Locale(defaultTimeLocale)
import System.Time

import HAppS.Protocols.HTTP.Clock
import HAppS.Protocols.HTTP.LazyLiner
import HAppS.Protocols.HTTP.Types
import HAppS.Protocols.HTTP.Multipart
import HAppS.Protocols.HTTP.RFC822Headers
import HAppS.Protocols.SURI(SURI(..))
import HAppS.Protocols.SURI.ParseURI
import HAppS.Util.ByteStringCompat
import HAppS.Util.TimeOut
#ifdef HAPPS_VERBOSE_HTTP
import Control.Concurrent(myThreadId)
#endif

p = P.pack
upa = P.unsafePackAddress
pa = P.packAddress

{-# INLINE debugPut #-}
debugPut :: P.ByteString -> IO ()
#ifdef HAPPS_VERBOSE_HTTP
debugPut x = do t <- myThreadId
                c <- getClockTime
                P.putStrLn (P.pack (show t++" @ "++show c++" >> ") `P.append` x)
                hFlush stdout
#else
debugPut _ = return ()
#endif

request :: Conf -> Handle -> Host -> (Request -> IO Result) -> IO ()
request conf h host handler = rloop conf h host handler =<< L.hGetContents h

rloop conf h host handler inputStr
    = join $ withTimeOut (30 * second) $
      do let parseRequest
                 = do (topStr, restStr) <- splitAtEmptyLine inputStr
                      (rql, headerStr) <- splitAtCRLF topStr
                      let (m,u,v) = requestLine rql
                      headers' <- parseHeaders "host" (L.unpack headerStr)
                      let headers = M.fromList $ map (\(x,y) -> (P.pack x, P.pack y)) headers'
                      let contentLength = fromMaybe 0 $ fmap fst (P.readInt =<< M.lookup contentlengthC headers)
                          (body, nextRequest)
                              | contentLength < 0              = error "Negative content-length"
                              | contentLength == 0             = (L.empty, restStr)
                              | contentLength > bodyLimit conf = error "Too large body in HTTP request"
                              | otherwise                      = L.splitAt (fromIntegral contentLength) restStr
                      L.length body `seq` return (Request m u v (Headers headers) (Body (P.concat $ L.toChunks body)) host, nextRequest)
         case parseRequest of
           Nothing -> error "failed to parse HTTP request"
           Just (req, rest)
               -> return $
                  do let ioseq act = act >>= \x -> x `seq` return x
                     res <- ioseq (handler req) `E.catch` \_ -> sresult 500 "Server error"
                     putAugmentedResult h req res
                     when (continueHTTP req res) $ rloop conf h host handler rest



presult :: Handle -> IO Result
presult h = do
    liner <- newLinerHandle h

    rls@(rql:hrl) <- headerLines liner
    debugPut $ P.unlines rls
    let c = responseLine rql
        hh = combineHeaders $ headers hrl []

    let mread k = return (fmap fst (P.readInt =<< M.lookup k hh))
    cl   <- mread contentlengthC
    body <- case cl of
              Nothing               -> fmap toChunks $ getRest liner
              Just c | c < 0        -> fail "Negative content-length"
                     | otherwise    -> fmap toChunks $ getBytes liner c
    return $ Result c (Headers hh) nullRsFlags body

headers []          acc = acc
headers (line:rest) acc =
  let space = let x = P.head line in x == ' ' || x == '\t' in
  case () of
    _ | space && null acc -> error "Continuation header as first header"
      | space             -> let ((k,v):r) = acc in headers rest ((k,P.append v line):r)
      | otherwise         -> let (k,raw) = breakChar ':' line
                                 v       = dropSpaceEnd $ dropSpace $ P.tail raw
                                 in headers rest ((k,v):acc)
  
requestLine l = case map (P.concat . L.toChunks) $ L.words l of
                  [rq,uri,ver] -> (method rq, SURI $ parseURIRef uri, version ver)

responseLine l = case P.words l of (v:c:_) -> version v `seq` fst (fromJust (P.readInt c))

method r = fj $ lookup r mtable
    where fj (Just x) = x
          fj Nothing  = error ("invalid request method")
          mtable = [(upa 3 "GET"#,    GET),
                    (upa 4 "HEAD"#,   HEAD),
                    (upa 4 "POST"#,   POST),
                    (upa 3 "PUT"#,    PUT),
                    (upa 6 "DELETE"#, DELETE),
                    (upa 5 "TRACE"#,  TRACE),
                    (upa 7 "OPTIONS"#,OPTIONS),
                    (upa 7 "CONNECT"#,CONNECT)]


combineHeaders :: [(P.ByteString,P.ByteString)] -> M.Map P.ByteString P.ByteString
combineHeaders = foldl' w M.empty
    where w m (k,v) = M.insertWith (\n o -> P.concat [o, upa 3 ", "#,n]) (P.map toLower k) v m

-- Result side

putAugmentedResult :: Handle -> Request -> Result -> IO ()
putAugmentedResult h req res = do
  let ph k v = [k, fsepC, v, crlfC]
  let f n d  = if hasHeader' n $ rsHeaders res then [] else ph n d
  raw <- getApproximateTime
  let cl = foldl' (\o a -> o + P.length a) 0 $ rsBody res
  let put x = debugPut x >> P.hPut h x
  put $ P.concat $ concat
    ((pversion $ rqVersion req) : [responseMessage $ rsCode res] :
     f serverC      happsC :
     f contentTypeC textHtmlC :
     f dateC        (p (formatCalendarTime defaultTimeLocale "%a, %d %b %Y %X GMT" (toUTCTime raw))) :
     (if continueHTTP req res then [] else [connectionCloseC]) :
     (if rsfContentLenth (rsFlags res) then f contentLengthC (p (show cl)) else []) :
     ([ ph k v | (k,v) <- M.toList $ unHeaders $ rsHeaders res] ++ [[crlfC]]))
  when (rqMethod req /= HEAD) $ mapM_ (P.hPut h) $ rsBody res
  hFlush h

-- Version

pversion (Version 1 1) = [http11]
pversion (Version 1 0) = [http10]
pversion (Version x y) = [p "HTTP/",p (show x), p ".", p (show y)]

version x | x == http09 = Version 0 9
          | x == http10 = Version 1 0
          | x == http11 = Version 1 1
          | otherwise   = error "Invalid HTTP version"

http09 = (upa 8 "HTTP/0.9"#)
http10 = (upa 8 "HTTP/1.0"#)
http11 = (upa 8 "HTTP/1.1"#)

-- Constants

connectionCloseC = upa 19 "Connection: close\r\n"#
crlfC            = upa  2 "\r\n"#
fsepC            = upa  2 ": "#
contentTypeC     = upa 12 "Content-Type"#
contentLengthC   = upa 14 "Content-Length"#
contentlengthC   = upa 14 "content-length"#
dateC            = upa  4 "Date"# 
serverC          = upa  6 "Server"#
happsC           = upa 11 "HAppS/0.8.4"#
textHtmlC        = upa 24 "text/html; charset=utf-8"#

-- Response code names

responseMessage 100 = pa " 100 Continue\r\n"#
responseMessage 101 = pa " 101 Switching Protocols\r\n"#
responseMessage 200 = pa " 200 OK\r\n"#
responseMessage 201 = pa " 201 Created\r\n"#
responseMessage 202 = pa " 202 Accepted\r\n"#
responseMessage 203 = pa " 203 Non-Authoritative Information\r\n"#
responseMessage 204 = pa " 204 No Content\r\n"#
responseMessage 205 = pa " 205 Reset Content\r\n"#
responseMessage 206 = pa " 206 Partial Content\r\n"#
responseMessage 300 = pa " 300 Multiple Choices\r\n"#
responseMessage 301 = pa " 301 Moved Permanently\r\n"#
responseMessage 302 = pa " 302 Found\r\n"#
responseMessage 303 = pa " 303 See Other\r\n"#
responseMessage 304 = pa " 304 Not Modified\r\n"#
responseMessage 305 = pa " 305 Use Proxy\r\n"#
responseMessage 307 = pa " 307 Temporary Redirect\r\n"#
responseMessage 400 = pa " 400 Bad Request\r\n"#
responseMessage 401 = pa " 401 Unauthorized\r\n"#
responseMessage 402 = pa " 402 Payment Required\r\n"#
responseMessage 403 = pa " 403 Forbidden\r\n"#
responseMessage 404 = pa " 404 Not Found\r\n"#
responseMessage 405 = pa " 405 Method Not Allowed\r\n"#
responseMessage 406 = pa " 406 Not Acceptable\r\n"#
responseMessage 407 = pa " 407 Proxy Authentication Required\r\n"#
responseMessage 408 = pa " 408 Request Time-out\r\n"#
responseMessage 409 = pa " 409 Conflict\r\n"#
responseMessage 410 = pa " 410 Gone\r\n"#
responseMessage 411 = pa " 411 Length Required\r\n"#
responseMessage 412 = pa " 412 Precondition Failed\r\n"#
responseMessage 413 = pa " 413 Request Entity Too Large\r\n"#
responseMessage 414 = pa " 414 Request-URI Too Large\r\n"#
responseMessage 415 = pa " 415 Unsupported Media Type\r\n"#
responseMessage 416 = pa " 416 Requested range not satisfiable\r\n"#
responseMessage 417 = pa " 417 Expectation Failed\r\n"#
responseMessage 500 = pa " 500 Internal Server Error\r\n"#
responseMessage 501 = pa " 501 Not Implemented\r\n"#
responseMessage 502 = pa " 502 Bad Gateway\r\n"#
responseMessage 503 = pa " 503 Service Unavailable\r\n"#
responseMessage 504 = pa " 504 Gateway Time-out\r\n"#
responseMessage 505 = pa " 505 HTTP Version not supported\r\n"#
responseMessage x   = p (show x ++ "\r\n")

