{-# OPTIONS -fglasgow-exts -cpp #-}
module HAppS.MACID.Transaction where

import Control.Concurrent
import Control.Concurrent.STM
import Control.Exception(handle,throw,Exception)
import Control.Monad.State
import qualified Data.Map as M
import qualified Data.Set as S
import System.IO.Unsafe
import System.Mem.StableName(makeStableName)
import System.Random
import System.Time(getClockTime,ClockTime(TOD))
import System.Log.Logger

import HAppS.MACID.Logger
import HAppS.MACID.Monad
import HAppS.MACID.Saver
import HAppS.MACID.Serialize
import HAppS.MACID.SideEffect
import HAppS.MACID.Types
import HAppS.MACID.Var

logMT = logM "HAppS.MACID.Transaction"

-- | Get current event.
getEvent :: Ev st ev ev
getEvent = sel (txEvent . evEvent)

getTime :: Ev st ev EpochTime
getTime = sel (txTime . evEvent)

getEventId :: Ev st ev TxId
getEventId = sel (txId . evEvent)

sap :: Monad m => (StateT st m (a -> b)) -> (st -> m (a,st)) -> (StateT st m b)
sap mf fun = do x <- StateT fun
                f <- mf
                return $ f x


instance Serialize a => Serialize (TxContext a) where
    typeString x = "TxContext " ++ (typeString $ untx x)
        where untx :: Proxy (TxContext a) -> Proxy a; untx _ = undefined
    decodeStringM ('1':r) = do (a,r) <- decodeStringM r
                               (b,r) <- decodeStringM r
                               (c,r) <- decodeStringM r
                               (d,r) <- decodeStringM r
                               return (TxContext a b c d, r)
    decodeStringM (c:_)  = fail ("Unknown version for event context - "++show c)
    decodeStringM []     = fail ("Empty string when decoding TxContext")
    encodeStringM (TxContext a b c d) = liftM concat $ sequence l
        where l = [return "1",encodeStringM a, encodeStringM b, encodeStringM c, encodeStringM d]

instance Serialize StdGen where
    typeString _  = "System.Random.StdGen"
    decodeStringM = defaultDecodeStringM
    encodeStringM = defaultEncodeStringM

type EvLoaders = M.Map String (String -> S.Set TxId -> IO (TxId,String))

proxy :: t -> Proxy t
proxy _ = Proxy

{- Durablity:
* Pending queue is TVar ([TxContext evt],Int)
* Get events from the input sources in circular fashion
* Dump events on disk before adding to pending queue
* Checkpoints as follows:
  * check point event arrives from one of the input sources
  * write a new checkpoint file with:
    + list of pending transactions (all non-pending are out of system)
    + next txid
    + save state
    + rotare log files
  * resume transaction processing

FIXME

* Delete events after a fixed number of failures

-}

type Runner ev res = IO (IO ev, res -> IO ())
type EH i o = i -> IO o
type E2EHIO st ev res = Ev st ev (IO res) -> ev -> IO res


#ifndef __HADDOCK__
data Handler st where
    (:=>)   :: (LogFormat ev, Serialize (TxContext ev)) => Runner ev res -> Ev st ev res -> Handler st
    (:=>>)  :: (LogFormat ev, Serialize (TxContext ev)) => Runner ev res -> Ev st ev (IO res) -> Handler st
    IoH     :: IO (Handler st) -> Handler st
    SyncH   :: (LogFormat ev, Serialize (TxContext ev)) => IO (E2EHIO st ev res -> EH ev res, EH ev res -> IO ()) -> Handler st
    LABEL   :: String -> Handler st -> Handler st

data IHandler st where
    IHandler :: (LogFormat ev, Serialize (TxContext ev)) => String -> IO ev -> (res -> IO ()) -> Ev st ev (IO res) -> IHandler st
    ISyncH   :: (LogFormat ev, Serialize (TxContext ev)) => String -> (E2EHIO st ev res -> EH ev res) -> (EH ev res -> IO ()) -> IHandler st

#else 
-- |
--
-- > data Handler st where
-- >    (:=>)   :: Serialize (TxContext ev) => Runner ev res -> Ev st ev res -> Handler st
-- >    (:=>>)  :: Serialize (TxContext ev) => Runner ev res -> Ev st ev (IO res) -> Handler st
-- >    SyncH   :: Serialize (TxContext ev) => IO (E2EHIO st ev res -> EH ev res, EH ev res -> IO ()) -> Handler st
-- >    LABEL :: String -> Handler st -> Handler st
--
data Handler = Handler

#endif


handlerToI :: Handler st -> IO (IHandler st)
handlerToI x = do { y <- deIO x; res (handlerTString y) (baseHandler y) }
    where res ts (ior :=> run)  = do (i,o) <- ior
                                     return $! IHandler ts i o (fmap return run)
          res ts (ior :=>> run) = do (i,o) <- ior
                                     return $! IHandler ts i o run
          res ts (SyncH hand)   = do (i,o) <- hand
                                     return $! ISyncH ts i o
          res _  (IoH _)        = fail "Unexpected Handler: IoH"
          res _  (LABEL _ _)    = fail "Unexpected Handler: LABEL" 
          deIO (IoH x)     = x
          deIO (LABEL s x) = deIO x >>= \v -> return (LABEL s v)
          deIO x           = return x
          

data IHR st = forall res ev. (Serialize (TxContext ev))
    => IHR String 
           (TxContext ev) 
           (TxContext ev -> st -> TxConfig -> IO (Maybe st,Maybe TxConfig, IO ()))
data Res a = Ok a | Retry | Error Exception
type TxIdSource = MVar TxId
data TxRun st   = TxRun !(MVar (IHR st)) !(Chan SideEffect) !TxIdSource !LogChan


newTxId :: TxIdSource -> IO TxId
newTxId mv = modifyMVar mv $ \x -> return (x+1,x)

createEventLoaders :: Monad m => TxRun st -> [IHandler st] -> m EvLoaders
createEventLoaders txrun = foldM w M.empty . map (createEventLoader txrun)
  where w m (k,v) | k `M.member` m = fail ("Multiple events with the same type: "++k++". Add a label to the handlers.")
                  | otherwise      = return $! M.insert k v m

createEventLoader :: TxRun st -> IHandler st -> (String,String -> S.Set TxId -> IO (TxId,String))
createEventLoader (TxRun queue sideQueue tisrc logger) (IHandler tstring _ out run) = (tstring,load)
    where load s ignore = do
              logMT NOTICE ("Loading H-event of type: "++show tstring)
              (ev,r) <- decodeStringM s
              sq <- if txId ev `S.member` ignore then logMT NOTICE "> ignoring side-effects" >> newChan else return sideQueue
              handleEvent (TxRun queue sq tisrc logger) tstring ev out run
              return (txId ev, r)
createEventLoader (TxRun queue sideQueue tisrc logger) (ISyncH tstring bfun _)      = (tstring, w)
    where w s ignore = do
              logMT NOTICE ("Loading S-event of type: "++show tstring)
              (ev,r) <- decodeStringM s
              sq <- if txId ev `S.member` ignore then logMT NOTICE "> ignoring side-effects" >> newChan else return sideQueue
              let rfun = bfun $ \evh _ -> do mv <- newEmptyMVar
                                             handleEvent (TxRun queue sq tisrc logger) tstring ev (putMVar mv) evh
                                             takeMVar mv
              rfun $ txEvent ev
              return (txId ev, r)

handlerTString :: Handler st -> String
handlerTString (LABEL s h) = s ++ "///" ++ handlerTString h
handlerTString (x :=> _)   = typeString (y x) where y :: IO (IO a,b) -> Proxy (TxContext a); y _ = Proxy
handlerTString (x :=>> _)  = typeString (y x) where y :: IO (IO a,b) -> Proxy (TxContext a); y _ = Proxy
handlerTString (SyncH x)   = typeString (y x) where y :: IO (a,(t -> b) -> c) -> Proxy (TxContext t); y _ = Proxy
handlerTString (IoH _)     = error "handlerTString for IoH"

baseHandler :: Handler st -> Handler st
baseHandler (LABEL _ x) = baseHandler x
baseHandler x           = x


runHandler :: TxRun st -> IHandler st -> IO ()
runHandler txrun@(TxRun _ _ tisrc _) (ISyncH tstring bfun real) = real $ bfun $ \evh ev -> do
    mv  <- newEmptyMVar
    cev <- addTxContext tisrc ev
    handleEvent txrun tstring cev (putMVar mv) evh
    takeMVar mv
runHandler txrun@(TxRun _ _ tisrc _) (IHandler tstring inp out run) = do
    let loop = do
        ev  <- inp
        tev <- addTxContext tisrc ev
        handleEvent txrun tstring tev out run
        loop
    loop

handleEvent :: (Serialize (TxContext ev), LogFormat ev)
               => TxRun st -> String -> TxContext ev 
            -> (res -> IO ()) 
            -> Ev st ev (IO res) 
            -> IO ()
handleEvent (TxRun queue sideQueue _ logger) tstring inp ofun (Ev run) = do
    let signal = putMVar queue $ IHR tstring inp h
        c tx st = do
          sar <- newRefSTM []
          str <- newRefSTM st
          rrr <- newRefSTM $ txStdGen tx
          let env = Env { evSideEffects = sar,
                          evEvent = tx,
                          evState = str,
                          evBackgroundIOCompletion = signal,
                          evRandoms = rrr
--                          evCreateEvent = \ev -> do cev <- addTxContext tisrc ev
--                                                    putMVar queue $ IHR tstring cev h
                        }
          res <- run env
          sav <- readRefSTM sar
          str <- readRefSTM str
          return $ Ok (res,sav,str)
        eh tx e   = do addToLog logger tx e
                       logMT ERROR ("handleEvent FAIL "++show e)
                       return (Nothing,Nothing,ofun (throw e))
        h tx s0 _ = handle (eh tx) $ do
          sn0 <- makeStableName s0
          x   <- atomically $ orElse (c tx s0) (return Retry)
          case x of
            Ok (res, sar, st) -> do let runSideEffects
                                            = unless (null sar) $ writeChan sideQueue $ SideEffect (txId tx) $ reverse sar
                                    sn <- makeStableName st
                                    return (if sn == sn0 && null sar then Nothing else Just st
                                           , Nothing
                                           , runSideEffects >> res >>= ofun)
            Error err         -> eh tx err
            Retry             -> do logMT WARNING ("RETRY: transaction id = "++show (txId tx))
                                    return (Nothing, Nothing, return ())
    signal


addTxContext :: TxIdSource -> a -> IO (TxContext a)
addTxContext mv x = do
  TOD sec _  <- getClockTime
  txid <- newTxId mv
  sgen <- modifyMVar globalRandomGen (return . split)
  return $ TxContext txid (fromIntegral sec) sgen x

{-# NOINLINE globalRandomGen #-}
globalRandomGen :: MVar StdGen
globalRandomGen = unsafePerformIO (newMVar =<< getStdGen)


data TxConfig = TxConfig
    { txcConcurrency         :: Int,       -- ^ Number of concurrent threads for event processing
      txcSideEffectThreads   :: Int,       -- ^ Number of concurrent threads for executing side-effects.
      txcCheckpointSeconds   :: Seconds,   -- ^ Perform checkpoint at least every N seconds.
      txcCheckpointEvents    :: Int,       -- ^ Perform checkpoint at least every N events. 
      txcSaver               :: Saver,     -- ^ Saver used
      txcSaverImpl_          :: SaverImpl, -- ^ Internal, event saver
      txcFileNum_            :: Int,       -- ^ Internal
      txcLogger              :: Saver      -- ^ Saver used for logging
    }

nullTxConfig :: TxConfig
nullTxConfig = TxConfig { txcConcurrency         = 4,
                          txcSideEffectThreads   = 60,
                          txcCheckpointSeconds   = 3600,
                          txcCheckpointEvents    = 10000,
                          txcFileNum_            = error "TxConfig: txcFileNum must be defined by recover",
                          txcSaver               = error "TxConfig: txcSaver must be defined by recover",
                          txcSaverImpl_          = error "TxConfig: txcSaverImpl_ must be defined by recover",
                          txcLogger              = error "TxConfig: txcLogger must be specified"
                        }

runTxLoop :: MVar (IHR st) -> st -> TxConfig -> IO ()
runTxLoop queue st0 conf0 = do
  stateLock <- newMVar (st0,conf0)
  let loop = handle print (do
      IHR tstring ev fun <- takeMVar queue
      logMT NOTICE $ ("> Event "++show (txId ev)++" of "++tstring)
      act <- modifyMVar stateLock $ \(st,conf) -> do
--          putStrLn (">>> before "++show tstring)
          (mst,mc,ra) <- fun ev st conf
--          putStrLn (">>> after "++show tstring)
          let c = maybe conf id mc
          case mst of
            -- State was not updated and there are no side-effects.
            --
            -- Thus the return action can be run without holding
            -- the lock.
            Nothing  -> do -- putStrLn (">>> leaving pure for "++show tstring)
                           return ((st,c),logMT NOTICE "> pure" >> ra)
            -- There is a new State and there may be side-effects.
            --
            -- We use the saver here holding the lock. This is because saving
            -- must be ordered in the current implementation. A non-ordered
            -- saver architecture would be possible.
            -- Note that saverAdd can return without yet writing the result
            -- as long as:
            -- 1) saverAdd calls honor the sequence in which they were made.
            -- 2) saverAdd calls execute the finalizers only after the value
            --    has been serialized. The finalizers typically return the
            --    result to the user so they should not be kept
            --    waiting too long.
            -- 3) This means that checkpoints need to flush the saver
            --    which will guarantee that all pending result/side-effects
            --    have been processed.
            -- 4) Savers must *not* block while running the finalizers
            Just st' -> do a <- encodeFPS (Wrap tstring)
                           b <- encodeFPS ev
--                           putStrLn (">>> leaving with disk: before saver "++show tstring)
                           saverAdd (txcSaverImpl_ c) (a ++ b) (logMT NOTICE "> disk " >> ra)
--                           putStrLn (">>> leaving with disk: after saver "++show tstring)
                           return ((st',c), return ())
      act) >> loop
  replicateM_ (txcConcurrency conf0) $ forkIO loop

