// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Thread for calculating hashes
// ****************************************************************************

#include <QtCore>

#include "common.h"
#include "device.h"
#include "threadhash.h"
#include "threadwrite.h"
#include "hash.h"

class t_ThreadHashLocal
{
   public:
      t_pDevice pDevice;
};

t_ThreadHash::t_ThreadHash(void)
{
   CHK_EXIT (ERROR_THREADHASH_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 not initialised


t_ThreadHash::t_ThreadHash (t_pDevice pDevice)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADHASH_CONSTRUCTOR_NOT_SUPPORTED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADHASH_LIBEWF_FAILED))
      Initialised = true;
   }

   pOwn = new t_ThreadHashLocal;
   pOwn->pDevice = pDevice;

   CHK_QT_EXIT (connect (this, SIGNAL(finished()), this, SLOT(SlotFinished())))
}

t_ThreadHash::~t_ThreadHash (void)
{
   delete pOwn;
}

void t_ThreadHash::run (void)
{
   t_pDevice          pDevice;
   t_pFifoBlock       pFifoBlock;
   bool                Finished;
   quint64           *pBlocks;
   quint64             BlocksCalculated = 0;
   quint64             BlocksVerified   = 0;
   t_HashContextMD5    HashContextMD5;
   t_HashContextSHA256 HashContextSHA256;
   bool                VerifyLoop = false;

   LOG_INFO ("Acquisition of %s: Hash thread started", QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice))
   pDevice = pOwn->pDevice;
   pBlocks = &BlocksCalculated;
   for (;;)  // The whole loop is done 2 times if the user chose to do a source verification, 1 time otherwise.
   {
      *pBlocks = 0;
      Finished = false;
      CHK_EXIT (HashMD5Init    (&HashContextMD5   ))
      CHK_EXIT (HashSHA256Init (&HashContextSHA256))
      do
      {
         CHK_EXIT (pDevice->pFifoHashIn->Get (pFifoBlock))
         if (pFifoBlock)
         {
            (*pBlocks)++;
            CHK_EXIT (HashMD5Append    (&HashContextMD5   , pFifoBlock->Buffer, pFifoBlock->DataSize))
            CHK_EXIT (HashSHA256Append (&HashContextSHA256, pFifoBlock->Buffer, pFifoBlock->DataSize))
            if (VerifyLoop)
            {
               pDevice->IncCurrentVerifyPos(pFifoBlock->DataSize);
               CHK_EXIT (t_Fifo::Destroy              (pFifoBlock))
            }
            else
            {
               CHK_EXIT (pDevice->pFifoHashOut->Insert (pFifoBlock))
            }
         }
         else
         {
            LOG_INFO ("Dummy block")
            Finished = true;
         }
      } while (!Finished && !pDevice->AbortRequest);
      if (pDevice->AbortRequest)
      {
         break;
      }
      else if (VerifyLoop)
      {
         CHK_EXIT (HashMD5Digest    (&HashContextMD5   , &pDevice->MD5DigestVerify   ))
         CHK_EXIT (HashSHA256Digest (&HashContextSHA256, &pDevice->SHA256DigestVerify))
         break;
      }
      else if (pDevice->Acquisition.VerifySource)
      {
         CHK_EXIT (HashMD5Digest    (&HashContextMD5   , &pDevice->MD5Digest   ))
         CHK_EXIT (HashSHA256Digest (&HashContextSHA256, &pDevice->SHA256Digest))
         pBlocks = &BlocksVerified;
         VerifyLoop = true;
      }
      else
      {
         CHK_EXIT (HashMD5Digest    (&HashContextMD5   , &pDevice->MD5Digest   ))
         CHK_EXIT (HashSHA256Digest (&HashContextSHA256, &pDevice->SHA256Digest))
         break;
      }
   }

   if (pDevice->Acquisition.VerifySource)
        LOG_INFO ("Hash thread exits now (device %s, %Ld blocks processed, %Ld blocks verified)", QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice), BlocksCalculated, BlocksVerified)
   else LOG_INFO ("Hash thread exits now (device %s, %Ld blocks processed)"                     , QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice), BlocksCalculated)
}


void t_ThreadHash::SlotFinished (void)
{
   emit SignalEnded (pOwn->pDevice);
}

