/* GStreamer process transform
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:gstproctrans
 * @short_description: Base class for process transformers
 *
 * <refsect2>
 * <para>
 * This is a base class for elements that process and/or transform data
 * by relying upon another program/process to do the actual processing.
 * This process is presented the data coming into the element on its stdin,
 * and the element will send out what it receives from the process' stdout.
 *
 * It provides for most of the details such as setting up sink and src pads,
 * state changes, etc.  An inheriting element need only concern about
 * providing for the proper pad templates and (optionally but recommended)
 * implementing a custom set_caps.  This call-back is called during (initial
 * and only) caps negotiation, and only upon return the child process is actually
 * started.  This typically gives the inheritor a chance to provide the right
 * arguments to the sub-program for the circumstances and data at hand.
 * </para>
 * </refsect2>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef _MSC_VER
#include <io.h>
#endif
#include <errno.h>
#include <string.h>

#include "gstproctrans.h"

#define PARENT_READ(trans)  trans->readpipe[0]
#define CHILD_WRITE(trans)  trans->readpipe[1]
#define CHILD_READ(trans)   trans->writepipe[0]
#define PARENT_WRITE(trans) trans->writepipe[1]

GST_DEBUG_CATEGORY_STATIC (proc_trans_debug);
#define GST_CAT_DEFAULT proc_trans_debug

/* pad templates are determined by inheriting element */

/* properties */
enum
{
  PROP_0,
  PROP_CMD,
  PROP_BLOCKSIZE
};

#define DEFAULT_BLOCKSIZE  16384

static void gst_proc_trans_finalize (GObject * object);
static gboolean gst_proc_trans_setcaps (GstPad * pad, GstCaps * caps);
static gboolean gst_proc_trans_sink_event (GstPad * pad, GstEvent * event);
static void gst_proc_trans_loop (GstProcTrans * proctrans);
static GstFlowReturn gst_proc_trans_chain (GstPad * pad, GstBuffer * buffer);
static gboolean gst_proc_trans_sink_activate_push (GstPad * pad,
    gboolean active);
static gboolean gst_proc_trans_src_activate_push (GstPad * pad,
    gboolean active);
static GstStateChangeReturn gst_proc_trans_change_state (GstElement * element,
    GstStateChange transition);

/* properties */
static void gst_proc_trans_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_proc_trans_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstProcTrans, gst_proc_trans, GstElement, GST_TYPE_ELEMENT);

static void
gst_proc_trans_base_init (gpointer klass)
{

}

static void
gst_proc_trans_class_init (GstProcTransClass * klass)
{
  GstElementClass *element_class;
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);

  GST_DEBUG_CATEGORY_INIT (proc_trans_debug, "proctrans", 0,
      "Process Transform");

  gobject_class->finalize = GST_DEBUG_FUNCPTR (gst_proc_trans_finalize);

  gobject_class->set_property = gst_proc_trans_set_property;
  gobject_class->get_property = gst_proc_trans_get_property;

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_CMD,
      g_param_spec_string ("command", "command",
          "Shell Command into which to send output", NULL, G_PARAM_READWRITE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_BLOCKSIZE,
      g_param_spec_uint ("blocksize", "Block size",
          "Size in bytes to read from process per buffer",
          1, G_MAXUINT, DEFAULT_BLOCKSIZE, G_PARAM_READWRITE));

  element_class->change_state = GST_DEBUG_FUNCPTR (gst_proc_trans_change_state);
}

static void
gst_proc_trans_init (GstProcTrans * proctrans, GstProcTransClass * klass)
{
  GstElement *element = GST_ELEMENT (proctrans);
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  proctrans->sinkpad =
      gst_pad_new_from_template (gst_element_class_get_pad_template
      (element_class, GST_PROC_TRANS_SINK_NAME), "sink");
  gst_pad_set_setcaps_function (proctrans->sinkpad,
      GST_DEBUG_FUNCPTR (gst_proc_trans_setcaps));
  gst_pad_set_event_function (proctrans->sinkpad,
      GST_DEBUG_FUNCPTR (gst_proc_trans_sink_event));
  gst_pad_set_chain_function (proctrans->sinkpad,
      GST_DEBUG_FUNCPTR (gst_proc_trans_chain));
  gst_pad_set_activatepush_function (proctrans->sinkpad,
      GST_DEBUG_FUNCPTR (gst_proc_trans_sink_activate_push));
  gst_element_add_pad (element, proctrans->sinkpad);

  proctrans->srcpad =
      gst_pad_new_from_template (gst_element_class_get_pad_template
      (element_class, GST_PROC_TRANS_SRC_NAME), "src");
  gst_pad_use_fixed_caps (proctrans->srcpad);
  gst_pad_set_activatepush_function (proctrans->srcpad,
      GST_DEBUG_FUNCPTR (gst_proc_trans_src_activate_push));
  gst_element_add_pad (element, proctrans->srcpad);

  /* init properties. */
  proctrans->cmd = NULL;
  proctrans->args = g_array_sized_new (TRUE, TRUE, sizeof (gchar *), 10);
  proctrans->blocksize = DEFAULT_BLOCKSIZE;

  PARENT_READ (proctrans) = -1;
  CHILD_WRITE (proctrans) = -1;
  CHILD_READ (proctrans) = -1;
  CHILD_WRITE (proctrans) = -1;
}

static void
gst_proc_trans_reset_args (GstProcTrans * proctrans)
{
  guint i;

  if (proctrans->args) {
    for (i = 0; i < proctrans->args->len; ++i) {
      g_free (g_array_index (proctrans->args, gchar *, i));
      g_array_remove_index_fast (proctrans->args, i);
    }
  }
}

static void
gst_proc_trans_finalize (GObject * object)
{
  GstProcTrans *proctrans = GST_PROC_TRANS (object);

  g_free (proctrans->cmd);
  gst_proc_trans_reset_args (proctrans);
  g_array_free (proctrans->args, TRUE);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

/* note this could also be called with caps = NULL */
static gboolean
gst_proc_trans_setcaps (GstPad * pad, GstCaps * caps)
{
  GstProcTrans *proctrans = GST_PROC_TRANS (GST_PAD_PARENT (pad));
  GstProcTransClass *klass = GST_PROC_TRANS_GET_CLASS (proctrans);
  gboolean ret = TRUE;
  GstCaps *outcaps = NULL;

  if (proctrans->pid)
    goto refuse_renegotiation;

  if (!klass->set_caps) {
    GST_WARNING_OBJECT (proctrans, "no set_caps function set");
    return FALSE;
  }

  if (!(ret = klass->set_caps (proctrans, caps, &outcaps)))
    goto refuse_caps;

  /* if none explictly given, we don't know outgoing caps, like e.g. filesrc */
  if (outcaps) {
    ret = gst_pad_set_caps (proctrans->srcpad, outcaps);
    gst_caps_unref (outcaps);
  }
  if (!ret)
    goto refuse_caps;

  return TRUE;

refuse_caps:
  {
    GST_WARNING_OBJECT (proctrans, "refused caps %" GST_PTR_FORMAT, caps);
    return FALSE;
  }
refuse_renegotiation:
  {
    GST_WARNING_OBJECT (proctrans, "refused renegotiation "
        "(to %" GST_PTR_FORMAT ")", caps);

    return FALSE;
  }
}

static gboolean
gst_proc_trans_sink_event (GstPad * pad, GstEvent * event)
{
  GstProcTrans *proctrans;
  gboolean result = TRUE;

  proctrans = GST_PROC_TRANS (GST_PAD_PARENT (pad));

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_FLUSH_START:
      /* forward event */
      result = gst_pad_push_event (proctrans->srcpad, event);

      /* no special action */
      goto done;
      break;
    case GST_EVENT_FLUSH_STOP:
      /* forward event */
      result = gst_pad_push_event (proctrans->srcpad, event);
      if (!result)
        goto done;

      /* this clears the error state in case of a failure;
       * so that we can carry on again */
      proctrans->srcresult = GST_FLOW_OK;
      gst_pad_start_task (proctrans->srcpad,
          (GstTaskFunction) gst_proc_trans_loop, proctrans);
      goto done;
      break;
    case GST_EVENT_EOS:
      /* eat this event for now, task will send eos when finished */
      gst_event_unref (event);
      /* make process feel we are finished */
      close (PARENT_WRITE (proctrans));
      PARENT_WRITE (proctrans) = -1;
      GST_DEBUG_OBJECT (proctrans, "closed for eos");
      goto done;
      break;
    default:
      break;
  }

  result = gst_pad_push_event (proctrans->srcpad, event);

done:
  return result;
}

static void
gst_proc_trans_loop (GstProcTrans * proctrans)
{
  GstBuffer *buf;
  int bytes;

  if (G_UNLIKELY (proctrans->srcresult != GST_FLOW_OK))
    goto done;

  buf = gst_buffer_new_and_alloc (proctrans->blocksize);
  if ((bytes = read (PARENT_READ (proctrans), GST_BUFFER_DATA (buf),
              proctrans->blocksize)) < 0)
    goto read_error;
  if (!bytes)
    goto eos;

  GST_BUFFER_SIZE (buf) = bytes;
  gst_buffer_set_caps (buf, GST_PAD_CAPS (proctrans->srcpad));
  proctrans->srcresult = gst_pad_push (proctrans->srcpad, buf);

  return;

read_error:
  {
    GST_ELEMENT_ERROR (proctrans, RESOURCE, READ, (NULL), GST_ERROR_SYSTEM);
    proctrans->srcresult = GST_FLOW_ERROR;

    gst_buffer_unref (buf);
    /* fall-through */
  }
done:
  {
    /* no need to run wildly, stopped elsewhere, e.g. state change */
    GST_DEBUG_OBJECT (proctrans, "encountered %s, pausing task",
        gst_flow_get_name (proctrans->srcresult));
    gst_pad_pause_task (proctrans->srcpad);

    return;
  }
eos:
  {
    GST_DEBUG_OBJECT (proctrans, "encoding task reached eos");
    gst_pad_push_event (proctrans->srcpad, gst_event_new_eos ());
    proctrans->srcresult = GST_FLOW_UNEXPECTED;

    gst_buffer_unref (buf);

    goto done;
  }
}

static gboolean
gst_proc_trans_setup (GstProcTrans * proctrans)
{
  GError *error = NULL;
  gchar *name;

  if (!proctrans->cmd)
    goto setup;

  /* first some variable abuse to announce some stuff */
  name = g_strjoinv (" ", (gchar **) proctrans->args->data);
  GST_INFO_OBJECT (proctrans, "executing %s with args %s", proctrans->cmd,
      name);
  g_free (name);

  /* first argument should be the file name of the process being exeucuted */
  name = g_strdup (proctrans->cmd);
  g_array_prepend_val (proctrans->args, name);

  /* this will also close all of the child's descriptors
   * except stdin/stdout/stderr; which is really needed,
   * otherwise if the main process closes a descriptor,
   * the child still pretends it is open, which may stall pipes */
  g_spawn_async_with_pipes (NULL, (gchar **) proctrans->args->data, NULL,
      G_SPAWN_SEARCH_PATH, NULL, NULL,
      &proctrans->pid, &PARENT_WRITE (proctrans), &PARENT_READ (proctrans),
      NULL, &error);

  /* remove the process from the args */
  g_free (name);
  g_array_remove_index (proctrans->args, 0);

  if (error)
    goto fork;

  /* the whole setup succeeded, now we can start task */
  gst_pad_start_task (proctrans->srcpad,
      (GstTaskFunction) gst_proc_trans_loop, proctrans);

  return TRUE;

  /* special cases */
fork:
  {
    GST_ELEMENT_ERROR (proctrans, RESOURCE, FAILED,
        ("glib error %s", error->message ? error->message : ""),
        GST_ERROR_SYSTEM);
    return FALSE;
  }
setup:
  {
    GST_ERROR_OBJECT (proctrans, "no command has been specified");
    return FALSE;
  }
}

static GstFlowReturn
gst_proc_trans_chain (GstPad * pad, GstBuffer * buf)
{
  GstProcTrans *proctrans = GST_PROC_TRANS (GST_PAD_PARENT (pad));

  /* it's ok in general if no ingoing caps were provided,
   * but we should call setcaps to setup things */
  if (G_UNLIKELY (!proctrans->pid && !GST_PAD_CAPS (proctrans->sinkpad))) {
    if (!gst_proc_trans_setcaps (pad, NULL))
      goto not_negotiated;
  }

  /* process is started when receiving first buffer;
   * gives child elements a chance to set up command line using caps info */
  if (G_UNLIKELY (!proctrans->pid))
    if (!gst_proc_trans_setup (proctrans)) {
      gst_buffer_unref (buf);
      return GST_FLOW_ERROR;
    }

  if (G_UNLIKELY (proctrans->srcresult != GST_FLOW_OK))
    goto flush;

  /* FIXME: we could get SIGPIPE if the process died un us ...
   * (or other SIG for that matter) */
  if (write (PARENT_WRITE (proctrans), GST_BUFFER_DATA (buf),
          GST_BUFFER_SIZE (buf)) < 0) {
    if (errno != EPIPE)
      goto write_error;
    /* otherwise loop will sense eos */
  }

  gst_buffer_unref (buf);
  return GST_FLOW_OK;

  /* special cases */
not_negotiated:
  {
    GST_ELEMENT_ERROR (proctrans, CORE, NEGOTIATION, (NULL),
        ("format wasn't negotiated before chain function"));

    gst_buffer_unref (buf);
    return GST_FLOW_NOT_NEGOTIATED;
  }
flush:
  {
    GST_DEBUG_OBJECT (proctrans,
        "ignoring buffer because writer task encountered %s",
        gst_flow_get_name (proctrans->srcresult));

    gst_buffer_unref (buf);
    return proctrans->srcresult;
  }
write_error:
  {
    GST_ELEMENT_ERROR (proctrans, RESOURCE, WRITE, (NULL), GST_ERROR_SYSTEM);

    gst_buffer_unref (buf);
    return GST_FLOW_ERROR;
  }
}

static void
gst_proc_trans_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstProcTrans *proctrans;

  g_return_if_fail (GST_IS_PROC_TRANS (object));

  proctrans = GST_PROC_TRANS (object);

  switch (prop_id) {
    case PROP_CMD:
      g_free (proctrans->cmd);
      proctrans->cmd = g_value_dup_string (value);
      break;
    case PROP_BLOCKSIZE:
      proctrans->blocksize = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_proc_trans_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstProcTrans *proctrans;

  g_return_if_fail (GST_IS_PROC_TRANS (object));

  proctrans = GST_PROC_TRANS (object);

  switch (prop_id) {
    case PROP_CMD:
      g_value_take_string (value, g_strdup (proctrans->cmd));
      break;
    case PROP_BLOCKSIZE:
      g_value_set_uint (value, proctrans->blocksize);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static gboolean
gst_proc_trans_sink_activate_push (GstPad * pad, gboolean active)
{
  gboolean result = TRUE;
  GstProcTrans *proctrans;

  proctrans = GST_PROC_TRANS (GST_PAD_PARENT (pad));

  if (active) {
    /* chain function will start task once all is setup */
  } else {
    /* streaming is suspended/flushing at this stage */
    /* process should get the message we want to stop by closing part of pipe
     * (if not already closed by eos) */
    if (PARENT_WRITE (proctrans) >= 0) {
      result = (close (PARENT_WRITE (proctrans)) >= 0);
      PARENT_WRITE (proctrans) = -1;
    }
  }

  return result;
}

static gboolean
gst_proc_trans_src_activate_push (GstPad * pad, gboolean active)
{
  gboolean result = TRUE;
  GstProcTrans *proctrans;

  proctrans = GST_PROC_TRANS (GST_PAD_PARENT (pad));

  if (active) {
    /* chain function will start task once all is setup */
  } else {
    /* at this stage, sink pad should be shut down and process sensing eos */
    /* wait for task to pause */
    /* FIXME: could be more forceful */
    GST_PAD_STREAM_LOCK (proctrans->srcpad);
    GST_PAD_STREAM_UNLOCK (proctrans->srcpad);
    /* make sure task can't take off again in wrong state */
    proctrans->srcresult = GST_FLOW_WRONG_STATE;
    result = gst_pad_stop_task (proctrans->srcpad);
    /* can't do anything about any error in close ... */
    close (PARENT_READ (proctrans));
    PARENT_READ (proctrans) = -1;
  }

  return result;
}


static GstStateChangeReturn
gst_proc_trans_change_state (GstElement * element, GstStateChange transition)
{
  GstProcTrans *proctrans = GST_PROC_TRANS (element);
  GstStateChangeReturn ret;

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto done;

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      g_spawn_close_pid (proctrans->pid);
      proctrans->pid = (GPid) 0;
      proctrans->srcresult = GST_FLOW_OK;
      gst_proc_trans_reset_args (proctrans);
      break;
    default:
      break;
  }

done:
  return ret;
}
