/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * transcode filter:
 * Copyright (C) 2003 Guillaume Cottenceau <gc at mandrakesoft.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-whitebalance
 *
 * <refsect2>
 * <para>
 * This filter allows correcting movies with a broken white balance,
 * e.g. bluish movies, images from a dv camcorder with an unset
 * white balance or wrongly forced to indoor or outdoor.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * transcode whitebalance filter [Guillaume Cottenceau]
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <math.h>


#define GST_TYPE_WHITEBALANCE \
  (gst_whitebalance_get_type())
#define GST_WHITEBALANCE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_WHITEBALANCE,GstWhitebalance))
#define GST_WHITEBALANCE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_WHITEBALANCE,GstWhitebalanceClass))
#define GST_IS_WHITEBALANCE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_WHITEBALANCE))
#define GST_IS_WHITEBALANCE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_WHITEBALANCE))


typedef struct _GstWhitebalance GstWhitebalance;
typedef struct _GstWhitebalanceClass GstWhitebalanceClass;

struct _GstWhitebalance
{
  GstVideoFilter videofilter;

  gint width, height;

  gint level;

  guint8 red_filter[256], blue_filter[256];
};


struct _GstWhitebalanceClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (whitebalance_debug);
#define GST_CAT_DEFAULT whitebalance_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_LEVEL
      /* FILL ME */
};

#define DEFAULT_LEVEL       40

static GstStaticPadTemplate gst_whitebalance_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_BGRx)
    );

static GstStaticPadTemplate gst_whitebalance_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_BGRx)
    );


static GstFlowReturn gst_whitebalance_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_whitebalance_start (GstBaseTransform * btrans);
static gboolean gst_whitebalance_stop (GstBaseTransform * btrans);

static void gst_whitebalance_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_whitebalance_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstWhitebalance, gst_whitebalance, GstVideoFilter,
    GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE (GstWhitebalance, gst_whitebalance);

static void
gst_whitebalance_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Whitebalance",
      "Filter/Effect/Video", "Correct images with a broken white balance",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n"
      "Guillaume Cottenceau");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_whitebalance_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_whitebalance_src_template));
}

static void
gst_whitebalance_class_init (GstWhitebalanceClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (whitebalance_debug, "whitebalance", 0,
      "whitebalance");

  gobject_class->set_property = gst_whitebalance_set_property;
  gobject_class->get_property = gst_whitebalance_get_property;

  g_object_class_install_property (gobject_class, PROP_LEVEL,
      g_param_spec_int ("level", "Level",
          "Level of blue-to-yellow white balance shifting",
          -1000, 1000, DEFAULT_LEVEL,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_whitebalance_set_caps);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_whitebalance_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_whitebalance_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_whitebalance_stop);
}

static void
gst_whitebalance_init (GstWhitebalance * filter, GstWhitebalanceClass * g_class)
{
  filter->level = DEFAULT_LEVEL;
}

static void
gst_whitebalance_precalc (GstWhitebalance * filter)
{
  gdouble factor;
  gint i;

  factor = 1 + ((gdouble) ABS (filter->level)) / 100;
  if (filter->level < 0)
    factor = 1 / factor;

  for (i = 0; i <= 255; i++) {
    filter->red_filter[i] = pow (((gdouble) i) / 255, 1 / factor) * 255;
    filter->blue_filter[i] = pow (((gdouble) i) / 255, factor) * 255;
  }
}

static GstFlowReturn
gst_whitebalance_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstWhitebalance *filter;
  guint8 *src;
  gint i;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_WHITEBALANCE (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);

  for (i = 0; i < filter->width * filter->height; i++) {
    /* we modify red and blue curves to enhance/reduce mostly mediums */
    src[0] = filter->blue_filter[src[0]];
    src[2] = filter->red_filter[src[2]];
    src += 4;
  }

  return GST_FLOW_OK;
}


static gboolean
gst_whitebalance_start (GstBaseTransform * btrans)
{
  GstWhitebalance *filter = GST_WHITEBALANCE (btrans);

  gst_whitebalance_precalc (filter);

  return TRUE;
}

static gboolean
gst_whitebalance_stop (GstBaseTransform * btrans)
{

  return TRUE;
}

static void
gst_whitebalance_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstWhitebalance *src;

  g_return_if_fail (GST_IS_WHITEBALANCE (object));
  src = GST_WHITEBALANCE (object);

  switch (prop_id) {
    case PROP_LEVEL:
      src->level = g_value_get_int (value);
      gst_whitebalance_precalc (src);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_whitebalance_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstWhitebalance *src;

  g_return_if_fail (GST_IS_WHITEBALANCE (object));
  src = GST_WHITEBALANCE (object);

  switch (prop_id) {
    case PROP_LEVEL:
      g_value_set_int (value, src->level);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
