/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * transcode decimate filter
 * Copyright (C) Thanassis Tsiodras - August 2002
 * Based on the excellent work of Donald Graft in Decomb.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-decimate
 * @see_also: ivtc
 *
 * <refsect2>
 * <para>
 * Converts an NTSC 29.997 (30) fps stream to a 23.976 (24) fps stream by
 * dropping a frame in each block of 5 frames that most resembles its
 * succeeding frame.
 * </para>
 * <para>
 * If this is applied to a purely telecined stream that has
 * already been processed by <link linkend="GstIvtc">ivtc</link>,
 * the result is the original (pre-telecine) stream.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * transcode decimate filter [Thanassis Tsiodras],
 * based on avisynth Decomb work [Donald Graft]
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"


#define GST_TYPE_DECIMATE \
  (gst_decimate_get_type())
#define GST_DECIMATE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_DECIMATE,GstDecimate))
#define GST_DECIMATE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_DECIMATE,GstDecimateClass))
#define GST_IS_DECIMATE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_DECIMATE))
#define GST_IS_DECIMATE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_DECIMATE))


typedef struct _GstDecimate GstDecimate;
typedef struct _GstDecimateClass GstDecimateClass;

struct _GstDecimate
{
  GstBaseTransform videofilter;

  gint width, height;

  /* count frames */
  gint count;
  /* hold some past frames */
  GQueue *queue;
};


struct _GstDecimateClass
{
  GstBaseTransformClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (decimate_debug);
#define GST_CAT_DEFAULT decimate_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0
      /* FILL ME */
};

static GstStaticPadTemplate gst_decimate_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_decimate_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static gboolean gst_decimate_hook_caps (GstDecimate * filter, GstCaps * incaps,
    GstCaps * outcaps);
static GstCaps *gst_decimate_transform_caps (GstBaseTransform * btrans,
    GstPadDirection direction, GstCaps * caps);
static void gst_decimate_flush (GstDecimate * filter, gboolean send);
static gboolean gst_decimate_sink_event (GstBaseTransform * btrans,
    GstEvent * event);
static GstFlowReturn gst_decimate_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_decimate_start (GstBaseTransform * btrans);
static gboolean gst_decimate_stop (GstBaseTransform * btrans);

static void gst_decimate_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_decimate_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstDecimate, gst_decimate, GstBaseTransform,
    GST_TYPE_BASE_TRANSFORM);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL (GstDecimate, gst_decimate,
    gst_decimate_hook_caps);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_decimate);

static void
gst_decimate_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Decimate",
      "Filter/Effect/Video", "NTSC Decimation",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>, Thanassis Tsiodras");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_decimate_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_decimate_src_template));
}

static void
gst_decimate_class_init (GstDecimateClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (decimate_debug, "decimate", 0, "decimate");

  gobject_class->set_property = gst_decimate_set_property;
  gobject_class->get_property = gst_decimate_get_property;

  trans_class->transform_caps = GST_DEBUG_FUNCPTR (gst_decimate_transform_caps);
  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_decimate_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_decimate_get_unit_size);
  trans_class->event = GST_DEBUG_FUNCPTR (gst_decimate_sink_event);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_decimate_transform);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_decimate_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_decimate_stop);
}

static void
gst_decimate_init (GstDecimate * filter, GstDecimateClass * g_class)
{
}


static gboolean
gst_decimate_hook_caps (GstDecimate * filter, GstCaps * incaps,
    GstCaps * outcaps)
{
  gst_decimate_flush (filter, TRUE);

  return TRUE;
}

static GstCaps *
gst_decimate_transform_caps (GstBaseTransform * btrans,
    GstPadDirection direction, GstCaps * caps)
{
  GstCaps *ret;
  gint i;

  ret = gst_caps_copy (caps);

  for (i = 0; i < gst_caps_get_size (ret); ++i) {
    GstStructure *s = gst_caps_get_structure (ret, i);
    const GValue *fps;
    if ((fps = gst_structure_get_value (s, "framerate"))
        && GST_VALUE_HOLDS_FRACTION (fps)) {
      gint fps_n = gst_value_get_fraction_numerator (fps);
      gint fps_d = gst_value_get_fraction_denominator (fps);

      if (direction == GST_PAD_SINK) {
        fps_n *= 4;
        fps_d *= 5;
      } else {
        fps_n *= 5;
        fps_d *= 4;
      }
      gst_structure_set (s, "framerate", GST_TYPE_FRACTION, fps_n, fps_d, NULL);
    }
  }

  return ret;
}

// TODO consider this as a library function
/* empties queue and resets frame count */
static void
gst_decimate_flush (GstDecimate * filter, gboolean send)
{
  GstBuffer *buf;
  GstBaseTransform *btrans;

  btrans = GST_BASE_TRANSFORM (filter);

  filter->count = -1;
  while (!g_queue_is_empty (filter->queue)) {
    buf = g_queue_pop_head (filter->queue);
    if (send) {
      gst_buffer_set_caps (buf, GST_PAD_CAPS (btrans->srcpad));
      gst_pad_push (btrans->srcpad, buf);
    } else
      gst_buffer_unref (buf);
  }
}

static gboolean
gst_decimate_sink_event (GstBaseTransform * btrans, GstEvent * event)
{
  GstDecimate *filter;

  filter = GST_DECIMATE (btrans);

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_EOS:
      gst_decimate_flush (filter, TRUE);
      break;
    default:
      break;
  }

  return GST_BASE_TRANSFORM_CLASS (parent_class)->event (btrans, event);
}

/* is called with 6 buffers in queue,
 * and drops frame that looks most like its successor */
static void
gst_decimate_drop (GstDecimate * filter)
{
  GstBuffer *buf;
  guint8 *pcur, *pnext;
  gint width, height;
  gint i, j, min, index;

  g_return_if_fail (g_queue_get_length (filter->queue) == 6);

  height = filter->height;
  width = GST_VIDEO_I420_Y_ROWSTRIDE (filter->width);

  min = G_MAXINT;
  index = 0;
  for (i = 0; i < 5; ++i) {
    gint diff = 0;

    pcur = GST_BUFFER_DATA ((GstBuffer *) g_queue_peek_nth (filter->queue, i));
    pnext =
        GST_BUFFER_DATA ((GstBuffer *) g_queue_peek_nth (filter->queue, i + 1));

    for (j = 0; j < width * height; j += 16) {
      diff += ABS (pcur[i] - pnext[i]);
      if (diff < min) {
        min = diff;
        index = j;
      }
    }
  }

  buf = g_queue_pop_nth (filter->queue, index);
  gst_buffer_unref (buf);
}

static GstFlowReturn
gst_decimate_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstDecimate *filter = GST_DECIMATE (btrans);

  g_queue_push_tail (filter->queue, in);
  /* keep a ref as it will be dropped in any case, as input buffer */
  gst_buffer_ref (in);

  filter->count++;

  /* drop a frame once-in-a-right-while */

  /* just store during start-up */
  if (G_UNLIKELY (filter->count < 5)) {
    return GST_BASE_TRANSFORM_FLOW_DROPPED;
  }

  if (filter->count % 5 == 0) {
    gst_decimate_drop (filter);
    return GST_BASE_TRANSFORM_FLOW_DROPPED;
  }

  /* if we make it here, just return the oldest queued frame */
  in = g_queue_pop_head (filter->queue);
  oil_memcpy (GST_BUFFER_DATA (out), GST_BUFFER_DATA (in),
      GST_BUFFER_SIZE (in));
  gst_buffer_unref (in);

  return GST_FLOW_OK;
}

static gboolean
gst_decimate_start (GstBaseTransform * btrans)
{
  GstDecimate *filter = GST_DECIMATE (btrans);

  filter->queue = g_queue_new ();
  filter->count = -1;

  return TRUE;
}

static gboolean
gst_decimate_stop (GstBaseTransform * btrans)
{
  GstDecimate *filter = GST_DECIMATE (btrans);

  gst_decimate_flush (filter, FALSE);
  g_queue_free (filter->queue);

  return TRUE;
}

static void
gst_decimate_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstDecimate *src;

  g_return_if_fail (GST_IS_DECIMATE (object));
  src = GST_DECIMATE (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_decimate_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstDecimate *src;

  g_return_if_fail (GST_IS_DECIMATE (object));
  src = GST_DECIMATE (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
