//*************************************"***************************************
//                              CmdNgSpicePR.cpp                              *
//                             ------------------                             *
// Started     : 16/10/2007                                                   *
// Last Update : 07/10/2009                                                   *
// Copyright   : (C) 2007 by M.S.Waters                                       *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/commands/CmdNgSpicePR.hpp"

//*****************************************************************************
// Constructor.

CmdNgSpicePR::CmdNgSpicePR( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

CmdNgSpicePR::~CmdNgSpicePR( )
{
}

//*****************************************************************************
// Check that the object attributes are valid.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpicePR::bValidate( void )
{
  CmdBase::bValidate( );

  if( m_osaNodes.GetCount( )<=0 && m_osaCpnts.GetCount( )<=0 )
    SetErrMsg( wxT("No components or nodes have been selected.") );
  if( ! m_bParmtrs[ ePAR_VLT ] )
    SetErrMsg( wxT("No parameters selected.") );

  return( bIsValid( ) );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpicePR::bClear( void )
{
  int  i1;

  CmdBase::bClear( );

  m_eSimEng  = eSIMR_NGSPICE;
  m_eCmdType = eCMD_PR;

  m_eAnaType = eCMD_NONE;

  m_osaNodes.Empty( );
  m_osaCpnts.Empty( );

  for( i1=(int)ePAR_FST; i1<=(int)ePAR_LST; i1++ ) m_bParmtrs[ i1 ] = FALSE;
  for( i1=(int)eCPX_FST; i1<=(int)eCPX_LST; i1++ ) m_bCpxPrts[ i1 ] = FALSE;

  return( TRUE );
}

//*****************************************************************************
// Set the analysis type.
//
// Argument List :
//   eAnaType - An analysis command type
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpicePR::bSetAnaType( eCmdType eAnaType )
{
  // Does anything actually need to be done?
  if( eAnaType == m_eAnaType ) return( TRUE );

  // Check that the desired analysis type is supported
  switch( eAnaType )
  {
    case eCMD_OP : eAnaType = eCMD_DC;
    case eCMD_DC :
    case eCMD_AC :
    case eCMD_TR : break;
    default      : return( FALSE );
  }

  m_eAnaType = eAnaType;

  return( TRUE );
}

//*****************************************************************************
// Parse the command string.
//
// Eg.s : .PRINT AC VM(6,0) 0-VM(5) VM(1,7) VM(2) VP(6,0) 0-VP(5) VP(1,7) VP(2)
//        .PRINT AC VDB(4)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpicePR::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1, os2;
  int                i1, i2;

  // Clear the object attributes
  os1 = *this;
  bClear( );
  assign( os1 );

  // Tokenize the command string
  ostk1.SetString( *this );
  if( ostk1.CountTokens( ) < 3 ) return( bValidate( ) );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 3 ).Upper( );
  if( os1 != wxT(".PR") )        return( bValidate( ) );

  // Extract the analysis type
  os1 = ostk1.GetNextToken( );
  if(      os1 == wxT("DC")   ) m_eAnaType = eCMD_DC;
  else if( os1 == wxT("AC")   ) m_eAnaType = eCMD_AC;
  else if( os1 == wxT("TRAN") ) m_eAnaType = eCMD_TR;
  else                           return( bValidate( ) );

  // Standardize the argument list format; there are 3 possible formats
  // eg. V(n), V(n,m) or 0-V(n), reformat the latter to : V(0,n).
  os2.Empty( );
  while( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( );
    if( os1.Length( ) < 4 )      return( bValidate( ) );

    if( os1.StartsWith( wxT("0-") ) )
    {
      i1 = os1.Index( wxT('(') );
      i2 = os1.Index( wxT(')') );
      if( i1!=-1 && i2!=-1 && i2>(i1+1) )
        os1 = os1.Mid( 2, (size_t) i1-1 ) + wxT("0,") + os1.Mid( (size_t) i1+1 );
    }

    os2 << os1 << wxT(' ');
  }
  ostk1.SetString( os2 );

  // Extract the parameters to derive, any complex parts and the test component
  // and/or test node labels
  while( ostk1.HasMoreTokens( ) )
  {
    // Extract the next field
    os1 = ostk1.GetNextToken( );

    // Extract the parameter specifiers
    switch( os1.GetChar( 0 ) )
    {
      case wxT('V') : m_bParmtrs[ ePAR_VLT ] = TRUE; break;
      case wxT('I') : m_bParmtrs[ ePAR_CUR ] = TRUE; break;
      case wxT('P') : m_bParmtrs[ ePAR_PWR ] = TRUE; break;
      case wxT('R') : m_bParmtrs[ ePAR_RES ] = TRUE; break;
      default :                  return( bValidate( ) );
    }

    // Extract the complex parts if the analysis type is AC
    if( m_eAnaType == eCMD_AC )
    {
      switch( os1.GetChar( 1 ) )
      {
        case wxT('M') : m_bCpxPrts[ eCPX_MAG   ] = TRUE; break;
        case wxT('P') : m_bCpxPrts[ eCPX_PHASE ] = TRUE; break;
        case wxT('R') : m_bCpxPrts[ eCPX_REAL  ] = TRUE; break;
        case wxT('I') : m_bCpxPrts[ eCPX_IMAG  ] = TRUE; break;
        case wxT('D') :
          if( os1.Mid( 1, 2 ).Upper( ) == wxT("DB") )
          {
            m_bCpxPrts[ eCPX_MAG   ] = TRUE;
            m_bCpxPrts[ eCPX_MAGDB ] = TRUE;
            break;
          }
        default :                return( bValidate( ) );
      }
    }

    // Extract the node and component label/s, there are 2 possible formats
    // eg. V(n) or V(n,m,...), the first denotes a node and the second a
    // component.
    i1 = os1.Index( wxT('(') );
    i2 = os1.Index( wxT(')') );
    if( i1!=-1 && i2!=-1 && i2>(i1+1) )
    {
      os1 = os1.Mid( (size_t) i1+1, (size_t) i2-i1-1 );

      if( os1.Find( wxT(',') ) == wxNOT_FOUND )
      { // This is a node, add it to the list if it's not already there
        if( m_osaNodes.Index( os1 ) == wxNOT_FOUND ) m_osaNodes.Add( os1 );
      }
      else
      { // This is a component, add it to the list if it's not already there
        if( m_osaCpnts.Index( os1 ) == wxNOT_FOUND ) m_osaCpnts.Add( os1 );
      }
    }
    else                         return( bValidate( ) );
  }

  return( bValidate( ) );
}

//*****************************************************************************
// Format the command string.
//
// Eg.s : .PRINT AC VM(6,0) 0-VM(5) VM(1,7) VM(2) VP(6,0) 0-VP(5) VP(1,7) VP(2)
//        .PRINT AC VDB(4)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpicePR::bFormat( void )
{
  wxString           osCmd, osParm, osCpxPrt, os1;
  wxStringTokenizer  ostk1;
  int                i1, i2, i3;

  // Set the command name
  osCmd = wxT(".PRINT ");

  // Append the analysis type
  switch( m_eAnaType )
  {
    case eCMD_DC : osCmd << wxT("DC");   break;
    case eCMD_AC : osCmd << wxT("AC");   break;
    case eCMD_TR : osCmd << wxT("TRAN"); break;
    default      : break;
  }

  // Sequence through the various parameter types
  for( i1=ePAR_FST; i1<=ePAR_LST; i1++ )
  {
    if( ! m_bParmtrs[ i1 ] ) continue;

    switch( (eParType) i1 )
    {
      case ePAR_VLT : osParm = wxT('V'); break;
      case ePAR_CUR : osParm = wxT('I'); break;
      case ePAR_PWR : osParm = wxT('P'); break;
      case ePAR_RES : osParm = wxT('R'); break;
      default       : break;
    }

    if( m_eAnaType != eCMD_AC )
    {
      // Sequence through all the components and nodes
      for( i3=0; i3<(int)m_osaCpnts.GetCount( ); i3++ )
        osCmd << wxT(' ') << osParm
              << wxT('(') << m_osaCpnts.Item( i3 ) << wxT(')');
      for( i3=0; i3<(int)m_osaNodes.GetCount( ); i3++ )
        osCmd << wxT(' ') << osParm
              << wxT('(') << m_osaNodes.Item( i3 ) << wxT(')');
    }
    else
    { // Sequence through the various complex parts
      for( i2=eCPX_FST; i2<=eCPX_LST-1; i2++ )
      {
        if( ! m_bCpxPrts[ i2 ] ) continue;

        switch( (eCpxType) i2 )
        {
          case eCPX_MAG   :
            osCpxPrt = ( m_bCpxPrts[ eCPX_MAGDB ] ? wxT("DB") : wxT("M") );
            break;
          case eCPX_PHASE : osCpxPrt = wxT('P'); break;
          case eCPX_REAL  : osCpxPrt = wxT('R'); break;
          case eCPX_IMAG  : osCpxPrt = wxT('I'); break;
          default         : break;
        }

        // Sequence throught all the components and nodes
        for( i3=0; i3<(int)m_osaCpnts.GetCount( ); i3++ )
          osCmd << wxT(' ') << osParm << osCpxPrt
                << wxT('(') << m_osaCpnts.Item( i3 ) << wxT(')');
        for( i3=0; i3<(int)m_osaNodes.GetCount( ); i3++ )
          osCmd << wxT(' ') << osParm << osCpxPrt
                << wxT('(') << m_osaNodes.Item( i3 ) << wxT(')');
      }
    }
  }

  // Modify the argument list slightly; reformat V(0,n) to 0-V(n) otherwise
  // NG-Spice will barf.
  ostk1.SetString( osCmd );
  osCmd.Empty( );
  while( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( );
    if( ( i1=os1.Find( wxT("(0,") ) ) != wxNOT_FOUND )
    {
      os1.Remove( (size_t) i1+1, 2 );
      os1.Prepend( wxT("0-") );
    }
    if( ! osCmd.IsEmpty( ) ) osCmd << wxT(' ');
    osCmd << os1;
  }

  assign( osCmd );

  return( bValidate( ) );
}

//*****************************************************************************
// Get the total number of parameters to derive.
//
// Return Values :
//   The number of parameters to derive

int  CmdNgSpicePR::iGetParaCnt( void )
{
  int  i1, i2, i3;

  i1 = m_osaNodes.GetCount( ) + m_osaCpnts.GetCount( );

  for( i2=0, i3=(int)ePAR_FST; i3<=(int)ePAR_LST; i3++ )
    if( m_bParmtrs[ i3 ] )
      i2++;

  if( m_eAnaType == eCMD_AC )
  {
    for( i3=(int)eCPX_FST; i3<=(int)eCPX_LST-1; i3++ )
      if( m_bCpxPrts[ i3 ] )
        i2++;
  }

  return( i1 * ( i2 - 1 ) );
}

//*****************************************************************************
// Copy the contents of another CmdNgSpicePR object.
//
// Argument List :
//   roCmdPR - A reference to a CmdNgSpicePR object
//
// Return Values :
//   A reference to this object

CmdNgSpicePR & CmdNgSpicePR::operator = ( const CmdNgSpicePR & roCmdPR )
{
  int  i1;

  (CmdBase &) *this = (CmdBase &) roCmdPR;

  bSetAnaType( roCmdPR.m_eAnaType );

  m_osaNodes = roCmdPR.m_osaNodes;
  m_osaCpnts = roCmdPR.m_osaCpnts;

  for( i1=(int)ePAR_FST; i1<=(int)ePAR_LST; i1++ )
    m_bParmtrs[ i1 ] = roCmdPR.m_bParmtrs[ i1 ];
  for( i1=(int)eCPX_FST; i1<=(int)eCPX_LST; i1++ )
    m_bCpxPrts[ i1 ] = roCmdPR.m_bCpxPrts[ i1 ];

  bFormat( );

  return( *this );
}

//*****************************************************************************
// Copy the contents of a CmdGnuCapPR object.
//
// Argument List :
//   roCmdPR - A reference to a CmdGnuCapPR object
//
// Return Values :
//   A reference to this object

CmdNgSpicePR & CmdNgSpicePR::operator = ( const CmdGnuCapPR & roCmdPR )
{
  int  i1;

  (CmdBase &) *this = (CmdBase &) roCmdPR;

  bSetAnaType( roCmdPR.m_eAnaType );

  m_osaNodes = roCmdPR.m_osaNodes;
  m_osaCpnts = roCmdPR.m_osaCpnts;

  for( i1=(int)ePAR_FST; i1<=(int)ePAR_LST; i1++ )
    m_bParmtrs[ i1 ] = roCmdPR.m_bParmtrs[ i1 ];
  for( i1=(int)eCPX_FST; i1<=(int)eCPX_LST; i1++ )
    m_bCpxPrts[ i1 ] = roCmdPR.m_bCpxPrts[ i1 ];

  bFormat( );

  return( *this );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  CmdNgSpicePR::Print( const wxString & rosPrefix )
{
  int     i1;
  size_t  sz1;

  CmdBase::Print( rosPrefix + wxT("CmdBase::") );

  cout << rosPrefix.mb_str( ) << "m_eAnaType      : ";
  switch( m_eAnaType )
  {
    case eCMD_OP   : cout << "eCMD_OP";   break;  // Operating point analysis
    case eCMD_DC   : cout << "eCMD_DC";   break;  // DC analysis
    case eCMD_AC   : cout << "eCMD_AC";   break;  // AC analysis
    case eCMD_TR   : cout << "eCMD_TR";   break;  // Transient analysis
    case eCMD_FO   : cout << "eCMD_FO";   break;  // Fourier analysis
    case eCMD_DI   : cout << "eCMD_DI";   break;  // Distortion analysis
    case eCMD_NO   : cout << "eCMD_NO";   break;  // Noise analysis
    case eCMD_PZ   : cout << "eCMD_PZ";   break;  // Pole-zero analysis
    case eCMD_SE   : cout << "eCMD_SE";   break;  // Sensitivity analysis
    case eCMD_TF   : cout << "eCMD_TF";   break;  // Transfer function analysis
    case eCMD_NONE : cout << "eCMD_NONE"; break;  // Analysis type not set
    default        : cout << "Invalid";   break;
  }
  cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_osaCpnts[ n ] :";
  for( sz1=0; sz1<m_osaCpnts.GetCount( ); sz1++ )
    cout << ' ' << m_osaCpnts[ sz1 ].mb_str( );
  cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_osaNodes[ n ] :";
  for( sz1=0; sz1<m_osaNodes.GetCount( ); sz1++ )
    cout << ' ' << m_osaNodes[ sz1 ].mb_str( );
  cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_bParms  [ n ] :";
  for( i1=(int)ePAR_FST; i1<=(int)ePAR_LST; i1++ )
    cout << ( m_bParmtrs[ i1 ] ? " TRUE " : " FALSE" );
  cout << '\n';

  cout << rosPrefix.mb_str( ) << "m_bCpxPrts[ n ] :";
  for( i1=(int)eCPX_FST; i1<=(int)eCPX_LST; i1++ )
    cout << ( m_bCpxPrts[ i1 ] ? " TRUE " : " FALSE" );
  cout << '\n';
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_UTIL

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdNgSpicePR Test Utility"
       << "\n     Version 1.01 (04/06/2009)\n";

  // Create a NG-SPICE PRINT command object
  CmdNgSpicePR  oCmd_PR;

  // Use the following command example to check the formatter and the parser :
  osCmd = wxT(".PRINT AC VM(6,0) 0-VM(5) VM(1,7) VM(2) VP(6,0) 0-VP(5) VP(1,7) VP(2)");

  // Set things up for a formatter test
  oCmd_PR.bClear( );
  oCmd_PR.m_eAnaType = eCMD_AC;
  oCmd_PR.m_osaCpnts.Add( wxT("6,0") );
  oCmd_PR.m_osaCpnts.Add( wxT("0,5") );
  oCmd_PR.m_osaCpnts.Add( wxT("1,7") );
  oCmd_PR.m_osaNodes.Add( wxT("2") );
  oCmd_PR.m_bParms  [ ePAR_VLT   ] = TRUE;
  oCmd_PR.m_bCpxPrts[ eCPX_MAG   ] = TRUE;
  oCmd_PR.m_bCpxPrts[ eCPX_PHASE ] = TRUE;
  cout << "\nRun Formatter   : " << ( oCmd_PR.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format : " << ( oCmd_PR == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command : " << osCmd .mb_str( );
  cout << "\noCmdPR Contents : " << oCmd_PR.mb_str( ) << '\n';

  // Set things up for a parser test
  oCmd_PR.bClear( );
  oCmd_PR.bSetString( osCmd );
  cout << "\nRun Parser      : " << ( oCmd_PR.bParse( ) ? "OK" : "FAULT" );
  oCmd_PR.bFormat( );
  cout << "\nTest Cmd Format : " << ( oCmd_PR == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command : " << osCmd .mb_str( );
  cout << "\noCmdPR Contents : " << oCmd_PR.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_UTIL

//*****************************************************************************
