//*****************************************************************************
//                               PnlNgSpiceDC.cpp                             *
//                              ------------------                            *
//  Started     : 08/05/2004                                                  *
//  Last Update : 05/10/2007                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/panels/PnlNgSpiceDC.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlNgSpiceDC, PnlAnaBase )

  EVT_CHOICE  ( ID_CHO_SIGSRCCPNT, PnlNgSpiceDC::OnSigSrcCpnt )
  EVT_RADIOBOX( ID_RBX_SWPTYPE,    PnlNgSpiceDC::OnSwpType    )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlNgSpiceDC::PnlNgSpiceDC( wxWindow * poWin ) : PnlAnaBase( poWin )
{
  bSetSimrType( eSIMR_NGSPICE );
  bSetAnaType ( eANA_DC );

  Create( );        // Create the analysis panel
  InitSwpUnits( );  // Initialize the sweep units
  bClear( );        // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlNgSpiceDC::~PnlNgSpiceDC( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlNgSpiceDC::Create( void )
{
  // Disable the checkboxes for the parameters NG-Spice cannot calculate
  m_oCbxCurrent.Disable( );
  m_oCbxPower  .Disable( );
  m_oCbxResist .Disable( );

  // Set the sweep parameter labels
  m_oSbxSwpPars.SetLabel( wxT(" DC Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Value") );
  m_oPnlStop   .bSetName( wxT("Stop Value") );
  m_oPnlStep   .bSetName( wxT("Step Increment") );

  // Create and add the sweep type radio buttons
  wxString  osSwpType[ 2 ] = { wxT("Source"), wxT("Temp.C ") };
  m_oRbxSwpType.Create( this, ID_RBX_SWPTYPE, wxT(" Sweep Type "),
                        wxPoint( 13, 123 ), wxDefaultSize, 2, osSwpType, 2 );

  // Create and add input source controls
  m_oLblSigSrcCpnt.Create( this, ID_UNUSED, wxT("Signal Source"),
                           wxPoint(  18, 176 ) );
  m_oChoSigSrcCpnt.Create( this, ID_CHO_SIGSRCCPNT,
                           wxPoint( 144, 171 ), wxSize( 110, 29 ) );

  // Note: m_oLblSrcCpnt and m_oChoSrcCpnt must be created before the next line
  bSetSwpType( eSWP_SIGSRC );  // Set the sweep type to a signal source

  PnlAnaBase::CreateTemp( );   // Create the analysis temperature controls
}

//*****************************************************************************
// Initialize the sweep type.

void  PnlNgSpiceDC::InitSwpType( void )
{
  int  ix, iy;

  m_oSbxSwpPars.GetSize( &ix, &iy );

  switch( (eSweepType) m_oRbxSwpType.GetSelection( ) )
  {
    case eSWP_SIGSRC :  // DC Voltage sweep
      m_oSbxSwpPars.SetSize( ix, 201 );
      m_oLblSigSrcCpnt.Show( TRUE );
      m_oChoSigSrcCpnt.Show( TRUE );
      m_oSbxTemp      .Show( TRUE );
      m_oPnlTemp      .Show( TRUE );
      InitSwpUnits( );
      break;

    case eSWP_TEMP :    // Temperature sweep
      m_oSbxSwpPars.SetSize( ix, 167 );
      m_oLblSigSrcCpnt.Show( FALSE );
      m_oChoSigSrcCpnt.Show( FALSE );
      m_oSbxTemp      .Show( FALSE );
      m_oPnlTemp      .Show( FALSE );
      InitSwpUnits( );
      break;

    default:
      break;
  }
}

//*****************************************************************************
// Initialize the sweep parameter units.

void  PnlNgSpiceDC::InitSwpUnits( void )
{
  wxString              os1;
  ChoUnits::eUnitsType  eUnits;

  if( m_oRbxSwpType.GetSelection( ) == 0 )
  {
    os1 = m_oChoSigSrcCpnt.GetStringSelection( );
    if( os1.IsEmpty( ) ) os1 = wxT('?');
    switch( wxToupper( os1.GetChar( 0 ) ) )
    {
      case wxT('R') :  // Units of resistance
        os1 = wxT("Resistance");
        eUnits = ChoUnits::eUNITS_RES;
        break;

      case wxT('I') :  // Units of current
        os1 = wxT("Current");
        eUnits = ChoUnits::eUNITS_CURR;
        break;

      case wxT('V') :  // Units of voltage
        os1 = wxT("Voltage");
        eUnits = ChoUnits::eUNITS_VOLT;
        break;

      default :        // No units
        os1 = wxT("Value");
        eUnits = ChoUnits::eUNITS_NONE;
    }
  }
  else
  {
    os1 = wxT("Temperature");
    eUnits = ChoUnits::eUNITS_TEMP;
  }

  m_oPnlStart.bSetName( wxString( wxT("Start ") ) + os1 );
  m_oPnlStop .bSetName( wxString( wxT("Stop ")  ) + os1 );
  m_oPnlStart.bSetUnitsType( eUnits );
  m_oPnlStop .bSetUnitsType( eUnits );
  m_oPnlStep .bSetUnitsType( eUnits );
}

//*****************************************************************************
// Set the state of the sweep type radio box.
//
// Argument List:
//   eSwpType - The enumerated sweep type specifier
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  PnlNgSpiceDC::bSetSwpType( eSweepType eSwpType )
{
  // Argument validity checks
  if( eSwpType<eSWP_FST || eSwpType>eSWP_LST )          return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxSwpType.GetCount( ) < (uint) eSwpType + 1 ) return( FALSE );
#else
  if( m_oRbxSwpType.GetCount( ) < (int)  eSwpType + 1 ) return( FALSE );
#endif

  // Set appropraite radio button
  m_oRbxSwpType.SetSelection( (int) eSwpType );

  // Initialize the sweep type
  InitSwpType( );

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceDC::bClear( void )
{
  bool  bRtnValue=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )      bRtnValue = FALSE;

  // Set the sweep parameters to their defaults
  m_oPnlStart.bSetValue( (float)   0.0 );
  m_oPnlStop .bSetValue( (float) 100.0 );
  m_oPnlStep .bSetValue( (float)  10.0 );

  // Set the default sweep type
  if( ! bSetSwpType( eSWP_SIGSRC ) ) bRtnValue = FALSE;
  InitSwpType( );

  // Set sweep default values
  m_oChoSigSrcCpnt.Clear( );
  m_oChoSigSrcCpnt.Append( wxT("None") );
  m_oChoSigSrcCpnt.SetSelection( 0 );

  return( bRtnValue );
}

//*****************************************************************************
// Load information from a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceDC::bLoad( Simulation & roSim )
{
  bool  bRtnValue=TRUE;

  // Load the components into the signal source choice box
  LoadSigSrcCpnts( roSim, wxT("VIR") );

  // Don't go any further if the analysis type doesn't match
  if( roSim.eGetAnaType( ) != eGetAnaType( ) )              return( TRUE );

  // Set the source component
  if( ! PnlAnaBase::bSetSigSrc( roSim ) )                   bRtnValue = FALSE;

  // Set the sweep type (ie. source or temperature)
  if( ! bSetSwpType( (eSweepType) roSim.iGetSwpScale( ) ) ) bRtnValue = FALSE;

  // Perform any base class load tasks
  if( ! PnlAnaBase::bLoad( roSim ) )                        bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Save information to a Simulation object.
// (Prior to it being passed to a simulator object.)
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceDC::bSave( Simulation & roSim )
{
  wxString  os1;

  // Execute base class save tasks
  PnlAnaBase::bSave( roSim );

  // Set the sweep type
  roSim.bSetSwpScale( m_oRbxSwpType.GetSelection( ) );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Sweep type radio box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceDC::OnSwpType( wxCommandEvent & roEvtCmd )
{
  InitSwpType( );
}

//*****************************************************************************
// Source component choice box event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceDC::OnSigSrcCpnt( wxCommandEvent & roEvtCmd )
{
  InitSwpUnits( );
}

//*****************************************************************************
