//*****************************************************************************
//                               NbkNgSpice.cpp                               *
//                              ----------------                              *
//  Started     : 08/05/2004                                                  *
//  Last Update : 09/10/2007                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/NbkNgSpice.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( NbkNgSpice, wxNotebook )

  EVT_BUTTON( PnlAnaBase::ID_BTN_OPTIONS, NbkNgSpice::OnBtnOptions )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Arguments:
//   poParent - The parent window
//   oWinID   - The display object ID
//   roPosn   - The notebook position
//   roSize   - The notebook size

NbkNgSpice::NbkNgSpice( wxWindow * poParent, wxWindowID oWinID )
                   : NbkSimrBase( poParent, oWinID ), m_oDlgOptions( poParent )
{
  // Set the simulator engine type specifier
  bSetSimrType( eSIMR_NGSPICE );

  // Create the various display objects
  m_poPnlNgSpiceDC = new PnlNgSpiceDC( this );
  m_poPnlNgSpiceAC = new PnlNgSpiceAC( this );
  m_poPnlNgSpiceTR = new PnlNgSpiceTR( this );

  // Add the display objects to the note book
  AddPage( m_poPnlNgSpiceDC, wxT( "     DC     " ) );
  AddPage( m_poPnlNgSpiceAC, wxT( "     AC     " ) );
  AddPage( m_poPnlNgSpiceTR, wxT( " Transient  " ) );

  // Specify the default page to be displayed
  SetSelection( 0 );
}

//*****************************************************************************
// Default constructor.
// (Used for two stage creation ie. must call Create( ) method.

NbkNgSpice::NbkNgSpice( void ) : NbkSimrBase( ), m_oDlgOptions( this )
{
  // Set the simulator engine type specifier
  bSetSimrType( eSIMR_NGSPICE );
}

//*****************************************************************************
// Destructor.

NbkNgSpice::~NbkNgSpice( )
{
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bClear( void )
{
  bool  bRtnValue=TRUE;

  if( ! m_poPnlNgSpiceDC->bClear( ) ) bRtnValue = FALSE;
  if( ! m_poPnlNgSpiceAC->bClear( ) ) bRtnValue = FALSE;
  if( ! m_poPnlNgSpiceTR->bClear( ) ) bRtnValue = FALSE;

  if( ! m_oDlgOptions    .bClear( ) ) bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List:
//   roSimultn - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bLoad( Simulation & roSim )
{
  bool  bRtnValue=TRUE;

  if( ! m_poPnlNgSpiceDC->bLoad( roSim ) )              bRtnValue = FALSE;
  if( ! m_poPnlNgSpiceAC->bLoad( roSim ) )              bRtnValue = FALSE;
  if( ! m_poPnlNgSpiceTR->bLoad( roSim ) )              bRtnValue = FALSE;

  if( m_oDlgOptions.bSetCmd( roSim.rosGetOPTIONS( ) ) ) bRtnValue = FALSE;

  if( ! bSetPage( roSim.eGetAnaType( ) ) )              bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List:
//   roSimultn - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bSave( Simulation & roSim )
{
  switch( GetSelection( ) )
  {
    case 0 : // DC analysis
      if( ! m_poPnlNgSpiceDC->bSave( roSim ) ) return( FALSE );
      break;
    case 1 : // AC analysis
      if( ! m_poPnlNgSpiceAC->bSave( roSim ) ) return( FALSE );
      break;
    case 2 : // TR analysis
      if( ! m_poPnlNgSpiceTR->bSave( roSim ) ) return( FALSE );
      break;
    default:                                   return( FALSE );
  }

  m_oDlgOptions.m_fTEMP = roSim.fGetTempC( );
  roSim.bSetOPTIONS( m_oDlgOptions.rosGetCmd( ) );

  return( TRUE );
}

//*****************************************************************************
// Set the page to be displayed.
//
// Argument List:
//   eType - The enumerated analysis type specifier
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bSetPage( eAnaType eType )
{
  int  iPage;

  switch( eType )
  {
    case eANA_OP :
    case eANA_DC : iPage = 0; break;
    case eANA_AC : iPage = 1; break;
    case eANA_TR : iPage = 2; break;
    default :      return( FALSE );
  }

  SetSelection( iPage ); // Specify the page to be displayed

  return( TRUE );
}

//*****************************************************************************
// Set the page to be displayed.
//
// Argument List:
//   psType - The two letter analysis type specifier (case ignored)
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkNgSpice::bSetPage( const wxChar * psType )
{
  eAnaType  eType=eANA_NONE;
  wxString  osType;

  osType = wxString( psType ).Upper( );

  if( osType.Length( ) == 2 )
  {
         if( osType == wxT("OP") ) eType = eANA_DC;
    else if( osType == wxT("DC") ) eType = eANA_DC;
    else if( osType == wxT("AC") ) eType = eANA_AC;
    else if( osType == wxT("TR") ) eType = eANA_TR;
  }

  return( bSetPage( eType ) );
}

//*****************************************************************************
// Get the two letter page specifier.
//
// Return Values:
//   Success - The two letter analysis type specifier (lower case)
//   Failure - An empty string

const wxString & NbkNgSpice::rosGetPage( void )
{
  static  wxString  osPage;

  switch( eGetPage( ) )
  {
    case eANA_DC : osPage = wxT("dc"); break;
    case eANA_AC : osPage = wxT("ac"); break;
    case eANA_TR : osPage = wxT("tr"); break;
    default:       osPage.Empty( );
  }

  return( osPage );
}

//*****************************************************************************
// Get the enumerated page specifier.
//
// Return Values:
//   Success - The enumerated analysis specifier
//   Failure - Simulation::eANA_NONE

eAnaType  NbkNgSpice::eGetPage( void )
{
  eAnaType  ePage;

  switch( GetSelection( ) )
  {
    case 0 :  ePage = eANA_DC; break;
    case 1 :  ePage = eANA_AC; break;
    case 2 :  ePage = eANA_TR; break;
    default : ePage = eANA_NONE;
  }

  return( ePage );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// .OPTIONS command setup button control event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  NbkNgSpice::OnBtnOptions( wxCommandEvent & roEvtCmd )
{
  PnlAnaBase * poPnlAna;
  double       df1;
  int          i1;

  // Set the temperature in the OPTIONS dialog
  poPnlAna = (PnlAnaBase *) GetPage( GetSelection( ) );
  if( poPnlAna->bIsCreatedTemp( ) )
  {
    df1 = poPnlAna->m_oPnlTemp.dfGetValue( );
    m_oDlgOptions.bSetValue( DlgNgSpiceOPT::ID_PNL_TEMP, (float) df1 );
  }

  // Display the OPTIONS dialog
  m_oDlgOptions.CenterOnParent( );
  i1 = m_oDlgOptions.ShowModal( );

  // Set the temperature in the analysis panel
  if( i1==wxID_OK && poPnlAna->bIsCreatedTemp( ) )
  {
    df1 = m_oDlgOptions.dfGetValue( DlgNgSpiceOPT::ID_PNL_TEMP );
    poPnlAna->m_oPnlTemp.bSetValue( df1 );
  }
}

//*****************************************************************************
