//*****************************************************************************
//                                 DlgGenCfg.cpp                              *
//                                ---------------                             *
//  Started     : 27/02/2004                                                  *
//  Last Update : 14/09/2007                                                  *
//  Copyright   : (C) 2004 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/dialogs/DlgSigGenCfg.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( DlgSigGenCfg, wxDialog )

  EVT_BUTTON( ID_BTN_OK,     DlgSigGenCfg::OnBtnOk     )
  EVT_BUTTON( ID_BTN_CLEAR,  DlgSigGenCfg::OnBtnClear  )
  EVT_BUTTON( ID_BTN_CANCEL, DlgSigGenCfg::OnBtnCancel )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Argument List:
//   poWin - A pointer to the dialog parent window

DlgSigGenCfg::DlgSigGenCfg( wxWindow * poWin ) : wxDialog( poWin, -1, wxT(""),
    wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE, wxDialogNameStr )
{
  SetTitle( wxT("Generator Setup") );
  Initialize( );
  bClear( );
  bSetDefaults( );
}

//*****************************************************************************
// Destructor.

DlgSigGenCfg::~DlgSigGenCfg( )
{
}

//*****************************************************************************
// Initialize object attributes.

void  DlgSigGenCfg::Initialize( void )
{
  // Call all the initialization functions
  Create( );
  ToolTips( );

  // Layout the of the display objects
  DoLayout( );

  // Initialize temporary storage with the spin control values
  GetValues( );
}

//*****************************************************************************
// Create the display objects.

void  DlgSigGenCfg::Create( void )
{
  wxPanel * poPnl;

  // Create the total panel and controls
  poPnl = new wxPanel( this );

  m_oPnlTotAmp   .bCreate( poPnl, ID_PNL_TOT_AMP,    95 );
  m_oPnlTotOffset.bCreate( poPnl, ID_PNL_TOT_OFFSET, 95 );

  m_oPnlTotAmp   .bSetName( wxT("Amplitude") );
  m_oPnlTotOffset.bSetName( wxT("DC Offset") );

  m_oPnlTotAmp   .bSetUnitsType( ChoUnits::eUNITS_SCLR );
  m_oPnlTotOffset.bSetUnitsType( ChoUnits::eUNITS_VOLT );

  // Create the sine wave panel and controls
  poPnl = new wxPanel( this );

  m_oPnlSinFreq  .bCreate( poPnl, ID_PNL_SIN_FREQ,  95 );
  m_oPnlSinPhase .bCreate( poPnl, ID_PNL_SIN_PHASE, 95 );

  m_oPnlSinFreq  .bSetName( wxT("Frequency")     );
  m_oPnlSinPhase .bSetName( wxT("Initial Phase") );

  m_oPnlSinFreq  .bSetUnitsType( ChoUnits::eUNITS_FREQ  );
  m_oPnlSinPhase .bSetUnitsType( ChoUnits::eUNITS_PHASE );

  // Create the pulse panel and controls
  poPnl = new wxPanel( this );

  m_oPnlPulInitV .bCreate( poPnl, ID_PNL_PUL_INITV,  95 );
  m_oPnlPulMinV  .bCreate( poPnl, ID_PNL_PUL_MINV,   95 );
  m_oPnlPulMaxV  .bCreate( poPnl, ID_PNL_PUL_MAXV,   95 );
  m_oPnlPulDelay .bCreate( poPnl, ID_PNL_PUL_DELAY,  95 );
  m_oPnlPulRise  .bCreate( poPnl, ID_PNL_PUL_RISE,   95 );
  m_oPnlPulWidth .bCreate( poPnl, ID_PNL_PUL_WIDTH,  95 );
  m_oPnlPulFall  .bCreate( poPnl, ID_PNL_PUL_FALL,   95 );
  m_oPnlPulPeriod.bCreate( poPnl, ID_PNL_PUL_PERIOD, 95 );

  m_oPnlPulInitV .bSetName( wxT("Initial Value") );
  m_oPnlPulMinV  .bSetName( wxT("Min Level")     );
  m_oPnlPulMaxV  .bSetName( wxT("Max Level")     );
  m_oPnlPulDelay .bSetName( wxT("Initial Delay") );
  m_oPnlPulRise  .bSetName( wxT("Rise Time")     );
  m_oPnlPulWidth .bSetName( wxT("Pulse Width")   );
  m_oPnlPulFall  .bSetName( wxT("Fall Time")     );
  m_oPnlPulPeriod.bSetName( wxT("Period")        );

  m_oPnlPulInitV .bSetUnitsType( ChoUnits::eUNITS_VOLT );
  m_oPnlPulMinV  .bSetUnitsType( ChoUnits::eUNITS_VOLT );
  m_oPnlPulMaxV  .bSetUnitsType( ChoUnits::eUNITS_VOLT );
  m_oPnlPulDelay .bSetUnitsType( ChoUnits::eUNITS_TIME );
  m_oPnlPulRise  .bSetUnitsType( ChoUnits::eUNITS_TIME );
  m_oPnlPulWidth .bSetUnitsType( ChoUnits::eUNITS_TIME );
  m_oPnlPulFall  .bSetUnitsType( ChoUnits::eUNITS_TIME );
  m_oPnlPulPeriod.bSetUnitsType( ChoUnits::eUNITS_TIME );

  m_oPnlSinFreq.bSetRange( 0.0, 1000.0 );
  m_oPnlSinFreq.bSetUnits( wxT("Hz") );

  // Create the buttons panel and the button controls
  poPnl = new wxPanel( this );

  m_oBtnOk    .Create( poPnl, ID_BTN_OK,     wxT("OK")     );
  m_oBtnClear .Create( poPnl, ID_BTN_CLEAR,  wxT("Clear")  );
  m_oBtnCancel.Create( poPnl, ID_BTN_CANCEL, wxT("Cancel") );
}

//*****************************************************************************
// Initialize the tool tips.

void  DlgSigGenCfg::ToolTips( void )
{
  m_oPnlTotAmp   .SetToolTip( wxT("The overall amplitude of the sine wave and the pulse train") );
  m_oPnlTotOffset.SetToolTip( wxT("The DC offset applied to the entire signal") );

  m_oPnlSinFreq  .SetToolTip( wxT("The frequency of the sine wave generator for a Transient analysis") );
  m_oPnlSinPhase .SetToolTip( wxT("The phase of the sine wave at startup") );

  m_oPnlPulInitV .SetToolTip( wxT("The initial value of the source") );
  m_oPnlPulMinV  .SetToolTip( wxT("The 'OFF' value for the pulse train") );
  m_oPnlPulMaxV  .SetToolTip( wxT("The 'ON' value for the pulse train") );
  m_oPnlPulDelay .SetToolTip( wxT("The time until the first pulse") );
  m_oPnlPulRise  .SetToolTip( wxT("The time taken to go from 0 to the pulsed value") );
  m_oPnlPulWidth .SetToolTip( wxT("The width of each pulse") );
  m_oPnlPulFall  .SetToolTip( wxT("The time taken to go from the pulsed value to 0") );
  m_oPnlPulPeriod.SetToolTip( wxT("The period of the pulse train") );
}

//*****************************************************************************
// Layout the dialog display objects.

void  DlgSigGenCfg::DoLayout( void )
{
  wxPanel     * poPnl;
  wxBoxSizer  * poSzr;
  wxSizerFlags  oFlags;

  // Create and set the underlying dialog's sizer
  poSzr = new wxBoxSizer( wxVERTICAL );
  SetSizer( poSzr );

  // Create and set the sizer for the total panel
  poPnl = (wxPanel *) m_oPnlTotAmp.GetParent( );
  poSzr = new wxStaticBoxSizer( wxVERTICAL, poPnl, wxT(" Overall ") );
  poPnl->SetSizer( poSzr );

  // Layout the total panel
  oFlags.Align( wxALIGN_CENTER );
  oFlags.Border( wxTOP | wxLEFT | wxRIGHT, 10 );
  poSzr->Add( &m_oPnlTotAmp,    oFlags );
  oFlags.Border( wxBOTTOM, 5 );
  poSzr->Add( &m_oPnlTotOffset, oFlags );
  poSzr->SetSizeHints( poPnl );

  oFlags.Border( wxTOP | wxLEFT | wxRIGHT, 10 );
  GetSizer( )->Add( poPnl, oFlags );

  // Create and set the sizer for the sinusoid parameter panel
  poPnl = (wxPanel *) m_oPnlSinFreq.GetParent( );
  poSzr = new wxStaticBoxSizer( wxVERTICAL, poPnl, wxT(" Sinusoid ") );
  poPnl->SetSizer( poSzr );

  // Layout the sine wave parameter panel
  oFlags.Border( wxTOP | wxLEFT | wxRIGHT, 10 );
  poSzr->Add( &m_oPnlSinFreq,  oFlags );
  oFlags.Border( wxBOTTOM, 5 );
  poSzr->Add( &m_oPnlSinPhase, oFlags );
  poSzr->SetSizeHints( poPnl );

  oFlags.Border( wxTOP | wxLEFT | wxRIGHT, 10 );
  GetSizer( )->Add( poPnl, oFlags );

  // Create and set the sizer for the pulse parameter panel
  poPnl = (wxPanel *) m_oPnlPulInitV.GetParent( );
  poSzr = new wxStaticBoxSizer( wxVERTICAL, poPnl, wxT(" Pulse ") );
  poPnl->SetSizer( poSzr );

  // Layout the pulse parameter panel
  oFlags.Border( wxTOP, 10 );
  poSzr->Add( &m_oPnlPulInitV,  oFlags );
  oFlags.Border( wxLEFT | wxRIGHT, 10 );
  poSzr->Add( &m_oPnlPulMinV,   oFlags );
  poSzr->Add( &m_oPnlPulMaxV,   oFlags );
  poSzr->Add( &m_oPnlPulDelay,  oFlags );
  poSzr->Add( &m_oPnlPulRise,   oFlags );
  poSzr->Add( &m_oPnlPulWidth,  oFlags );
  poSzr->Add( &m_oPnlPulFall,   oFlags );
  oFlags.Border( wxBOTTOM, 5 );
  poSzr->Add( &m_oPnlPulPeriod, oFlags );
  poSzr->SetSizeHints( poPnl );

  oFlags.Border( wxALL, 10 );
  GetSizer( )->Add( poPnl, oFlags );

  // Create and set the sizer for the buttons panel
  poPnl = (wxPanel *) m_oBtnOk.GetParent( );
  poSzr = new wxBoxSizer( wxHORIZONTAL );
  poPnl->SetSizer( poSzr );

  // Layout the button panel
  oFlags.Border( wxTOP | wxBOTTOM, 10 );
  oFlags.Align( wxALIGN_RIGHT );
  poSzr->Add( &m_oBtnOk,     oFlags );
  poSzr->AddSpacer( 10 );
  oFlags.Align( wxALIGN_CENTER );
  poSzr->Add( &m_oBtnClear,  oFlags );
  poSzr->AddSpacer( 10 );
  oFlags.Align( wxALIGN_LEFT );
  poSzr->Add( &m_oBtnCancel, oFlags );
  poSzr->SetSizeHints( poPnl );

  oFlags.Align( wxALIGN_CENTER );
  oFlags.Border( wxBOTTOM, 10 );
  GetSizer( )->Add( poPnl, oFlags );

  // Set dialogues minimum size and initial size as calculated by the sizer
  GetSizer( )->SetSizeHints( this );
}

//*****************************************************************************
// Get a reference to a value panel associated with a control ID number.
//
// Argument List:
//   iPnlID - The control ID number
//
// Return Values:
//   A reference to a value panel

PnlValue & DlgSigGenCfg::roGetPanel( int iPnlID )
{
  // No argument tests here so use this function responsibly or suffer the pain
  switch( iPnlID )
  {
    case ID_PNL_TOT_AMP    : return( m_oPnlTotAmp    );
    case ID_PNL_TOT_OFFSET : return( m_oPnlTotOffset );

    case ID_PNL_SIN_FREQ   : return( m_oPnlSinFreq   );
    case ID_PNL_SIN_PHASE  : return( m_oPnlSinPhase  );

    case ID_PNL_PUL_INITV  : return( m_oPnlPulInitV  );
    case ID_PNL_PUL_MINV   : return( m_oPnlPulMinV   );
    case ID_PNL_PUL_MAXV   : return( m_oPnlPulMaxV   );
    case ID_PNL_PUL_DELAY  : return( m_oPnlPulDelay  );
    case ID_PNL_PUL_RISE   : return( m_oPnlPulRise   );
    case ID_PNL_PUL_WIDTH  : return( m_oPnlPulWidth  );
    case ID_PNL_PUL_FALL   : return( m_oPnlPulFall   );
    case ID_PNL_PUL_PERIOD : return( m_oPnlPulPeriod );

    default                : return( m_oPnlSinFreq   );
  }
}

//*****************************************************************************
// Set the values in the spin controls from temporary storage.

void  DlgSigGenCfg::SetValues( void )
{
  SetEvtHandlerEnabled( FALSE );

  for( int i1=ID_PNL_FST; i1<=ID_PNL_LST; i1++ )
    roGetPanel( i1 ).bSetValue( (float) m_dfPnlValue[ i1 ] );

  SetEvtHandlerEnabled( TRUE );
}

//*****************************************************************************
// Get the values from the spin controls and put them in temporary storage.

void  DlgSigGenCfg::GetValues( void )
{
  SetEvtHandlerEnabled( FALSE );

  for( int i1=ID_PNL_FST; i1<=ID_PNL_LST; i1++ )
    m_dfPnlValue[ i1 ] = roGetPanel( i1 ).dfGetValue( );

  SetEvtHandlerEnabled( TRUE );
}

//*****************************************************************************
// Test the values against each other.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigGenCfg::bValidate( void )
{
  wxString  osErrMsg;
  double    df1, df2, df3, df4;

  // Test the pulse time values
  df1 = m_oPnlPulRise  .dfGetValue( );
  df2 = m_oPnlPulWidth .dfGetValue( );
  df3 = m_oPnlPulFall  .dfGetValue( );
  df4 = m_oPnlPulPeriod.dfGetValue( );
  if(      df4 < df2 )
    osErrMsg << wxT("The rise time is greater than the pulse period.\n");
  else if( df4 < df3 )
    osErrMsg << wxT("The pulse width is greater than the pulse period.\n");
  else if( df4 < df4 )
    osErrMsg << wxT("The fall time is greater than the pulse period.\n");
  else if( df4 < df1+df2+df3 )
    osErrMsg << wxT("The pulse period is less then the total pulse width.\n");

  // Test the pulse level values
  df1 = m_oPnlPulInitV.dfGetValue( );
  df2 = m_oPnlPulMinV .dfGetValue( );
  df3 = m_oPnlPulMaxV .dfGetValue( );
  if(      df1 < df2 )
    osErrMsg << wxT("The initial value is less than the minimum value.\n");
  else if( df1 > df3 )
    osErrMsg << wxT("The initial value is greater than the maximum value.\n");
  else if( df2 > df3 )
    osErrMsg << wxT("The minimum value is less than the maximum value.\n");

  // Test overall logic
  if( df4>0.0  && (df2==0.0 && df3==0.0) )
    osErrMsg << wxT("Pulse timing has been defined but no levels.\n");
  if( df4==0.0 && (df2!=0.0 || df3!=0.0) )
    osErrMsg << wxT("Pulse levels have been defined but no timing.\n");

  if( ! osErrMsg.IsEmpty( ) )
  {
    wxMessageDialog  oMsgDlg( this, osErrMsg, wxT("Error"), wxOK | wxICON_ERROR );
    oMsgDlg.ShowModal( );
  }

  return( osErrMsg.IsEmpty( ) );
}

//*****************************************************************************
// Parse a generator command string.
//
// Argument List:
//   rosCmd - The generator command to be parsed
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigGenCfg::bSetGenCmd( const wxString & rosCmd )
{
  wxStringTokenizer  ostk1;
  wxString  os1, os2;

  // Change all characters to upper case
  os1 = rosCmd.Upper( );

  // Extract the generator command parameter list
  ostk1.SetString( os1, wxT(" =\t\r\n") );
  if( ostk1.CountTokens( ) < 3 )        return( FALSE );
  os1 = ostk1.GetNextToken( );
  if( ! os1.StartsWith( wxT(".GEN") ) ) return( FALSE );

  // Set all panel values to zero
  bClear( );
  for( int i1=ID_PNL_FST; i1<=ID_PNL_LST; i1++ )
    roGetPanel( i1 ).bSetValue( (float) 0.0, TRUE );

  // Extract each generator parameter value
  while( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( );
    if( ! ostk1.HasMoreTokens( ) ) break;
    os2 = ostk1.GetNextToken( );
    if(      os1.StartsWith( wxT("AM") ) ) m_oPnlTotAmp   .bSetValue( os2 );
    else if( os1.StartsWith( wxT("OF") ) ) m_oPnlTotOffset.bSetValue( os2 );
    else if( os1.StartsWith( wxT("FR") ) ) m_oPnlSinFreq  .bSetValue( os2 );
    else if( os1.StartsWith( wxT("PH") ) ) m_oPnlSinPhase .bSetValue( os2 );
    else if( os1.StartsWith( wxT("IN") ) ) m_oPnlPulInitV .bSetValue( os2 );
    else if( os1.StartsWith( wxT("MI") ) ) m_oPnlPulMinV  .bSetValue( os2 );
    else if( os1.StartsWith( wxT("MA") ) ) m_oPnlPulMaxV  .bSetValue( os2 );
    else if( os1.StartsWith( wxT("DE") ) ) m_oPnlPulDelay .bSetValue( os2 );
    else if( os1.StartsWith( wxT("RI") ) ) m_oPnlPulRise  .bSetValue( os2 );
    else if( os1.StartsWith( wxT("WI") ) ) m_oPnlPulWidth .bSetValue( os2 );
    else if( os1.StartsWith( wxT("FA") ) ) m_oPnlPulFall  .bSetValue( os2 );
    else if( os1.StartsWith( wxT("PE") ) ) m_oPnlPulPeriod.bSetValue( os2 );
  }

  // Update temporary storage
  GetValues( );

  return( TRUE );
}

//*****************************************************************************
// Create the generator command string.
//
// Return Values:
//   The generator command

const wxString & DlgSigGenCfg::rosGetGenCmd( void )
{
  m_osGenCmd = wxT(".GENERATOR");

  if( roGetPanel( ID_PNL_TOT_AMP ).IsEnabled( ) )
    m_osGenCmd << wxT(" AMP=")    << roGetPanel( ID_PNL_TOT_AMP    ).rosGetValue( );
  if( roGetPanel( ID_PNL_TOT_OFFSET ).IsEnabled( ) )
    m_osGenCmd << wxT(" OFFSET=") << roGetPanel( ID_PNL_TOT_OFFSET ).rosGetValue( );

  if( roGetPanel( ID_PNL_SIN_FREQ ).dfGetValue( ) != 0.0 )
  {
    if( roGetPanel( ID_PNL_SIN_FREQ ).IsEnabled( ) )
      m_osGenCmd << wxT(" FREQ=")  << roGetPanel( ID_PNL_SIN_FREQ  ).rosGetValue( );
    if( roGetPanel( ID_PNL_SIN_PHASE ).IsEnabled( ) )
      m_osGenCmd << wxT(" PHASE=") << roGetPanel( ID_PNL_SIN_PHASE ).rosGetValue( );
  }

  if( roGetPanel( ID_PNL_PUL_MAXV ).dfGetValue( ) != 0.0 )
  {
    if( roGetPanel( ID_PNL_PUL_INITV ).IsEnabled( ) )
      m_osGenCmd << wxT(" INIT=")   << roGetPanel( ID_PNL_PUL_INITV  ).rosGetValue( );
    if( roGetPanel( ID_PNL_PUL_MINV ).IsEnabled( ) )
      m_osGenCmd << wxT(" MIN=")    << roGetPanel( ID_PNL_PUL_MINV   ).rosGetValue( );
    if( roGetPanel( ID_PNL_PUL_MAXV ).IsEnabled( ) )
      m_osGenCmd << wxT(" MAX=")    << roGetPanel( ID_PNL_PUL_MAXV   ).rosGetValue( );
    if( roGetPanel( ID_PNL_PUL_DELAY ).IsEnabled( ) )
      m_osGenCmd << wxT(" DELAY=")  << roGetPanel( ID_PNL_PUL_DELAY  ).rosGetValue( );
    if( roGetPanel( ID_PNL_PUL_RISE ).IsEnabled( ) )
      m_osGenCmd << wxT(" RISE=")   << roGetPanel( ID_PNL_PUL_RISE   ).rosGetValue( );
    if( roGetPanel( ID_PNL_PUL_WIDTH ).IsEnabled( ) )
      m_osGenCmd << wxT(" WIDTH=")  << roGetPanel( ID_PNL_PUL_WIDTH  ).rosGetValue( );
    if( roGetPanel( ID_PNL_PUL_FALL ).IsEnabled( ) )
      m_osGenCmd << wxT(" FALL=")   << roGetPanel( ID_PNL_PUL_FALL   ).rosGetValue( );
    if( roGetPanel( ID_PNL_PUL_PERIOD ).IsEnabled( ) )
      m_osGenCmd << wxT(" PERIOD=") << roGetPanel( ID_PNL_PUL_PERIOD ).rosGetValue( );
  }

  return( m_osGenCmd );
}

//*****************************************************************************
// Reset all dialog settings to defaults.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigGenCfg::bClear( void )
{
  bool  bRtn=TRUE;

  // Set default spin control values
  if( ! m_oPnlTotAmp   .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlTotOffset.bClear( ) ) bRtn = FALSE;

  if( ! m_oPnlSinFreq  .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlSinPhase .bClear( ) ) bRtn = FALSE;

  if( ! m_oPnlPulInitV .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulMinV  .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulMaxV  .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulDelay .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulRise  .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulWidth .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulFall  .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulPeriod.bClear( ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Enable or disable a value panel.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigGenCfg::bEnablePanel( int iPnlID, bool bState )
{
  if( iPnlID<ID_PNL_FST || iPnlID>ID_PNL_LST ) return( FALSE );

  if( roGetPanel( iPnlID ).Enable( bState ) )  return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Set the value on a panel.
//
// Argument List:
//   iPnlID   - Spin control identifier
//   fValue   - The value to set the spin control
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigGenCfg::bSetValue( int iPnlID, float fValue,
                            const wxString & rosUnits )
{
  // Check for a valid panel identifier
  if( iPnlID<ID_PNL_FST || iPnlID>ID_PNL_LST )                 return( FALSE );

  // Set the panel value
  if( ! roGetPanel( iPnlID ).bSetValue( (float)fValue,TRUE ) ) return( FALSE );
  m_dfPnlValue[ iPnlID ] = fValue;

  if( ! rosUnits.IsEmpty( ) )
  { // Set the units
    if( ! roGetPanel( iPnlID ).bSetUnits( rosUnits ) )         return( FALSE );
    m_dfPnlValue[ iPnlID ] = roGetPanel( iPnlID ).dfGetValue( );
  }

  return( TRUE );
}

//*****************************************************************************
// Set default values in all of the dialog panels.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigGenCfg::bSetDefaults( void )
{
  bool  bRtn=TRUE;

  if( ! bSetValue( ID_PNL_TOT_AMP,    1.0, wxT("x 1")    ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_TOT_OFFSET, 0.0, wxT("V")      ) ) bRtn = FALSE;

  if( ! bSetValue( ID_PNL_SIN_FREQ,   0.0, wxT("kHz")    ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_SIN_PHASE,  0.0, wxT("Degree") ) ) bRtn = FALSE;

  if( ! bSetValue( ID_PNL_PUL_INITV,  0.0, wxT("V")      ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_MINV,   0.0, wxT("V")      ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_MAXV,   0.0, wxT("V")      ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_DELAY,  0.0, wxT("msec")   ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_RISE,   0.0, wxT("msec")   ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_WIDTH,  0.0, wxT("msec")   ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_FALL,   0.0, wxT("msec")   ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_PERIOD, 0.0, wxT("msec")   ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
//
//                             Event Handlers
//
//*****************************************************************************
// OK button event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event (not used)

void  DlgSigGenCfg::OnBtnOk( wxCommandEvent & roEvtCmd )
{
  if( bValidate( ) )
  {
    GetValues( );
    EndModal( wxID_OK );
    rosGetGenCmd( );
  }
}

//*****************************************************************************
// Clear button event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event (not used)

void  DlgSigGenCfg::OnBtnClear( wxCommandEvent & roEvtCmd )
{
  bSetDefaults( );
}

//*****************************************************************************
// Cancel button event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event (not used)

void  DlgSigGenCfg::OnBtnCancel( wxCommandEvent & roEvtCmd )
{
  SetValues( );
  EndModal( wxID_CANCEL );
}

//*****************************************************************************
