//*****************************************************************************
//                               PnlAnaBase.cpp                               *
//                              ----------------                              *
//  Started     : 26/04/2004                                                  *
//  Last Update : 08/10/2007                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "base/PnlAnaBase.hpp"

//*****************************************************************************
// Constructor.
//
// Argument List:
//   poWin - A pointer to the parent window

PnlAnaBase::PnlAnaBase( wxWindow * poWin ) : wxPanel( poWin )
{
  m_eSimrType = eSIMR_NONE;
  m_eAnaType  = eANA_NONE;

  CreateBase( );  // Create the analysis panel
  bClear( );      // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlAnaBase::~PnlAnaBase( )
{
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bClear( void )
{
  bool  bRtnValue=TRUE;

  // Set sweep default values
  if( ! m_oPnlStart.bClear( ) ) bRtnValue = FALSE;
  if( ! m_oPnlStop .bClear( ) ) bRtnValue = FALSE;
  if( ! m_oPnlStep .bClear( ) ) bRtnValue = FALSE;

  // Set input source default values
  if( bIsCreatedSigSrc( ) && m_oSbxSigSrc.IsShown( ) )
  {
    m_oChoSigSrcCpnt.Clear( );
    m_oChoSigSrcCpnt.Append( wxT("None") );
    m_oChoSigSrcCpnt.SetSelection( 0 );
    if( m_oPnlSigSrcLvl.GetParent( ) != NULL )
    {
      m_oPnlSigSrcLvl.bSetValue( (float) 0.0 );
      m_oPnlSigSrcLvl.bSetUnitsType( ChoUnits::eUNITS_NONE );
    }
  }

  // Set parameters check box default values
  m_oCbxVoltage.SetValue( TRUE );
  m_oCbxCurrent.SetValue( FALSE );
  m_oCbxPower  .SetValue( FALSE );
  m_oCbxResist .SetValue( FALSE );

  // Set the analysis temperature default value
  if( bIsCreatedTemp( ) && m_oSbxTemp.IsShown( ) )
    if( ! m_oPnlTemp.bClear( ) )
      bRtnValue = FALSE;

  m_osErrMsg.Empty( ); // Clear the error string

  return( bRtnValue );
}

//*****************************************************************************
// Create the basic display objects.

void  PnlAnaBase::CreateBase( void )
{
  if( bIsCreatedBase( ) ) return;

  // Create and add sweep parameter labels
  m_oSbxSwpPars.Create( this, ID_UNUSED, wxT(" Sweep Parameters "),
                        wxPoint( 6, 5 ), wxSize( 330, 120 ) );
  m_oPnlStart.bCreate( this, ID_PNL_START, 120, wxPoint( 18, 28 ) );
  m_oPnlStop .bCreate( this, ID_PNL_STOP,  120, wxPoint( 18, 58 ) );
  m_oPnlStep .bCreate( this, ID_PNL_STEP,  120, wxPoint( 18, 88 ) );
  m_oPnlStart.bSetName( wxT("Start Value") );
  m_oPnlStop .bSetName( wxT("Stop Value") );
  m_oPnlStep .bSetName( wxT("Step Increment") );
  m_oSbxSwpPars.SetOwnFont( FONT_SLANT );

  // Create and add simulation parameter check boxes
  m_oSbxCalcPars.Create( this, ID_UNUSED, wxT(" Parameters "),
                         wxPoint( 343, 5 ), wxSize( 105, 122 ) );
  m_oCbxVoltage .Create( this, ID_CBX_VOLT, wxT("Voltage"),
                         wxPoint( 349, 25 ) );
  m_oCbxCurrent .Create( this, ID_CBX_CURR, wxT("Current"),
                         wxPoint( 349, 50 ) );
  m_oCbxPower   .Create( this, ID_CBX_PWR,  wxT("Power"),
                         wxPoint( 349, 75 ) );
  m_oCbxResist  .Create( this, ID_CBX_RES,  wxT("Resistance"),
                         wxPoint( 349, 100 ) );
  m_oSbxCalcPars.SetOwnFont( FONT_SLANT );

  // Create and add .OPTIONS configuration dialog button
  m_oBtnOPTIONS.Create( this, ID_BTN_OPTIONS, wxT(".OPTIONS ..."),
                        wxPoint( 343, 137 ), wxSize( 105, 29 ) );
}

//*****************************************************************************
// Create the source component display objects.

void  PnlAnaBase::CreateSigSrc( void )
{
  if( bIsCreatedSigSrc( ) ) return;

  // Create and add input source
  m_oSbxSigSrc    .Create ( this, ID_UNUSED, wxT(" Signal Source "),
                            wxPoint(  6, 175 ), wxSize( 330, 55 ) );
  m_oChoSigSrcCpnt.Create ( this, ID_CHO_SIGSRCCPNT,
                            wxPoint( 18, 195 ), wxSize( 121, -1 ) );
  m_oPnlSigSrcLvl .bCreate( this, ID_PNL_SIGSRCLVL, -1, wxPoint( 143, 193 ) );

  m_oSbxSigSrc.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Create the simulation parameter complex part display objects.

void  PnlAnaBase::CreateCpxPrt( void )
{
  if( bIsCreatedCpxPrt( ) ) return;

  // Create and add simulation parameter complex part check boxes
  m_oSbxCpxPrt.Create( this, ID_UNUSED,    wxT(" Complex Parts "),
                       wxPoint( 455,  5 ), wxSize( 110, 146 ) );
  m_oCbxMag   .Create( this, ID_CBX_MAG,   wxT("Magnitude"),
                       wxPoint( 461, 24 ) );
  m_oCbxPhase .Create( this, ID_CBX_PHASE, wxT("Phase"),
                       wxPoint( 461, 49 ) );
  m_oCbxReal  .Create( this, ID_CBX_REAL,  wxT("Real Part"),
                       wxPoint( 461, 74 ) );
  m_oCbxImag  .Create( this, ID_CBX_IMAG,  wxT("Imag. Part"),
                       wxPoint( 461, 99 ) );
  m_oCbxMagDb .Create( this, ID_CBX_MAGDB, wxT("Mag. in dBV"),
                       wxPoint( 461,124 ) );

  m_oSbxCpxPrt.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Create the analysis temperature display objects.

void  PnlAnaBase::CreateTemp( void )
{
  if( bIsCreatedTemp( ) ) return;

  // Create and add analysis temperature
  m_oSbxTemp. Create( this, ID_UNUSED, wxT(" Temperature "),
                      wxPoint( 343, 175 ), wxSize( 205, 55 ) );
  m_oPnlTemp.bCreate( this, ID_PNL_TEMP, -1,
                      wxPoint( 355, 193 ) );
  m_oPnlTemp.bSetUnitsType( ChoUnits::eUNITS_TEMP );

  m_oSbxTemp.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Load the signal source choice box with the components which may be chosen as
// a signal source.
//
// Argument List:
//   roSim    - The Simulation object the components come from
//   osPrefix - A string containing the first letter of the desired components

void  PnlAnaBase::LoadSigSrcCpnts( Simulation & roSim, wxString osPrefixes )
{
  wxString  os1;
  wxChar    oc1;
  uint      ui1;

  // Clear the signal source choice box
  m_oChoSigSrcCpnt.Clear( );
  m_oChoSigSrcCpnt.Append( wxT("None") );

  // Load the selected components into the signal source choice box
  const wxArrayString & roasCpnts = roSim.roasGetCpntLbls( );
  for( ui1=0; ui1<roasCpnts.GetCount( ); ui1++ )
  {
    os1 = roasCpnts.Item( ui1 );
    if( os1.Length( ) <= 0 )                    continue;

    oc1 = os1.GetChar( 0 );
    if( osPrefixes.Find( oc1 ) == wxNOT_FOUND ) continue;

    m_oChoSigSrcCpnt.Append( os1 );
  }

  // Select the default component
  m_oChoSigSrcCpnt.SetStringSelection( wxT("None") );
}

//*****************************************************************************
// Set the signal source.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetSigSrc( Simulation & roSim )
{
  wxString  os1;

  if( ! bIsCreatedSigSrc( ) )                               return( FALSE );

  if( ! roSim.rosGetSigSrc( ).IsEmpty( ) )
  {
    // Extract and set the source component label
    os1 = roSim.rosGetSigSrc( ).BeforeFirst( wxT(' ') );
    if( m_oChoSigSrcCpnt.FindString( os1 ) == wxNOT_FOUND ) return( FALSE );
    m_oChoSigSrcCpnt.SetStringSelection( os1 );

    // Extract and set the source value and units
    if( m_oPnlSigSrcLvl.GetParent( ) != NULL )
    {
      if( ! bSetSigSrcUnits( ) )                            return( FALSE );
      os1 = roSim.rosGetSigSrc( ).AfterLast( wxT(' ') );
      if( m_oPnlSigSrcLvl.bSetValue( os1 ) )                return( FALSE );
    }
  }
  else
  {
    // Set the default source component label
    m_oChoSigSrcCpnt.SetStringSelection( wxT("None") );

    // Set the default source level
    if( m_oPnlSigSrcLvl.GetParent( ) != NULL )
    {
      m_oPnlSigSrcLvl.bSetValue( (float) 0.0 );
      m_oPnlSigSrcLvl.bSetUnitsType( ChoUnits::eUNITS_NONE );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Set the source units type based on the type of the source component.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetSigSrcUnits( void )
{
  if( m_oPnlSigSrcLvl.GetParent( ) == NULL ) return( FALSE );

  switch( wxToupper( m_oChoSigSrcCpnt.GetStringSelection( ).GetChar( 0 ) ) )
  {
    case wxT('C') :  // Units of capacitance
      m_oPnlSigSrcLvl.bSetUnitsType( ChoUnits::eUNITS_CAP );
      break;

    case wxT('L') :  // Units of inductance
      m_oPnlSigSrcLvl.bSetUnitsType( ChoUnits::eUNITS_IND );
      break;

    case wxT('R') :  // Units of resistance
      m_oPnlSigSrcLvl.bSetUnitsType( ChoUnits::eUNITS_RES );
      break;

    case wxT('I') :  // Units of current
      m_oPnlSigSrcLvl.bSetUnitsType( ChoUnits::eUNITS_CURR );
      break;

    case wxT('V') :  // Units of voltage
      m_oPnlSigSrcLvl.bSetUnitsType( ChoUnits::eUNITS_VOLT );
      break;

    default :        // No units
      m_oPnlSigSrcLvl.bSetUnitsType( ChoUnits::eUNITS_NONE );
      break;
  }

  return( TRUE );
}

//*****************************************************************************
// Select the simulator engine to be used.
//
// Arguments:
//   eSimrType - The simulator engine type
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetSimrType( eSimrType eSimrType )
{
  if( eSimrType<eSIMR_FST || eSimrType>eSIMR_LST ) return( FALSE );

  m_eSimrType = eSimrType;

  return( TRUE );
}

//*****************************************************************************
// Select the analysis to be performed.
//
// Arguments:
//   eAnaType - The analysis type
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetAnaType( eAnaType eAnaType )
{
  if( eAnaType<eANA_FST || eAnaType>eANA_LST ) return( FALSE );

  m_eAnaType = eAnaType;

  return( TRUE );
}

//*****************************************************************************
// Load information from a Simulation object.
//
// Argument List:
//   roSim - The Simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bLoad( Simulation & roSim )
{
  bool  bRtnValue=TRUE;

  // Set the sweep values
  if( ! m_oPnlStart.bSetValue( (double) roSim.fGetSwpStart( ) ) )
    bRtnValue = FALSE;
  if( ! m_oPnlStop .bSetValue( (double) roSim.fGetSwpStop( )  ) )
    bRtnValue = FALSE;
  if( ! m_oPnlStep .bSetValue( (double) roSim.fGetSwpStep( )  ) )
    bRtnValue = FALSE;
  if( m_oPnlStart.dfGetValue( ) == 0.0 )
    m_oPnlStart.bSetUnits( m_oPnlStop.rosGetUnits( ) );

  // Set the parameters to derive
  m_oCbxVoltage.SetValue( roSim.bGetOutPar( ePAR_VLT ) );
  if( roSim.eGetSimrType( ) == eSIMR_GNUCAP )
  {
    m_oCbxCurrent.SetValue( roSim.bGetOutPar( ePAR_CUR ) );
    m_oCbxPower  .SetValue( roSim.bGetOutPar( ePAR_PWR ) );
    m_oCbxResist .SetValue( roSim.bGetOutPar( ePAR_RES ) );
  }

  // Set the analysis temperature
  if( bIsCreatedTemp( ) )
    if( ! m_oPnlTemp.bSetValue( roSim.fGetTempC( ) ) )
      bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Save information to a Simulation object.
//
// Argument List:
//   roSim - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSave( Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  float              f1;
  size_t             sz1;
  bool               b1;

  m_osErrMsg.Empty( );

  roSim.ClrSimCmds( );  // Clear simulation commands attribute
  roSim.ClrSigSrc( );   // Clear source component attribute

  // Set the simulation type
  if( ! roSim.bSetAnaType( m_eAnaType ) )
    SetErrMsg( wxT("Invalid simulation type specifier.") );

  // Set the sweep start value
  f1 = (float) m_oPnlStart.dfGetValue( );
  if( ! roSim.bSetSwpStart( f1 ) )
    SetErrMsg( wxT("Invalid sweep start value.") );

  // Set the sweep stop value
  f1 = (float) m_oPnlStop.dfGetValue( );
  if( ! roSim.bSetSwpStop( f1 ) )
    SetErrMsg( wxT("Invalid sweep stop value.") );
  if( roSim.fGetSwpStop( ) < roSim.fGetSwpStart( ) )
    SetErrMsg( wxT("Start value greater than stop value is not permitted.") );

  // Set the sweep scale or the initial conditions
  if( bIsCreatedScale( ) )
    if( ! roSim.bSetSwpScale( m_oRbxStepScale.GetSelection( ) ) )
      SetErrMsg( wxT("Invalid step scale value.") );
  if( bIsCreatedInitC( ) )
    if( ! roSim.bSetSwpScale( m_oRbxInitC.GetSelection( ) ) )
      SetErrMsg( wxT("Invalid initial conditions value.") );

  // Set the sweep step value
  f1 = (float) m_oPnlStep.dfGetValue( );
  if( f1 > 0.0 )
  {
    if( roSim.fGetSwpStop( ) != roSim.fGetSwpStart( ) )
    {
      if( bIsCreatedScale( ) )
      {
        switch( m_oRbxStepScale.GetSelection( ) )
        {
          case eSCALE_LIN :
            if( f1 > fabs( roSim.fGetSwpStop( ) - roSim.fGetSwpStart( ) ) )
              SetErrMsg( wxT("Step size is greater than sweep range.") );
            break;

          case eSCALE_LOG :
            if( f1 == 1.0 )
              SetErrMsg( wxT("Step multiplier invalid, must be non-unity.") );
            break;

          case eSCALE_DEC :
          case eSCALE_OCT :
          default :         break;
        }
      }

      if( bIsCreatedInitC( ) )
        if( f1 > fabs( roSim.fGetSwpStop( ) - roSim.fGetSwpStart( ) ) )
          SetErrMsg( wxT("Step size is greater than sweep range.") );
    }

    if( ! roSim.bSetSwpStep( f1 ) )
      SetErrMsg( wxT("Invalid sweep step value.") );
  }
  else
    SetErrMsg( wxT("Step size less than or equal to zero is not permitted.") );

  // Set the component to be used as the sweep source
  if( bIsCreatedSigSrc( ) && m_oChoSigSrcCpnt.IsShown( ) )
  {
    if( m_oChoSigSrcCpnt.GetSelection( ) > 0 )
    {
      // Find the component in the net list
      for( sz1=0; sz1<roSim.roasGetCpnts( ).GetCount( ); sz1++ )
      {
        os1 = roSim.roasGetCpnts( ).Item( sz1 );
        if( os1.BeforeFirst(wxT(' ')) == m_oChoSigSrcCpnt.GetStringSelection() )
          break;
      }

      // Create the new component definition line
      ostk1.SetString( os1 );
      os1.Empty( );
      for( sz1=0; sz1<3; sz1++ )
        os1 << ostk1.GetNextToken( ) << wxT(' ');
      if( m_oPnlSigSrcLvl.GetParent( ) != NULL )
           os1 << m_oPnlSigSrcLvl.rosGetValue( );
      else os1 << wxT('0');
    }
    else
    {
      if( eGetAnaType( ) != eANA_OP )
      {
        os1.Empty( );
        SetErrMsg( wxT("No source component has been selected.") );
      }
    }
    roSim.bSetSigSrc( os1 );
  }

  // Store the parameters to derive
  roSim.bSetOutPar( ePAR_VLT, m_oCbxVoltage.GetValue( ) );
  roSim.bSetOutPar( ePAR_CUR, m_oCbxCurrent.GetValue( ) );
  roSim.bSetOutPar( ePAR_PWR, m_oCbxPower  .GetValue( ) );
  roSim.bSetOutPar( ePAR_RES, m_oCbxResist .GetValue( ) );
  for( sz1=ePAR_FST, b1=FALSE; sz1<=ePAR_LST; sz1++ )
    if( roSim.bGetOutPar( (eParType) sz1 ) ) b1 = TRUE;
  if( ! b1 ) SetErrMsg( wxT("No parameters have been selected.") );

  // Set the analysis temperature
  if( bIsCreatedTemp( ) && m_oSbxTemp.IsShown( ) )
  {
    f1 = (float) m_oPnlTemp.dfGetValue( );
    if( ! roSim.bSetTempC( f1 ) )
      SetErrMsg( wxT("Invalid analysis temperature value.") );
  }

  return( bIsOk( ) );
}

//*****************************************************************************
