/* $Id: gvdevice_xlib.c,v 1.14 2006/01/30 15:43:05 ellson Exp $ $Revision: 1.14 $ */
/* vim:set shiftwidth=4 ts=8: */

/**********************************************************
*      This software is part of the graphviz package      *
*                http://www.graphviz.org/                 *
*                                                         *
*            Copyright (c) 1994-2004 AT&T Corp.           *
*                and is licensed under the                *
*            Common Public License, Version 1.0           *
*                      by AT&T Corp.                      *
*                                                         *
*        Information and Software Systems Research        *
*              AT&T Research, Florham Park NJ             *
**********************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <cairo.h>
#include <cairo-xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/Xrender.h>

#include "gvplugin_device.h"

typedef struct window_xlib_s {
    Window win;
    unsigned long event_mask;
    Pixmap pix;
    GC gc;
    Visual *visual;
    Colormap cmap;
    int depth;
    Atom wm_delete_window_atom;
} window_xlib_t;

static void handle_configure_notify_xlib(GVJ_t * job, XConfigureEvent * cev)
{
    if (job->fit_mode)
        job->zoom = MIN((double) cev->width / (double) job->width,
			(double) cev->height / (double) job->height);
    if (cev->width > job->width || cev->height > job->height)
        job->has_grown = 1;
    job->width = cev->width;
    job->height = cev->height;
    job->needs_refresh = 1;
}

static void handle_expose_xlib(GVJ_t * job, XExposeEvent * eev)
{
    window_xlib_t *window;

    window = (window_xlib_t *)job->window;
    XCopyArea(eev->display, window->pix, eev->window, window->gc,
              eev->x, eev->y, eev->width, eev->height, eev->x, eev->y);
}

static void handle_client_message_xlib(GVJ_t * job, XClientMessageEvent * cmev)
{
    window_xlib_t *window;

    window = (window_xlib_t *)job->window;
    if (cmev->format == 32
        && (Atom) cmev->data.l[0] == window->wm_delete_window_atom)
        exit(0);
}

static bool handle_keypress_xlib(GVJ_t *job, XKeyEvent *kev)
{
    
    int i;
    KeyCode *keycodes;

    keycodes = (KeyCode *)job->keycodes;
    for (i=0; i < job->numkeys; i++) {
	if (kev->keycode == keycodes[i])
	    return (job->keybindings[i].callback)(job);
    }
    return FALSE;
}

static Visual *find_argb_visual(Display * dpy, int scr)
{
    XVisualInfo *xvi;
    XVisualInfo template;
    int nvi;
    int i;
    XRenderPictFormat *format;
    Visual *visual;

    template.screen = scr;
    template.depth = 32;
    template.class = TrueColor;
    xvi = XGetVisualInfo(dpy,
                         VisualScreenMask |
                         VisualDepthMask |
                         VisualClassMask, &template, &nvi);
    if (!xvi)
        return 0;
    visual = 0;
    for (i = 0; i < nvi; i++) {
        format = XRenderFindVisualFormat(dpy, xvi[i].visual);
        if (format->type == PictTypeDirect && format->direct.alphaMask) {
            visual = xvi[i].visual;
            break;
        }
    }

    XFree(xvi);
    return visual;
}

static void xlibgen_init_window(GVJ_t *job, Display *dpy, int scr)
{
    int argb = 0;
    const char *geometry = NULL;
    const char *base = "";
    XGCValues gcv;
    XSetWindowAttributes attributes;
    XWMHints *wmhints;
    XSizeHints *normalhints;
    XClassHint *classhint;
    unsigned long attributemask = 0;
    char *name;
    window_xlib_t *window;

    window = (window_xlib_t *)malloc(sizeof(window_xlib_t));
    if (window == NULL) {
	fprintf(stderr, "Failed to malloc window_xlib_t\n");
	return;
    }
    job->window = (void *)window;
    job->fit_mode = 0;
    job->needs_refresh = 1;

    job->dpi.x = DisplayWidth(dpy, scr) * 25.4 / DisplayWidthMM(dpy, scr);
    job->dpi.y = DisplayHeight(dpy, scr) * 25.4 / DisplayHeightMM(dpy, scr);

    /* adjust width/height for real dpi */
    job->width *= job->dpi.x / POINTS_PER_INCH;
    job->height *= job->dpi.y / POINTS_PER_INCH;

    if (argb && (window->visual = find_argb_visual(dpy, scr))) {
        window->cmap = XCreateColormap(dpy, RootWindow(dpy, scr),
                                    window->visual, AllocNone);
        attributes.override_redirect = False;
        attributes.background_pixel = 0;
        attributes.border_pixel = 0;
        attributes.colormap = window->cmap;
        attributemask = (CWBackPixel |
                         CWBorderPixel | CWOverrideRedirect | CWColormap);
        window->depth = 32;
    } else {
        window->cmap = DefaultColormap(dpy, scr);
        window->visual = DefaultVisual(dpy, scr);
        attributes.background_pixel = WhitePixel(dpy, scr);
        attributes.border_pixel = BlackPixel(dpy, scr);
        attributemask = (CWBackPixel | CWBorderPixel);
        window->depth = DefaultDepth(dpy, scr);
    }

    if (geometry) {
        int x, y;
        XParseGeometry(geometry, &x, &y, &job->width, &job->height);
    }

    window->win = XCreateWindow(dpy, RootWindow(dpy, scr),
                             0, 0, job->width, job->height, 0, window->depth,
                             InputOutput, window->visual,
                             attributemask, &attributes);

    name = malloc(strlen("graphviz: ") + strlen(base) + 1);
    strcpy(name, "graphviz: ");
    strcat(name, base);

    normalhints = XAllocSizeHints();
    normalhints->flags = 0;
    normalhints->x = 0;
    normalhints->y = 0;
    normalhints->width = job->width;
    normalhints->height = job->height;

    classhint = XAllocClassHint();
    classhint->res_name = "graphviz";
    classhint->res_class = "Graphviz";

    wmhints = XAllocWMHints();
    wmhints->flags = InputHint;
    wmhints->input = True;

    Xutf8SetWMProperties(dpy, window->win, name, base, 0, 0,
                         normalhints, wmhints, classhint);
    XFree(wmhints);
    XFree(classhint);
    XFree(normalhints);
    free(name);

    window->pix = XCreatePixmap(dpy, window->win, job->width, job->height,
		window->depth);
    if (argb)
        gcv.foreground = 0;
    else
        gcv.foreground = WhitePixel(dpy, scr);
    window->gc = XCreateGC(dpy, window->pix, GCForeground, &gcv);
    XFillRectangle(dpy, window->pix, window->gc, 0, 0, job->width, job->height);

    window->event_mask = (
          ButtonPressMask
        | ButtonReleaseMask
        | PointerMotionMask
        | KeyPressMask
        | StructureNotifyMask
        | ExposureMask);
    XSelectInput(dpy, window->win, window->event_mask);
    window->wm_delete_window_atom =
        XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(dpy, window->win, &window->wm_delete_window_atom, 1);
    XMapWindow(dpy, window->win);
}

static void xlibgen_finalize(GVJ_t *firstjob)
{
    GVJ_t *job;
    Display *dpy;
    int scr;
    window_xlib_t *window;
    Pixmap pix;
    XEvent xev;
    cairo_surface_t *surface;
    bool done = FALSE;
    const char *display_name = NULL;
    int i;
    KeySym keysym;
    KeyCode *keycodes;
    pointf pointer;

    dpy = XOpenDisplay(display_name);
    if (dpy == NULL) {
	fprintf(stderr, "Failed to open XLIB display: %s\n",
		XDisplayName(NULL));
	return;
    }
    scr = DefaultScreen(dpy);

    keycodes = (KeyCode *)malloc(firstjob->numkeys * sizeof(KeyCode));
    if (keycodes == NULL) {
        fprintf(stderr, "Failed to malloc %d*KeyCode\n", firstjob->numkeys);
        return;
    }
    for (i = 0; i < firstjob->numkeys; i++) {
        keysym = XStringToKeysym(firstjob->keybindings[i].keystring);
        if (keysym == NoSymbol)
            fprintf(stderr, "ERROR: No keysym for \"%s\"\n", firstjob->keybindings[i].keystring);
        else
            keycodes[i] = XKeysymToKeycode(dpy, keysym);
    }
    firstjob->keycodes = (void*)keycodes;

    for (job = firstjob; job; job = job->next_active)
	xlibgen_init_window(job, dpy, scr);

    while (1) {
        if (!XPending(dpy)) {
	    for (job = firstjob; job; job = job->next_active) {
 	        window = (window_xlib_t *)job->window;

		if (job->has_grown) {
		    pix = XCreatePixmap(dpy, window->win,
			job->width, job->height, window->depth);
		    XFillRectangle(dpy, pix, window->gc, 0, 0,
			job->width, job->height);
		    XCopyArea(dpy, window->pix, pix, window->gc, 0, 0,
			job->width, job->height, 0, 0);
		    XFreePixmap(dpy, window->pix);
    		    window->pix = pix;
		    job->has_grown = 0;
		    job->needs_refresh = 1;
		}
		if (job->needs_refresh) {
		    XFillRectangle(dpy, window->pix, window->gc, 0, 0,
                	job->width, job->height);
		    surface = cairo_xlib_surface_create(dpy,
					window->pix, window->visual,
					job->width, job->height);
    		    job->surface = (void *)cairo_create(surface);
		    cairo_surface_destroy(surface);
                    (job->callbacks->refresh)(job);
		    XCopyArea(dpy, window->pix, window->win, window->gc,
			0, 0, job->width, job->height, 0, 0);
                    job->needs_refresh = 0;
		}
            }
	}
    
        XNextEvent(dpy, &xev);

	for (job = firstjob; job; job = job->next_active) {
	    window = (window_xlib_t *)job->window;
	    if (xev.xany.window == window->win) {
                switch (xev.xany.type) {
                case ButtonPress:
		    pointer.x = (double)xev.xbutton.x;
		    pointer.y = (double)xev.xbutton.y;
                    (job->callbacks->button_press)(job, xev.xbutton.button, pointer);
                    break;
                case MotionNotify:
		    pointer.x = (double)xev.xbutton.x;
		    pointer.y = (double)xev.xbutton.y;
                    (job->callbacks->motion)(job, pointer);
                    break;
                case ButtonRelease:
		    pointer.x = (double)xev.xbutton.x;
		    pointer.y = (double)xev.xbutton.y;
                    (job->callbacks->button_release)(job, xev.xbutton.button, pointer);
                    break;
                case KeyPress:
		    done = handle_keypress_xlib(job, &xev.xkey);
                    break;
                case ConfigureNotify:
                    handle_configure_notify_xlib(job, &xev.xconfigure);
                    break;
                case Expose:
                    handle_expose_xlib(job, &xev.xexpose);
                    break;
                case ClientMessage:
                    handle_client_message_xlib(job, &xev.xclient);
                    break;
                }
	    	break;
	    }
	}
	if (done)
	    break;
    }
    XCloseDisplay(dpy);
    free(keycodes);
    firstjob->keycodes = NULL;
}

gvdevice_engine_t xlibgen_device_engine = {
    xlibgen_finalize,
};

gvplugin_installed_t gvdevice_xlibgen_types[] = {
    {0, "xlib", 0, &xlibgen_device_engine, NULL},
    {0, NULL, 0, NULL, NULL}
};
