/*
 * Copyright 2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.gradle.api.internal.resource

import org.gradle.util.TemporaryFolder
import org.gradle.util.TestFile
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import static org.hamcrest.Matchers.*
import static org.junit.Assert.*

class UriResourceTest {
    private TestFile testDir;
    private File file;
    private URI fileUri;
    @Rule
    public TemporaryFolder tmpDir = new TemporaryFolder();

    @Before
    public void setUp() throws URISyntaxException {
        testDir = tmpDir.createDir('dir');
        file = new File(testDir, 'build.script');
        fileUri = file.toURI();
    }

    private URI createJar() throws URISyntaxException {
        TestFile jarFile = tmpDir.dir.file('test.jar');
        testDir.file('ignoreme').write('content');
        testDir.zipTo(jarFile);
        return new URI("jar:${jarFile.toURI()}!/build.script")
    }

    @Test
    public void canConstructResourceFromFile() {
        UriResource resource = new UriResource('<display-name>', file);
        assertThat(resource.file, equalTo(file));
        assertThat(resource.URI, equalTo(fileUri));
    }

    @Test
    public void canConstructResourceFromFileURI() {
        UriResource resource = new UriResource('<display-name>', fileUri);
        assertThat(resource.file, equalTo(file));
        assertThat(resource.URI, equalTo(fileUri));
    }

    @Test
    public void canConstructResourceFromJarURI() {
        URI jarUri = createJar()
        UriResource resource = new UriResource('<display-name>', jarUri);
        assertThat(resource.file, nullValue());
        assertThat(resource.URI, equalTo(jarUri));
    }

    @Test
    public void readsFileContentWhenFileExists() throws IOException {
        file.text = '<content>'

        UriResource resource = new UriResource('<display-name>', file);
        assertTrue(resource.exists)
        assertThat(resource.text, equalTo('<content>'));
    }

    @Test
    public void hasNoContentWhenFileDoesNotExist() {
        UriResource resource = new UriResource('<display-name>', file);
        assertFalse(resource.exists)
        try {
            resource.text
            fail()
        } catch (ResourceNotFoundException e) {
            assertThat(e.message, equalTo("Could not read <display-name> '$file' as it does not exist." as String))
        }
    }

    @Test
    public void hasNoContentWhenFileIsADirectory() {
        TestFile dir = testDir.file('somedir').createDir()
        UriResource resource = new UriResource('<display-name>', dir);
        assertTrue(resource.exists)
        try {
            resource.text
            fail()
        } catch (ResourceException e) {
            assertThat(e.message, equalTo("Could not read <display-name> '$dir' as it is a directory." as String))
        }
    }
    
    @Test
    public void readsFileContentUsingFileUriWhenFileExists() {
        file.text = '<content>'

        UriResource resource = new UriResource('<display-name>', fileUri);
        assertTrue(resource.exists)
        assertThat(resource.text, equalTo('<content>'));
    }

    @Test
    public void hasNoContentWhenUsingFileUriAndFileDoesNotExist() {
        UriResource resource = new UriResource('<display-name>', fileUri);
        assertFalse(resource.exists)
        try {
            resource.text
            fail()
        } catch (ResourceNotFoundException e) {
            assertThat(e.message, equalTo("Could not read <display-name> '$file' as it does not exist." as String))
        }
    }

    @Test
    public void readsFileContentUsingJarUriWhenFileExists() {
        file.text = '<content>'

        UriResource resource = new UriResource('<display-name>', createJar());
        assertTrue(resource.exists)
        assertThat(resource.text, equalTo('<content>'));
    }

    @Test
    public void hasNoContentWhenUsingJarUriAndFileDoesNotExistInJar() {
        URI jarUri = createJar()
        UriResource resource = new UriResource('<display-name>', jarUri);
        assertFalse(resource.exists)
        try {
            resource.text
            fail()
        } catch (ResourceNotFoundException e) {
            assertThat(e.message, equalTo("Could not read <display-name> '$jarUri' as it does not exist." as String))
        }
    }

    @Test
    public void hasNoContentWhenUsingHttpUriAndFileDoesNotExist() {
        UriResource resource = new UriResource('<display-name>', new URI("http://www.gradle.org/unknown.txt"));
        assertFalse(resource.exists)
        try {
            resource.text
            fail()
        } catch (ResourceNotFoundException e) {
            assertThat(e.message, equalTo("Could not read <display-name> 'http://www.gradle.org/unknown.txt' as it does not exist." as String))
        }
    }

    @Test
    public void usesFilePathToBuildDisplayNameWhenUsingFile() {
        UriResource resource = new UriResource("<file-type>", file);
        assertThat(resource.displayName, equalTo(String.format("<file-type> '%s'", file.absolutePath)));
    }

    @Test
    public void usesFilePathToBuildDisplayNameWhenUsingFileUri() {
        UriResource resource = new UriResource("<file-type>", fileUri);
        assertThat(resource.displayName, equalTo(String.format("<file-type> '%s'", file.absolutePath)));
    }

    @Test
    public void usesUriToBuildDisplayNameWhenUsingHttpUri() {
        UriResource resource = new UriResource("<file-type>", new URI("http://www.gradle.org/unknown.txt"));
        assertThat(resource.displayName, equalTo('<file-type> \'http://www.gradle.org/unknown.txt\''))
    }
}
