#include <qsizepolicy.h>
#include <qpainter.h>
#include <qmessagebox.h>
#include <qcursor.h>

#include "gpsshogi/gui/board.h"
#include "osl/pieceStand.h"
#include "osl/state/numEffectState.h"

#if QT_VERSION >= 0x040000
#include <QMouseEvent>
#endif

bool gpsshogi::gui::Board::highlight_enabled = false;

gpsshogi::gui::Board::Board(const osl::state::NumEffectState& state,
			    QWidget *parent)
  : AbstractBoard(parent),
    senteView(true), state(state), allowManualMovement(true),
    effect(NONE)
{
  QColor color(255, 255, 223); 
  QPalette palette; 
  palette.setColor(QPalette::Window, color); 
  setPalette(palette); 
  setAutoFillBackground(true);
  setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
}

void gpsshogi::gui::
Board::mousePressEvent(QMouseEvent *e)
{
  if (!allowManualMovement)
    return;
  osl::Square pos = getSquare(e->x(), e->y());
  if (!pos.isValid())
    return;
  if (! pos.isPieceStand())
  {
    osl::Piece piece = pieceAt(pos);
    if (piece.isPiece() && piece.owner() == state.turn())
      selectedPiece = state.pieceAt(pos);
  }
  else
  {
    int pieceNum = -1;
    if (((state.turn() == osl::BLACK && senteView)
	 || (state.turn() == osl::WHITE && !senteView))
	&& BLACK_STAND_X <= e->x() && e->x() <= BLACK_STAND_X + BOX_SIZE)
    {
      pieceNum = (e->y() - BOX_SIZE * 2) / BOX_SIZE;
    }
    if (((state.turn() == osl::WHITE && senteView)
	 || (state.turn() == osl::BLACK && !senteView))
	&& 0 <= e->x() && e->x() <= 0 + BOX_SIZE)
    {
      pieceNum = 9 - (e->y() / BOX_SIZE);
    }
    if (0 <= pieceNum && pieceNum < (int)osl::PieceStand::order.size())
    {
      osl::Ptype pt = osl::PieceStand::order[pieceNum];
      for (int i = 0; i < 40; i++)
      {
	const osl::Piece p = pieceOf(i);
	if (p.ptype() == pt && p.square() == osl::Square::STAND()
	    && p.owner() == state.turn())
	{
	  selectedPiece = p;
	  break;
	}
      }
    }
  }
  if (!selectedPiece.isEmpty())
  {
    QCursor cursor(pieceToImage(selectedPiece));
    setCursor(cursor);
    update();
  }
}

void gpsshogi::gui::
Board::mouseReleaseEvent(QMouseEvent *e)
{
  if (!selectedPiece.isEmpty())
  {
    osl::Square pos = getSquare(e->x(), e->y());
    if (pos.isValid() && pos.isOnBoard() && pieceAt(pos).ptype() != osl::KING)
    {
      osl::Move mv = osl::Move::INVALID();
      osl::Move m(selectedPiece.square(), pos,
		  selectedPiece.ptype(),
		  pieceAt(pos).ptype(),
		  false, state.turn());
      if (state.isValidMove(m, false))
	mv = m;

      osl::Ptype piece = selectedPiece.ptype();
      if (osl::isBasic(piece) && piece != osl::KING && piece != osl::GOLD
	  && selectedPiece.square() != osl::Square::STAND()
	  && (((state.turn() == osl::BLACK && 1 <= pos.y() && pos.y() <= 3)
	      || (state.turn() == osl::WHITE
		  && 7 <= pos.y() && pos.y() <= 9))
	      || ((state.turn() == osl::BLACK
		   && 1 <= selectedPiece.square().y() && selectedPiece.square().y() <= 3)
		  || (state.turn() == osl::WHITE
		      && 7 <= selectedPiece.square().y() && selectedPiece.square().y() <= 9))))
      {
	osl::Move promote_move(selectedPiece.square(), pos,
			       osl::promote(selectedPiece.ptype()),
			       state.pieceAt(pos).ptype(),
			       true, state.turn());
	if (state.isValidMove(promote_move, false))
	{
	  // non-promote move is invalid
	  if (! mv.isNormal())
	    mv = promote_move;
	  else
	  {
	    int ret = QMessageBox::question(this, "Promote", "Promote?",
					    QMessageBox::Yes | QMessageBox::Default,
					    QMessageBox::No);
	    if (ret == QMessageBox::Yes)
	      mv = promote_move;
	  }
	}
      }
      if (mv.isNormal())
	move(mv);
    }
    selectedPiece = osl::Piece::EMPTY();
  }
  setCursor(QCursor());
  update();
}


void gpsshogi::gui::
Board::move(osl::Move m)
{
  state.makeMove(m);  
  highlight = m.to();
  emit moved(m);
  emit statusChanged();
  update();
}

void gpsshogi::gui::
Board::setState(const osl::state::NumEffectState& sstate, osl::Move last_move)
{
  if (!(state == sstate))
  {
    state = sstate;
    highlight = last_move.isNormal() ? last_move.to() : osl::Square();
    update();
    emit statusChanged();
  }
}

void gpsshogi::gui::
Board::setView(bool sente)
{
  senteView = sente;
  update();
}

void gpsshogi::gui::
Board::toggleOrientation()
{
  senteView = !senteView;
  update();
}

osl::Piece gpsshogi::gui::Board::pieceOf(int i)
{
  return state.pieceOf(i);
}

osl::Piece gpsshogi::gui::Board::pieceAt(osl::Square pos)
{
  return state.pieceAt(pos);
}

int gpsshogi::gui::Board::countPiecesOnStand(osl::Player p, osl::Ptype ptype)
{
  return state.countPiecesOnStand(p, ptype);
}

osl::Piece gpsshogi::gui::Board::getReservePiece(osl::Ptype)
{
  return osl::Piece::EMPTY();
}

osl::Piece gpsshogi::gui::Board::getStandPiece(osl::Player p, osl::Ptype ptype)
{
  for (size_t i = 0; i < 40; ++i)
  {
    const osl::Piece piece = pieceOf(i);
    if (piece.ptype() == ptype &&
	!piece.square().isValid() && piece.square().isPieceStand() &&
	piece.owner() == p)
    {
      return piece;
    }
  }
  return osl::Piece::EMPTY();
}

void gpsshogi::gui::Board::paintEvent(QPaintEvent *pe)
{
  QPainter painter(this);
  if (effect != NONE)
  {
    osl::state::NumEffectState nState(state);
    for (int x = 1; x <= 9; x++)
    {
      for (int y = 1; y <= 9; y++)
      {
	int r = 255;
	int g = 255;
	int b = 255;
	const osl::Square pos(x, y);
	int black_count = nState.countEffect(osl::BLACK, pos);
	int white_count = nState.countEffect(osl::WHITE, pos);

	if (effect != WHITE)
	  r = std::max(255 - black_count * 16, 0);
	if (effect != BLACK)
	  b = std::max(255 - white_count * 16, 0);
	g = std::max(255 - (black_count + white_count) * 16, 0);
	
	{
	  QColor color(r, g, b);
	  QPoint point = positionToPoint(pos);
	  QBrush brush(color);
	  painter.fillRect(point.x(), point.y(), BOX_SIZE, BOX_SIZE, brush);
	}
      }
    }
  }
  if (highlight_enabled && highlight.isOnBoard()) 
  {
    QPoint point = positionToPoint(highlight);
    QBrush brush(QColor(255, 230, 200));
    painter.fillRect(point.x(), point.y(), BOX_SIZE, BOX_SIZE, brush);
  }
  painter.end();
  AbstractBoard::paintEvent(pe);
}
