/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*---------------------------------------------------------------------------

   manipiv - manipulates (flipping / rotating, etc) PIV data

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

-----------------------------------------------------------------------

  Flipx; flips data left - right
  Flipy; flips data up - down


------------------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <gpiv.h>

/* #define PARFILE "manipiv.par" */	/* Parameter file name */
#define PARFILE "gpivrc"	/* Parameter file name */
#define USAGE "\
Usage: manipiv | fasty | flipx | flipy | revert | rot90 | rot180 \n\
               [-f filename] [-fi x0 y0 x1 y1] [-no_fi | --pass x0 y0 x1 y1] \n\
               [-fy] [-h | --help] [-i] [-p | --print] [-r] [--rev] \n\
               [-v | --version] [-x] [-y] < stdin > stdout \n\
\n\
keys: \n\
-f filename:           files (without .piv extension) instead of stdin and \n\
                       stdout\n\
-fi x0 y0 x1 y1:       filters out all data from (x0,y0) to (x1,y1) \n\
-no_fi | --pass x0 y0 x1 y1: passes through the data from (x0,y0) to (x1,y1) \n\
-fy:                   fast y; returns fast running y-positions of data with \n\
                       fast running x\n\
-h | --help:           this on-line help \n\
-i:                    interchanges indexes of output data for increasing \n\
                       order \n\
-p | --print:          prints parameters to stdout \n\
-r:                    rotates over 90 degrees \n\
--rev:                 reverts array indexes of output data for getting \n\
                       reversed order \n\
-v | --version:        version number \n\
-x:                    flips data in x-direction (vertical axis) \n\
-y:                    flips data in y-direction (horizontal axis) \n\
"

#ifdef DEBUG
#define USAGE_DEBUG "\
Developers version also contains:  \n\
               [-p_main][-p_median_residu][-p_subst_residu] \n\
               [-p_check_residu][-p_residu_statistic] \n\
keys: \n\
-p_'function' N: prints data to be generated in the function; the \n\
                 higher N, the more detailed the output. \n\
                 For N = 10, err_vec will exit at the end of the function"
#endif


#define HELP  "\
manipulates order and positions of PIV data, flips in horizontal \n\
or vertical direction and rotates over 90 or 180 degrees"

#define RCSID "$Id: manipiv.c,v 2.8 2006/01/31 14:18:04 gerber Exp $"

/*
 * Global variables 
*/
int fname_logic = 0;
int print_par = 0;
GpivPivData in_data, out_data;

#ifdef DEBUG
/*
 * Variables for development version 
 */
int print_main = 0, print_flip = 0, print_fasty = 0, print_fread_pivdata =
    0;
#endif



void 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS],
             GpivPostPar * piv_post_par
             )
/* ----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c;
    int argc_next;
    while (--argc > 0 && (*++argv)[0] == '-') {
	argc_next = 0;

/*
 * argc_next is set to 1 if the next cmd line argument has to be searched for; 
 * in case that the command line argument concerns more than one char or cmd
 * line argument needs a parameter
 */
	while (argc_next == 0 && (c = *++argv[0])) {
	    switch (c) {
	    case 'v':
/*
 * Use Revision Control System (RCS) for version
 */
		printf("\n%s\n", RCSID);
		exit(0);
		break;
	    case 'h':
		printf("\n%s", RCSID);
		printf("\n%s", HELP);
		printf("\n%s", USAGE);
#ifdef DEBUG
		printf("\n%s\n", USAGE_DEBUG);
#endif
		exit(0);
		break;
	    case 'r':
		    piv_post_par->operator_manipiv = GPIV_ROT90;
		    piv_post_par->operator_manipiv_logic = 1;
		break;
	    case 'x':
		piv_post_par->operator_manipiv = GPIV_FLIP_X;
		piv_post_par->operator_manipiv_logic = 1;
		break;
	    case 'y':
		piv_post_par->operator_manipiv = GPIV_FLIP_Y;
		piv_post_par->operator_manipiv_logic = 1;
		break;
	    case 'f':
		if (strcmp("fy", *argv) != 0 && strcmp("fi", *argv) != 0) {
		    strcpy(fname, *++argv);
		    fname_logic = 1;
		    argc_next = 1;
		    --argc;
		} else if (strcmp("fi", *argv) == 0) {
/* 
 * filters block of data from data stream
 */
		    piv_post_par->operator_manipiv = GPIV_FILTER_BLOCK;
		    piv_post_par->operator_manipiv_logic = 1;
		    block.x_1 = atoi(*++argv);
		    --argc;
		    block.y_1 = atoi(*++argv);
		    --argc;
		    block.x_2 = atoi(*++argv);
		    --argc;
		    block.y_2 = atoi(*++argv);
		    --argc;
		    argc_next = 1;
		} else if (strcmp("fy", *argv) == 0) {
		    piv_post_par->operator_manipiv = GPIV_FAST_Y;
		    piv_post_par->operator_manipiv_logic = 1;
		}
		break;
	    case 'p':
#ifdef DEBUG
	      if (
		       (strcmp(*argv, "p_main") != 0) &&
		       (strcmp(*argv, "p_flip") != 0) &&
		       (strcmp(*argv, "p_fasty") != 0)
		    ) {
#endif /* DEBUG */
		    print_par = 1;
#ifdef DEBUG
		} else if (strcmp("p_main", *argv) == 0) {
		    print_main = atoi(*++argv);
		    --argc;
		    argc_next = 1;
		} else if (strcmp("p_flip", *argv) == 0) {
		    print_flip = atoi(*++argv);
		    --argc;
		    argc_next = 1;
		} else if (strcmp("p_fasty", *argv) == 0) {
		    print_fasty = atoi(*++argv);
		    --argc;
		    argc_next = 1;
		} else if (strcmp("p_fread_pivdata", *argv) == 0) {
		    print_fread_pivdata = atoi(*++argv);
		    --argc;
		    argc_next = 1;
		}
#endif /* DEBUG */
		break;
/*
 * negotion of settings
 */
	    case 'n':
                if (strcmp(*argv, "no_fi") == 0) {
                    piv_post_par->operator_manipiv = GPIV_PASS_BLOCK;
                    piv_post_par->operator_manipiv_logic = 1;
                    block.x_1 = atoi(*++argv);
                    --argc;
                    block.y_1 = atoi(*++argv);
                    --argc;
                    block.x_2 = atoi(*++argv);
                    --argc;
                    block.y_2 = atoi(*++argv);
                    --argc;
                    argc_next = 1;
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
                break;

/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = 1;
		} else if (strcmp("-rev", *argv) == 0) {
/*
 * reverts indexes of output data for increasing order
 */
		    piv_post_par->operator_manipiv = GPIV_REVERT;
		    piv_post_par->operator_manipiv_logic = 1;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                } else if (strcmp("-pass", *argv) == 0) {
                    piv_post_par->operator_manipiv = GPIV_PASS_BLOCK;
                    piv_post_par->operator_manipiv_logic = 1;
                    block.x_1 = atoi(*++argv);
                    --argc;
                    block.y_1 = atoi(*++argv);
                    --argc;
                    block.x_2 = atoi(*++argv);
                    --argc;
                    block.y_2 = atoi(*++argv);
                    --argc;
                    argc_next = 1;
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

	    default:
		gpiv_error("%s: unknown option: %s", RCSID, *argv);
		break;
	    }
	}
    }

    if (argc != 0) {
	gpiv_error("%s: unknown argument: %s", RCSID, *argv);
    }


}



int
make_fname(char *fname, 
           char *fname_in_piv, 
           char *fname_parameter,
           char *fname_out
           )
/*-----------------------------------------------------------------------------
 * generate filenames
 */
{

    if (fname_logic == 0) {
	gpiv_error("%s: Filename has to be set", RCSID);
    }
    gpiv_io_make_fname(fname, GPIV_EXT_PAR, fname_parameter);
    if (print_par)
	printf("# Data parameter file: %s\n", fname_parameter);

    gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_in_piv);
    if (print_par)
	printf("# Input data file: %s\n", fname_in_piv);

    gpiv_io_make_fname(fname, GPIV_EXT_MANI, fname_out);
    if (print_par)
	printf("# Output file: %s\n", fname_out);

    return 0;
}



int 
main(int argc, 
     char *argv[]
     )
/*-----------------------------------------------------------------------------
*/
{
    char *err_msg = NULL;
    FILE *fp_par_dat;
    char fname[GPIV_MAX_CHARS], fname_out[GPIV_MAX_CHARS],
	fname_parameter[GPIV_MAX_CHARS], fname_in_piv[GPIV_MAX_CHARS];
    char d_line[GPIV_MAX_LINES][GPIV_MAX_CHARS], 
        c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    char return_val, *c = NULL;
    int nd_lines = 0, nc_lines = 0, scale = 0;

    GpivPostPar piv_post_par, piv_post_par_default;

    gpiv_post_parameters_logic(&piv_post_par, FALSE);
    gpiv_post_default_parameters(&piv_post_par_default, TRUE);
/*     piv_post_par.operator_manipiv_logic = 0; */

/*
 * Define GpivOperation type from program name, which is a symbolic link to manipiv
 */
    if ((c = strstr(argv[0], "manipiv")) != NULL) {
	piv_post_par.operator_manipiv = GPIV_FLIP_X;
	piv_post_par.operator_manipiv_logic = 0;
    } else if ((c = strstr(argv[0], "flipx")) != NULL) {
	piv_post_par.operator_manipiv = GPIV_FLIP_X;
	piv_post_par.operator_manipiv_logic = 1;
    } else if ((c = strstr(argv[0], "flipy")) != NULL) {
	piv_post_par.operator_manipiv = GPIV_FLIP_Y;
	piv_post_par.operator_manipiv_logic = 1;
    } else if ((c = strstr(argv[0], "rot180")) != NULL) {
	piv_post_par.operator_manipiv = GPIV_ROT180;
	piv_post_par.operator_manipiv_logic = 1;
    } else if ((c = strstr(argv[0], "rot90")) != NULL) {
	piv_post_par.operator_manipiv = GPIV_ROT90;
	piv_post_par.operator_manipiv_logic = 1;
    } else if ((c = strstr(argv[0], "revert")) != NULL) {
	piv_post_par.operator_manipiv = GPIV_REVERT;
	piv_post_par.operator_manipiv_logic = 1;
    } else if ((c = strstr(argv[0], "fasty")) != NULL) {
	piv_post_par.operator_manipiv = GPIV_FAST_Y;
	piv_post_par.operator_manipiv_logic = 1;
    } else {
        gpiv_error("manipiv: unvalid program name or symlink");
    }

    command_args(argc, argv, fname, &piv_post_par);
    if (print_par) {
      printf("# %s\n# Command line options:\n", RCSID);
      gpiv_post_print_parameters(piv_post_par);
    }



    if (fname_logic == 1) {
	if ((return_val = make_fname(fname, fname_in_piv, fname_parameter,
				     fname_out)) != 0) {
	    gpiv_error("%s: Failure calling make_fname",
		    RCSID);
	}


/*
 * Prints command line parameters to par-file
 */
	if ((fp_par_dat = fopen(fname_parameter, "a")) == NULL) {
	    gpiv_error("%s: failure opening %s for input",
		    RCSID, fname_parameter);
	}
	fprintf(fp_par_dat, "\n\n# %s\n# Command line options:\n", RCSID);
	gpiv_post_fprint_parameters(fp_par_dat, piv_post_par);


/*
 * Reading parametes from PARFILE (and writing to data par-file)
 */
        gpiv_scan_parameter(GPIV_POST_PAR_KEY, PARFILE, &piv_post_par, print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_POST_PAR_KEY, &piv_post_par, print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
        gpiv_post_fprint_parameters(fp_par_dat, piv_post_par);
	fclose(fp_par_dat);


    } else {
        gpiv_scan_parameter(GPIV_POST_PAR_KEY, PARFILE, &piv_post_par, print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_POST_PAR_KEY, &piv_post_par, print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
    }

    gpiv_post_check_parameters_read(&piv_post_par, piv_post_par_default);

/*
 * As in_data.nx or in_data.ny are not known, the input data file will first 
 * be read
 */

    if (fname_logic == 1) {
	if ((scale = gpiv_fcount_pivdata(fname_in_piv, &in_data)) == -1) {
	    gpiv_error("%s: Failure calling gpiv_count_pivdata",
		    RCSID);
	}
    } else {
	if ((scale =
	     gpiv_count_pivdata(&in_data, d_line, &nd_lines)) == -1) {
	    gpiv_error("%s: Failure calling gpiv_count_pivdata",
		       RCSID);
	}
    }
    scale = 1;

/*
 * Now the parameters are known, data  memory can be allocated
 */
    gpiv_alloc_pivdata(&in_data);
    if (piv_post_par.operator_manipiv == GPIV_ROT90) {
	out_data.ny = in_data.nx;
	out_data.nx = in_data.ny;
    } else {
	out_data.ny = in_data.ny;
	out_data.nx = in_data.nx;
    }
    gpiv_alloc_pivdata(&out_data);

/*
 * Here the function calls of the post-processing; flipping x and y data etc.
 * Inclusive reading of the PIV data its own functions.
 */

    if ((err_msg = gpiv_post_manipiv(fname_in_piv, fname_logic, 
					in_data, &out_data,
					piv_post_par, d_line, nd_lines, c_line,
					&nc_lines))
	!= NULL) gpiv_error ("%s: ", RCSID, err_msg);


/*
 * And writing to output
 */
    if (fname_logic == 1) {
	if ((err_msg = 
             gpiv_fwrite_pivdata(fname_out, &out_data, c_line,
                                 nc_lines, scale, RCSID)) 
            != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
    } else {
	gpiv_write_pivdata(&out_data, c_line, nc_lines, scale, RCSID);
    }


/*
 * Freeing allocated memory of matrices
 */
    gpiv_free_pivdata(&in_data);
    gpiv_free_pivdata(&out_data);
    if (print_par == 1)
	printf("\n");

    return 0;
}







