/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*---------------------------------------------------------------------------

   piv2gnuplt - converts PIV data gnuplot data

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

------------------------------------------------------------------------*/

#include <stdio.h> 
#include <stdlib.h> 
#include <gpiv.h>

/* #define PARFILE "scale.par"    */  /* Parameter file name */
#define PARFILE "gpivrc"     /* Parameter file name */
#define USAGE "\
Usage: piv2gnuplt [-al L] [-au UNIT] [-h | --help] [-p | --print] [-s SCALE] \n\
                  [-v] filename \n\
\n\
keys: \n\
-al L:                 draws an annotation vector with lenght L \n\
-au UNIT:              define unit of annotation vector (defaul m/s) \n\
-h | --help:           this on-line help \n\
-p | --print:          print parameters to stdout \n\
-s SCALE:              scale factor for graphic output with gnuplot \n\
-v | --version:        version number \n\
-z                     full zoom (for scaled data) \n\
"

#define HELP  "\
piv2gnuplt - converts PIV data gnuplot data"

#define RCSID "$Id: piv2gnuplt.c,v 1.8 2006/01/31 14:18:04 gerber Exp $"
#define GNUPLOT_DISPLAY_SIZE 500
#define GNUPLOT_DISPLAY_COLOR "LightBlue"

gboolean print_par = FALSE, zoom = FALSE;
gboolean scale_logic = FALSE, annot_logic = FALSE;
float scale = 1.0, annot_length = 1.0;
char annot_unit[GPIV_MAX_CHARS] = "m/s";

/*
TODO:
Annotation; length, unit
Full Zoom
*/

void 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS]
             )
/*-----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c = '\0';
    int argc_next = 0;


    while (--argc > 0 && (*++argv)[0] == '-') {
        argc_next = 0;
/*
 * argc_next is set to 1 if the next cmd line argument has to be searched for; 
 * in case that the command line argument concerns more than one char or cmd 
 * line argument needs a parameter 
 */

	while (argc_next == 0 && (c = *++argv[0]))
            switch (c) {
            case 'a':
		if (strcmp(*argv, "al") != 0) {
                    annot_length = atof(*++argv);
                } else if (strcmp(*argv, "au") != 0) {
		    strcpy(annot_unit, *++argv);
/* 		    annot_logic = TRUE; */
		    argc_next = 1;
		    --argc;
                } else {
                    gpiv_error("%s: %s", RCSID, USAGE);
                }
                annot_logic = TRUE;
                break;
            case 'v':
                printf("%s\n", RCSID); 
                exit(0);
                break;
             case 'h':
                printf("%s\n", RCSID); 
                printf("%s\n",HELP);
                printf("%s\n",USAGE);
                exit(0);
                break;
            case 'p':
                print_par = TRUE;
                break;
            case 's':	       
                scale = atof(*++argv);
                scale_logic = TRUE;
                --argc;
                argc_next = 1;
                break;
            case 'z':
                zoom = TRUE;
                break;

/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

            default:
                fprintf (stderr,USAGE);
                exit(1);
                break;
            }
    }

     if(argc != 1) { 
	  gpiv_error("%s: %s", RCSID, USAGE);
     }
     strcpy(fname, argv[0]); 


}


void 
make_fname(char *fname, 
           char *fname_header, 
           char *fname_in, 
           char *fname_out
           )
/*-----------------------------------------------------------------------------
 * function to generate filenames
 */
{
     gpiv_io_make_fname(fname, GPIV_EXT_HEADER, fname_header);
     printf("# Header data file: %s\n",fname_header);
     
     gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_in);
     printf("# Input data file: %s\n",fname_in);
     
     gpiv_io_make_fname(fname, ".piv.hd5", fname_out);
     printf("# Output file: %s\n",fname_out);
     
}



int 
main(int argc, 
     char *argv[]
     )
/*-----------------------------------------------------------------------------
 * main program to convert PIV data gnuplot data
 */
{
    char *err_msg = NULL;
    FILE *fp_par_out = NULL;
    char fname[GPIV_MAX_CHARS], 
        fname_header[GPIV_MAX_CHARS], 
        fname_out[GPIV_MAX_CHARS], 
        fname_in[GPIV_MAX_CHARS],
        fname_parameter[GPIV_MAX_CHARS];
    char  c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    int nc_lines = 0, var_scale = 0;
    GpivPivData piv_data;
    GpivImagePar image_par;
    GpivEvalPar piv_eval_par, piv_eval_par_default;
    
    piv_data.nx = 0;
    piv_data.ny = 0;
    piv_data.point_x = NULL;
    piv_data.point_y = NULL;
    piv_data.dx = NULL;
    piv_data.dy = NULL;
    piv_data.snr = NULL;
    piv_data.peak_no = NULL;


/* 
 *Initializing parameters
 * General
 */
    piv_eval_par.print_piv = 0;
    piv_eval_par.print_cov_logic = FALSE;

/*
 * Image
 */
    image_par.ncolumns_logic = FALSE;
    image_par.nrows_logic = FALSE;
    image_par.depth_logic = FALSE;
    image_par.x_corr_logic = FALSE;

    gpiv_piv_parameters_logic(&piv_eval_par, FALSE);
    gpiv_piv_default_parameters(&piv_eval_par_default, TRUE);



    command_args(argc, argv, fname);
    if (print_par) {
	 printf("\n# %s\n# Command line options:\n", RCSID);
	 gpiv_img_print_header(image_par);
	 gpiv_piv_print_parameters(piv_eval_par);
    }
    make_fname(fname, fname_header, fname_in, fname_out);
	  
    gpiv_io_make_fname(fname, GPIV_EXT_PAR, fname_parameter);
    if (print_par)
        printf("\n# Parameters written to: %s", fname_parameter);
    if ((fp_par_out = fopen(fname_parameter, "w")) == NULL) {
        gpiv_error("%s error: failure opening %s for input",
                   RCSID, fname_parameter);
/*
 * Write command-line args to parameterfile if -f has been used 
 */
        fprintf(fp_par_out, "\n# %s\n# Command line options:\n", RCSID);
        gpiv_img_fprint_header(fp_par_out, image_par);
        gpiv_piv_fprint_parameters(fp_par_out, piv_eval_par);
    }


/*
 * Reads image header data from file.h and/or resource files 
 * if not overridden by the commandline options
 */
    gpiv_scan_parameter( "", fname_header, &image_par, print_par);
    gpiv_scan_parameter( "", PARFILE, &image_par, print_par);
    if ((err_msg =
         gpiv_scan_resourcefiles("", &image_par, print_par))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

    gpiv_img_fprint_header(fp_par_out, image_par);
    

/*
 * Reads piv program parameters from PARFILE and/or resource files 
 * if not overridden by the commandline options
 */
    gpiv_scan_parameter(GPIV_EVAL_PAR_KEY, PARFILE, &piv_eval_par, print_par);
    if ((err_msg =
         gpiv_scan_resourcefiles(GPIV_EVAL_PAR_KEY, &piv_eval_par, print_par))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

    gpiv_piv_fprint_parameters(fp_par_out, piv_eval_par);
    

/*
 * Check if all parameters have been read
 */
/*     gpiv_img_check_header_read(GPIV_IMAGE_PAR_KEY); */
    if ((err_msg =
         gpiv_piv_check_parameters_read(&piv_eval_par, piv_eval_par_default))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);


/*
 * As piv_data.nx and piv_data.ny are not known, the input data file will 
 * first be read
 */
         printf("AA:: nx = %d ny = %d\n", piv_data.nx, piv_data.ny);
     if ((var_scale = gpiv_fcount_pivdata(fname_in, &piv_data)) == -1) {
         gpiv_error("%s: Failure calling gpiv_count_pivdata", RCSID);
     } else if (print_par) {
         printf("nx = %d ny = %d\n", piv_data.nx, piv_data.ny);
     }
     gpiv_alloc_pivdata(&piv_data);
    if ((err_msg =
         gpiv_fread_pivdata(fname_in, &piv_data, c_line, &nc_lines))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

/* 
 * plotting with gnuplot
 */
     gpiv_piv_gnuplot(fname, fname, scale, 
/*                       zoom, annot_length, annot_unit, */
                      GNUPLOT_DISPLAY_COLOR, GNUPLOT_DISPLAY_SIZE,
                      image_par, piv_eval_par, piv_data, RCSID);

/*
 * Freeing allocated memory of matrices
 */
     gpiv_free_pivdata(&piv_data);
     exit(0);
}






