/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*-----------------------------------------------------------------------------

   dav2piv - converts hdf5 PIV data ASCII data

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

-----------------------------------------------------------------------------*/

#include <stdio.h> 
#include <stdlib.h> 
#include <assert.h> 
#include <gpiv.h>

/* #define PARFILE "scale.par"    */  /* Parameter file name */
#define PARFILE "gpivrc"     /* Parameter file name */
#define USAGE "\
Usage: dav2piv [-h | --help] [-p | --print] [-v | --version] filename \n\
\n\
keys: \n\
-h | --help:           this on-line help \n\
-p | --print:          print parameters to stdout \n\
-v | --version:        version number \n\
"

#define HELP  "\
dav2piv - converts LaVision's DaVis (tm) image to raw data image"

#define RCSID "$Id: dav2piv.c,v 1.4 2006/01/31 14:18:04 gerber Exp $"

gboolean print_par = FALSE;

void 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS]
             )
/* ----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c = '\0';
    int argc_next;

    while (--argc > 0 && (*++argv)[0] == '-') {

/*
 * argc_next is set to 1 if the next cmd line argument has to be searched for; 
 * in case that the command line argument concerns more than one char or cmd 
 * line argument needs a parameter 
 */

        argc_next = 0;
	while (argc_next == 0 && (c = *++argv[0]))
            
            switch (c) {
            case 'v':
                printf("%s\n", RCSID); 
                exit(0);
                break;
            case 'h':
                printf("%s\n", RCSID); 
                printf("%s\n",HELP);
                printf("%s\n",USAGE);
                exit(0);
                break;
            case 'p':
                print_par = TRUE;
                break;

/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

            default:
                gpiv_error(USAGE);
                break;
            }
    }

    if(argc != 1) { 
        gpiv_error("%s: %s", RCSID, USAGE);
    }

    strcpy(fname, argv[0]);
}



void 
make_fname(char *fname, 
           char *fname_in, 
           char *fname_out_img,
           char *fname_out_header
           )
/* ----------------------------------------------------------------------------
 * define filenames
 */
{     
     gpiv_io_make_fname(fname, GPIV_EXT_DAVIS, fname_in);
     if (print_par) printf("#Input file: %s\n",fname_in);

     gpiv_io_make_fname(fname, GPIV_EXT_RAW_IMAGE, fname_out_img);
     if (print_par) printf("#Output data file: %s\n",fname_out_img);
     
     gpiv_io_make_fname(fname, GPIV_EXT_HEADER, fname_out_header);
     if (print_par) printf("#Output header file: %s\n",fname_out_header);
         
}



int 
main(int argc, 
     char *argv[]
     )
/* ----------------------------------------------------------------------------
 */
{
    FILE *fp_out = NULL;
/*     char *err_msg = NULL; */
    char fname[GPIV_MAX_CHARS], 
        fname_in[GPIV_MAX_CHARS],
        fname_out_img[GPIV_MAX_CHARS], 
        fname_out_header[GPIV_MAX_CHARS];
    guint16 **img1 = NULL, **img2 = NULL;

    GpivImagePar image_par;

/* 
 * Initializing of the parameters 
 * Image parametes
 */
    gpiv_img_parameters_logic(&image_par, FALSE);
    command_args(argc, argv, fname);
    make_fname(fname, fname_in, fname_out_img, fname_out_header);
	  
/* 
 * reading and writing image data and parameters
 */
    gpiv_img_fread_davis_parameters(fname_in, &image_par);
        
    img1 = gpiv_alloc_img(image_par);
    img2 = gpiv_alloc_img(image_par);
        
    gpiv_fread_davis_image(fname_in, img1, img2, &image_par);
    gpiv_fwrite_image (fname_out_img, img1, img2, image_par);
    gpiv_free_img(img1, image_par);
    gpiv_free_img(img2, image_par);

    if ((fp_out = fopen(fname_out_header, "w")) == NULL) {
        gpiv_error("%s error: failure opening %s for output",
                   RCSID, fname_out_header);
    }
    gpiv_img_fprint_header(fp_out, image_par);
    fclose(fp_out);

    if (print_par) {
        gpiv_img_print_parameters(image_par);
    }


    exit(0);
}






