/*
     This file is part of GNUnet.
     (C) 2005, 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/meta.c
 * @brief code for dealing with meta and keyword tree views
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include "helper.h"
#include "meta.h"
#include "fs.h"
#include <extractor.h>

static int
publishMetaData (EXTRACTOR_KeywordType type, const char *data, void *cls)
{
  GtkListStore *store = cls;
  const char *stype;
  GtkTreeIter iter;

  stype = EXTRACTOR_getKeywordTypeAsString (type);
  if (stype == NULL)            /* wrong LE version!? */
    stype = _("unknown");
  gtk_list_store_append (store, &iter);
  gtk_list_store_set (store,
                      &iter,
                      META_TYPE, type,
                      META_STYPE, stype, META_VALUE, data, -1);
  return OK;
}

void
createMetaDataListTreeView (GladeXML * xml,
                            const char *name,
                            const char *previewName,
                            const struct ECRS_MetaData *init)
{
  GtkTreeView *metaList;
  GtkListStore *metamodel;
  GtkCellRenderer *renderer;
  GtkWidget *preview;
  GdkPixbuf *pixbuf;
  GdkPixbufLoader *loader;
  unsigned char *thumb;
  size_t ts;
  GtkTreeViewColumn *column;
  int col;

  DEBUG_BEGIN ();
  metamodel
    = gtk_list_store_new (META_NUM, G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING);
  metaList = GTK_TREE_VIEW (glade_xml_get_widget (xml, name));
  gtk_tree_selection_set_mode (gtk_tree_view_get_selection (metaList),
                               GTK_SELECTION_MULTIPLE);
  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_insert_column_with_attributes (metaList,
                                                     -1,
                                                     _("Category"),
                                                     renderer,
                                                     "text", META_STYPE,
                                                     NULL);
  column = gtk_tree_view_get_column (metaList, col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, META_STYPE);
  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_insert_column_with_attributes (metaList,
                                                     -1,
                                                     _("Value"),
                                                     renderer,
                                                     "text", META_VALUE,
                                                     NULL);
  column = gtk_tree_view_get_column (metaList, col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, META_VALUE);
  gtk_tree_view_set_model (metaList, GTK_TREE_MODEL (metamodel));

  if (init != NULL)
    ECRS_getMetaData (init, &publishMetaData, metamodel);
  if ((previewName == NULL) || (init == NULL))
    return;
  preview = glade_xml_get_widget (xml, previewName);

  thumb = NULL;
  ts = ECRS_getThumbnailFromMetaData (init, &thumb);
  preview = glade_xml_get_widget (xml, previewName);
  if (ts != 0)
    {
      loader = gdk_pixbuf_loader_new ();
      gdk_pixbuf_loader_write (loader, (const guchar *) thumb, ts, NULL);
      pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
      g_object_ref (pixbuf);
      gdk_pixbuf_loader_close (loader, NULL);
      if (pixbuf != NULL)
        {
          gtk_image_set_from_pixbuf (GTK_IMAGE (preview), pixbuf);
          g_object_unref (pixbuf);
        }
      UNREF (loader);
    }
  DEBUG_END ();
}

static int
publishKeyword (const char *data, void *cls)
{
  GtkListStore *store = cls;
  GtkTreeIter iter;

  DEBUG_BEGIN ();
  gtk_list_store_append (store, &iter);
  gtk_list_store_set (store, &iter, 0, data, -1);
  DEBUG_END ();
  return OK;
}

void
createKeywordListTreeView (GladeXML * xml,
                           const char *name, const struct ECRS_URI *init)
{
  GtkTreeView *keywordList;
  GtkListStore *keymodel;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  DEBUG_BEGIN ();
  keymodel = gtk_list_store_new (1, G_TYPE_STRING);
  keywordList = GTK_TREE_VIEW (glade_xml_get_widget (xml, name));
  gtk_tree_selection_set_mode (gtk_tree_view_get_selection (keywordList),
                               GTK_SELECTION_MULTIPLE);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_insert_column_with_attributes (keywordList,
                                               -1,
                                               _("Keyword"),
                                               renderer, "text", 0, NULL);
  column = gtk_tree_view_get_column (keywordList, 0);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, 0);
  gtk_tree_view_set_model (keywordList, GTK_TREE_MODEL (keymodel));
  if (init != NULL)
    ECRS_getKeywordsFromUri (init, &publishKeyword, keymodel);
  DEBUG_END ();
}

void
createMetaTypeComboBox (GladeXML * xml, const char *name)
{
  const char *stype;
  GtkWidget *metaType;
  GtkCellRenderer *renderer;
  GtkListStore *keywordTypeModel;
  GtkTreeIter iter;
  EXTRACTOR_KeywordType type;

  DEBUG_BEGIN ();
  keywordTypeModel
    = gtk_list_store_new (KTYPE_NUM, G_TYPE_STRING, G_TYPE_INT);
  metaType = glade_xml_get_widget (xml, name);
  gtk_combo_box_set_model (GTK_COMBO_BOX (metaType),
                           GTK_TREE_MODEL (keywordTypeModel));
  for (type = 0; type < EXTRACTOR_getHighestKeywordTypeNumber (); type++)
    {
      stype = EXTRACTOR_getKeywordTypeAsString (type);
      gtk_list_store_append (keywordTypeModel, &iter);
      gtk_list_store_set (keywordTypeModel,
                          &iter, KTYPE_STRING, stype, KTYPE_TYPE, type, -1);
    }
  renderer = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (metaType), renderer, FALSE);
  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (metaType),
                                 renderer, "text", KTYPE_STRING);
  gtk_combo_box_set_active (GTK_COMBO_BOX (metaType), 0);
  DEBUG_END ();
}

void
handleKeywordListUpdate (GladeXML * xml,
                         const char *inputLineName,
                         const char *keywordListName)
{
  const char *keyword;
  GtkWidget *keywordList;
  GtkWidget *entryLine;
  GtkListStore *keymodel;
  GtkTreeIter iter;

  DEBUG_BEGIN ();
  keywordList = glade_xml_get_widget (xml, keywordListName);
  keymodel
    = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (keywordList)));

  entryLine = glade_xml_get_widget (xml, inputLineName);
  keyword = gtk_entry_get_text (GTK_ENTRY (entryLine));
  gtk_list_store_append (keymodel, &iter);
  gtk_list_store_set (keymodel, &iter, 0, keyword, -1);
  gtk_entry_set_text (GTK_ENTRY (entryLine), "");
  DEBUG_END ();
}

void
handleMetaDataListUpdate (GladeXML * xml,
                          const char *typeInputLineName,
                          const char *valueInputLineName,
                          const char *metaDataListName)
{
  const char *value;
  GtkWidget *metaList;
  GtkWidget *entryLine;
  GtkWidget *typeCB;
  EXTRACTOR_KeywordType type;
  GtkListStore *metamodel;
  GtkListStore *typemodel;
  GtkTreeIter iter;
  char *stype;

  DEBUG_BEGIN ();
  metaList = glade_xml_get_widget (xml, metaDataListName);
  metamodel
    = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (metaList)));

  entryLine = glade_xml_get_widget (xml, valueInputLineName);
  value = gtk_entry_get_text (GTK_ENTRY (entryLine));
  if ((value == NULL) || (strlen (value) == 0))
    return;
  typeCB = glade_xml_get_widget (xml, typeInputLineName);
  typemodel
    = GTK_LIST_STORE (gtk_combo_box_get_model (GTK_COMBO_BOX (typeCB)));
  if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (typeCB), &iter))
    return;                     /* oops */
  gtk_tree_model_get (GTK_TREE_MODEL (typemodel),
                      &iter, KTYPE_STRING, &stype, KTYPE_TYPE, &type, -1);
  gtk_list_store_append (metamodel, &iter);
  gtk_list_store_set (metamodel,
                      &iter,
                      META_TYPE, type,
                      META_STYPE, stype, META_VALUE, value, -1);
  gtk_entry_set_text (GTK_ENTRY (entryLine), "");
  FREENONNULL (stype);
  DEBUG_END ();
}

static void
removeRow (GtkTreeModel * model,
           GtkTreePath * path, GtkTreeIter * iter, gpointer unused)
{
  gtk_list_store_remove (GTK_LIST_STORE (model), iter);
}

void
handleListRemove (GladeXML * xml, const char *listName)
{
  GtkWidget *list;

  list = glade_xml_get_widget (xml, listName);
  ggc_tree_selection_selected_foreach
    (gtk_tree_view_get_selection (GTK_TREE_VIEW (list)), &removeRow, NULL);
}

#define THUMBSIZE 128


struct ECRS_MetaData *
getMetaDataFromList (GladeXML * xml,
                     const char *name, const char *previewName)
{
  GtkWidget *metaList;
  GtkWidget *preview;
  GtkTreeModel *metamodel;
  GtkTreeIter iter;
  struct ECRS_MetaData *meta;
  EXTRACTOR_KeywordType type;
  GdkPixbuf *pixbuf;
  GdkPixbuf *out;
  char *mvalue;
  size_t length;
  unsigned long width;
  unsigned long height;
  char *thumb;

  metaList = glade_xml_get_widget (xml, name);
  metamodel = gtk_tree_view_get_model (GTK_TREE_VIEW (metaList));
  meta = ECRS_createMetaData ();
  if (gtk_tree_model_get_iter_first (metamodel, &iter))
    {
      do
        {
          gtk_tree_model_get (metamodel,
                              &iter,
                              META_TYPE, &type, META_VALUE, &mvalue, -1);
          ECRS_addToMetaData (meta, type, mvalue);
          FREENONNULL (mvalue);
        }
      while (gtk_tree_model_iter_next (metamodel, &iter));
    }
  if (previewName == NULL)
    return meta;

  preview = glade_xml_get_widget (xml, previewName);

  if (GTK_IMAGE_PIXBUF == gtk_image_get_storage_type (GTK_IMAGE (preview)))
    {
      pixbuf = gtk_image_get_pixbuf (GTK_IMAGE (preview));
      if (pixbuf == NULL)
        return meta;
      height = gdk_pixbuf_get_height (pixbuf);
      width = gdk_pixbuf_get_width (pixbuf);
      if ((height > THUMBSIZE) || (width > THUMBSIZE))
        {
          if (height > THUMBSIZE)
            {
              width = width * THUMBSIZE / height;
              height = THUMBSIZE;
            }
          if (width > THUMBSIZE)
            {
              height = height * THUMBSIZE / width;
              width = THUMBSIZE;
            }
          out = gdk_pixbuf_scale_simple (pixbuf,
                                         width, height, GDK_INTERP_BILINEAR);
          g_object_unref (pixbuf);
          pixbuf = out;
        }
      if (pixbuf != NULL)
        {
          thumb = NULL;
          if (gdk_pixbuf_save_to_buffer (pixbuf,
                                         &thumb, &length, "png", NULL, NULL))
            {
              char *binary;

              binary = EXTRACTOR_binaryEncode ((const unsigned char *) thumb,
                                               length);
              free (thumb);
              ECRS_addToMetaData (meta, EXTRACTOR_THUMBNAIL_DATA, binary);
              FREE (binary);
            }
        }
    }
  return meta;
}

struct ECRS_URI *
getKeywordURIFromList (GladeXML * xml, const char *name)
{
  GtkTreeModel *keymodel;
  GtkTreeView *keyList;
  GtkTreeIter iter;
  struct ECRS_URI *keywordURI;
  char *mvalue;
  char **keywords;
  unsigned int kpos;
  unsigned int ksize;

  DEBUG_BEGIN ();
  keyList = GTK_TREE_VIEW (glade_xml_get_widget (xml, name));
  keymodel = gtk_tree_view_get_model (keyList);

  keywords = NULL;
  ksize = 0;
  GROW (keywords, ksize, 64);
  kpos = 0;
  if (gtk_tree_model_get_iter_first (keymodel, &iter))
    {
      do
        {
          gtk_tree_model_get (keymodel, &iter, 0, &mvalue, -1);
          keywords[kpos++] = mvalue;
          if (kpos == ksize)
            GROW (keywords, ksize, kpos * 2);
        }
      while (gtk_tree_model_iter_next (keymodel, &iter));
    }
  keywords[kpos] = NULL;

  keywordURI = ECRS_keywordsToUri ((const char **) keywords);
  while (kpos > 0)
    FREE (keywords[--kpos]);
  GROW (keywords, ksize, 0);
  DEBUG_END ();
  return keywordURI;
}

char *
updateIntervalToString (TIME_T interval)
{
  if (interval == ECRS_SBLOCK_UPDATE_SPORADIC)
    return strdup (_("--sporadic update--"));
  if (interval == ECRS_SBLOCK_UPDATE_NONE)
    return strdup (_("--no update--"));
  return string_get_fancy_time_interval (interval * cronSECONDS);
}

int
parseTimeInterval (const char *timeSpec, TIME_T * interval)
{
  if (timeSpec == NULL)
    return SYSERR;
  if (0 == strcmp (_("--sporadic update--"), timeSpec))
    *interval = ECRS_SBLOCK_UPDATE_SPORADIC;
  else if (0 == strcmp (_("--no update--"), timeSpec))
    *interval = ECRS_SBLOCK_UPDATE_NONE;
  else
    return parseTime (timeSpec, interval);
  return OK;
}

int
tryParseTimeInterval (GladeXML * xml,
                      const char *intervalComboBoxEntry, TIME_T * interval)
{
  return parseTimeInterval (getEntryLineValue (xml,
                                               intervalComboBoxEntry),
                            interval);
}

unsigned int
getSpinButtonValue (GladeXML * xml, const char *spinButtonName)
{
  GtkWidget *spin;

  spin = glade_xml_get_widget (xml, spinButtonName);
  return gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));
}



char *
getMimeTypeFromMetaData (const struct ECRS_MetaData *meta)
{
  char *mime;

  mime = ECRS_getFromMetaData (meta, EXTRACTOR_MIMETYPE);
  if (mime == NULL)
    mime = STRDUP (_("unknown"));
  return validate_utf8 (mime);
}

char *
getFileNameFromMetaData (const struct ECRS_MetaData *meta)
{
  char *name;

  name = ECRS_getFirstFromMetaData (meta,
                                    EXTRACTOR_FILENAME,
                                    EXTRACTOR_TITLE,
                                    EXTRACTOR_ARTIST,
                                    EXTRACTOR_AUTHOR,
                                    EXTRACTOR_PUBLISHER,
                                    EXTRACTOR_CREATOR,
                                    EXTRACTOR_PRODUCER,
                                    EXTRACTOR_UNKNOWN, -1);
  if (name == NULL)
    {
      name = STRDUP (_("no name given"));
    }
  else
    {
      char *dotdot;

      while (NULL != (dotdot = strstr (name, "..")))
        dotdot[0] = dotdot[1] = '_';
    }
  return validate_utf8 (name);
}

char *
getDescriptionFromMetaData (const struct ECRS_MetaData *meta)
{
  char *desc;

  desc = ECRS_getFirstFromMetaData (meta,
                                    EXTRACTOR_DESCRIPTION,
                                    EXTRACTOR_GENRE,
                                    EXTRACTOR_ALBUM,
                                    EXTRACTOR_COMMENT,
                                    EXTRACTOR_SUBJECT,
                                    EXTRACTOR_FORMAT,
                                    EXTRACTOR_SIZE, EXTRACTOR_KEYWORDS, -1);
  if (desc == NULL)
    desc = STRDUP ("");
  return validate_utf8 (desc);
}

GdkPixbuf *
getThumbnailFromMetaData (const struct ECRS_MetaData * meta)
{
  GdkPixbuf *pixbuf;
  GdkPixbufLoader *loader;
  size_t ts;
  unsigned char *thumb;

  thumb = NULL;
  ts = ECRS_getThumbnailFromMetaData (meta, &thumb);
  if (ts != 0)
    {
      loader = gdk_pixbuf_loader_new ();
      gdk_pixbuf_loader_write (loader, (const guchar *) thumb, ts, NULL);
      pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
      gdk_pixbuf_loader_close (loader, NULL);
      if (pixbuf != NULL)
        g_object_ref (pixbuf);
      UNREF (loader);
    }
  else
    {
      pixbuf = NULL;
    }
  FREENONNULL (thumb);
  return pixbuf;
}

GtkWidget *
extractMainWidgetFromWindow (GladeXML * xml, const char *windowName)
{
  GtkContainer *window;
  GtkWidget *ret;

  window = GTK_CONTAINER (glade_xml_get_widget (xml, windowName));
  ret = gtk_bin_get_child (GTK_BIN (window));
  gtk_widget_ref (ret);
  gtk_container_remove (window, ret);
  gtk_widget_destroy (GTK_WIDGET (window));
  return ret;
}

int
getToggleButtonValue (GladeXML * xml, const char *widgetName)
{
  return
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
                                  (glade_xml_get_widget (xml, widgetName)));
}

const char *
getEntryLineValue (GladeXML * xml, const char *widgetName)
{
  GtkBin *line;
  GtkEntry *entry;

  line = GTK_BIN (glade_xml_get_widget (xml, widgetName));
  entry = GTK_ENTRY (gtk_bin_get_child (line));
  return gtk_entry_get_text (entry);
}

/* end of meta.c */
