/* Thin binding for the WTX protocol, for GDB.

   Copyright 2004 Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/* For wtx.h : GDB is run on the host.  */
#define HOST     

#include "wtx.h"

/* Some macros are not defined in WTX 2.0. Define them as constant when
   possible.  */

#ifndef WTX_PROT_VERSION
#define WTX_PROT_VERSION 2
#endif

#ifndef WTX_PD_CURRENT
extern const TGT_ADDR_T WTX_PD_CURRENT;
#endif

#ifndef WTX_PD_ALL
extern const TGT_ADDR_T WTX_PD_ALL;
#endif

#ifndef WTX_MOD_FIND_IN_ALL_PD
extern const TGT_ADDR_T WTX_MOD_FIND_IN_ALL_PD;
#endif

#ifndef WTX_SYM_FIND_IN_ALL_PD
extern const int WTX_SYM_FIND_IN_ALL_PD;
#endif

/* On WTX 3.0, some additional values has been added in the enums to
   deal with the protection domains. Emulate these value in WTX 2.0.  */

#if WTX_PROT_VERSION == 2
extern const WTX_CONTEXT_TYPE WTX_CONTEXT_PD;
extern const WTX_ACTION_TYPE WTX_ACTION_PD_STOP;
extern const int WTX_ERR_PD_INVALID_PD;
#endif

/* The structures defined below are used to resolve differences
   between version of the WTX protocol.

   Note those differences:
   - When the WTX structure contains a BOOL32, the WTX API structure uses an
   int; if it is FALSE in the WTX structure, it is 0 in the WTX API structure;
   - When the WTX structure contains a UINT32, the WTX API uses an int; the
   conversion is straightforward.  */

/* Protection domain ID.  */

typedef TGT_ADDR_T pd_id_t;

/* Module ID.  */

typedef UINT32 module_id_t;

/* Event point id.  */

typedef UINT32 evtpt_id_t;


/* Invalid identifiers. Equivalent to WTX_ERROR.  */

extern const pd_id_t invalid_pd_id;
extern const evtpt_id_t invalid_module_id;
extern const evtpt_id_t invalid_evtpt_id;
extern const WTX_CONTEXT_ID_T invalid_context_id;
extern const WTX_AGENT_MODE_TYPE invalid_agent_mode;
extern const WTX_CONTEXT_STATUS invalid_context_status;

/* Event. Same as WTX_EVENT in WTX 3.0, and WTX_EVENT_2 in WTX
   2.0.  */

struct wtxapi_event
{
  WTX_EVENT_TYPE event_type;
  int num_args;
  TGT_ARG_T *args;
};

/* Action. Based on WTX_ACTION.  */

struct wtxapi_action
{
  /* Action type to perform.  */
  WTX_ACTION_TYPE action_type;
  /* Action dependent argument.  */
  int action_arg;
  /* Function to ACTION_CALL.  */
  TGT_ADDR_T call_rtn;
  /* Function argument.  */
  TGT_ARG_T call_arg;
};

/* Context. Same as WTX_CONTEXT */

struct wtxapi_context
{
  WTX_CONTEXT_TYPE context_type;
  WTX_CONTEXT_ID_T context_id;
};

/* Eventpoint desc. Same as WTX_EVTPT in WTX 3.0, and WTX_EVTPT_2
   in WTX 2.0.  */

struct wtxapi_evtpt
{
  struct wtxapi_event event;
  struct wtxapi_context context;
  struct wtxapi_action action;
};

/* Eventpoint info. Same as WTX_EVTPT_INFO.  */

struct wtxapi_evtpt_info
{
  /* Eventpoint descriptor.  */
  struct wtxapi_evtpt wtx_evtpt;
  /* Tool identifier.  */
  int tool_id;
  /* Eventpoint identifier.  */
  evtpt_id_t evtpt_num;
};


/* Eventpoint list message. Same as WTX_EVTPT_LIST in WTX 3.0,
   and WTX_EVTPT_LIST_2 in WTX 2.0.  */

struct wtxapi_evtpt_list
{
  int n_evtpt;
  struct wtxapi_evtpt_info *p_evtpt_info;
};

/* Section descriptor. Based on WTX_SECTION_DESC (WTX 3.0)
   and LD_M_SECTION (WTX 2.0).  */

struct wtxapi_section_desc
{
  /* Section flags.  */
  int flags;
  /* Section base address.  */
  TGT_ADDR_T base_addr;
  /* Section length.  */
  int length;
};

/* Symbol description. Based on WTX_SYMBOL.  */

struct wtxapi_symbol
{
  /* Returned value for find request.  */
  int status;
  /* Protection domain ID.  */
  pd_id_t pd_id;
  /* Symbol name.  */
  char *name;
  /* 0 iff symbol name prefixed by an underscore.  */
  int exact_name;
  /* Symbol value.  */
  TGT_ADDR_T value;
  /* symbol type.  */
  int type;
  /* symbol type mask for lookup.  */
  int type_mask;
  /* Module name.  */
  char *module_name;
};

/* Symbol copy options for copy_current_symbol.  */

extern const int wtxapi_symbol_copy_none;
extern const int wtxapi_symbol_copy_name;
extern const int wtxapi_symbol_copy_module_name;
#define wtxapi_symbol_copy_all (wtxapi_symbol_copy_module_name \
                                  | wtxapi_symbol_copy_name)

enum wtxapi_symbol_copy_options
  {
    wtxapi_symbol_copy_full = 1,
    wtxapi_symbol_copy_dont_copy_name = 2,
    wtxapi_symbol_copy_dont_copy_module_name = 3,
    wtxapi_symbol_copy_dont_copy_strings = 4
  };

/* Symbol list. Based on WTX_SYMBOL and WTX_SYMBOL_LIST.  */

struct wtxapi_symbol_list;

/* Object module information. Based on WTX_MODULE_INFO.  */

struct wtxapi_module_info
{
  /* Protection domain ID.  */
  pd_id_t pd_id;
  /* Module ID.  */
  module_id_t module_id;
  /* Module name (with path).  */
  char *module_name;
  /* Flags used to load module.  */
  int load_flag;
  /* Number of sections.  */
  int n_sections;
  /* Section description.  */
  struct wtxapi_section_desc *section;
  /* List of undefined symbols. Only initialized after a 'load' operation.  */
  struct wtxapi_symbol_list *undef_list;
};

/* Module ID list. Based on WTX_MODULE_LIST (WTX 2.0).  */

struct wtxapi_module_list
{
  /* Number of module in list.  */
  int num_obj_mod;
  /* Arrays of object module id and pd id. For one index, you have a
     couple (module id, pd id).  */
  module_id_t *mod_id_array;
  pd_id_t *pd_id_array;  
};

/* task context descriptor. Same as WTX_TASK_CONTEXT_DEF (WTX 3.0) and
   equivalent to WTX_CONTEXT_DESC (WTX 2.0).  */

struct wtxapi_task_context_desc
{
  /* Context PD ID.  */
  pd_id_t pd_id;
  /* Integer or double.  */
  WTX_RETURN_TYPE return_type;
  /* Task name.  */
  char *name;
  /* Priority.  */
  int priority;
  /* Options.  */
  int options;
  /* Base of stack.  */
  TGT_ADDR_T stack_base;
  /* Stack size.  */
  int stack_size;
  /* Context entry point.  */
  TGT_ADDR_T entry;
  /* Redir. in file or NULL.  */
  INT32 redir_in;
  /* Redir. out file or NULL.  */
  INT32 redir_out;
  /* Redirection error file or NULL.  */
  INT32 redir_err;
  /* Arguments.  */
  int argc;
  TGT_ARG_T *argv;
};

/* PD context descriptor. Same on WTX_PD_CONTEXT_DEF (WTX 3.0) and
   equivalent to WTX_CONTEXT_DESC (WTX 2.0).  */

struct wtxapi_pd_context_desc
{
  /* PD name.  */
  char *name;
  /* Options.  */
  int options;
  /* Size of the PD's heap.  */
  int heap_size;
  /* Lowest task's priority.  */
  int low_priority;
  /* Highest task's priority.  */
  int high_priority;
  /* Page pool list name to use.  */
  TGT_ADDR_T page_pool_list;
  /* Rtn. address to call on deletion.  */
  TGT_ADDR_T destroy_rtn;
  /* Initial link path for this PD.  */
  char *link_path;
  /* Redirection in file or NULL.  */
  INT32 redir_in;
  /* Redirection out file or NULL.  */
  INT32 redir_out;
  /* Redirection error file or NULL.  */
  INT32 redir_err;
  /* Extra argument count (in argv).  */
  int argc;
  /* Extra argument array.  */
  TGT_ARG_T *argv;
};

/* Context descriptor. Based on WTX_CONTEXT_DESC (WTX 3.0). Different
   from WTX_CONTEXT_DESC in WTX 2.0.  */

struct wtxapi_context_desc
{
  /* Type of context.  */
  WTX_CONTEXT_TYPE context_type;

  union _wtx_context_def
  {
    /* Task context definition.  */
    struct wtxapi_task_context_desc task_context;
    /* PD context definition.  */
    struct wtxapi_pd_context_desc  pd_context;
  } wtx_context_def;
};

/* PD descriptor. Same as WTX_PD_DESC (WTX 3.0).  */

struct wtxapi_pd_desc
{
  /* Protection domain ID.  */
  pd_id_t  pd_id;
  /* Protection domain name.  */
  char *pd_name;
  /* Protection domain flags.  */
  int pd_flags;
  /* Link path string.  */
  char *pd_link_path_str;
  /* Nb of PD in linkpath.  */
  int pd_link_path_count;
  /* PD ids of the linkpath.  */
  pd_id_t  *pd_link_path;
  int pd_attach_to_count;
  pd_id_t  *pd_attach_to;
};

/* PD descriptor list. Same as WTX_PD_DESC_Q (WTX 3.0).  */

struct wtxapi_pd_desc_q
{
  struct wtxapi_pd_desc pd_desc;
  struct wtxapi_pd_desc_q *next;
};

/* Symbol table info. Same as WTX_SYM_TBL_INFO (WTX 3.0).  */

struct wtxapi_sym_tbl_info
{
  /* Protection domain ID.  */
  pd_id_t pd_id;
  /* Number of symbols.  */
  int sym_num;
  /* Name clash policy.  */
  int same_name_ok;
};

/* Target server information message. Based on WTX_TS_INFO.  */

struct wtxapi_ts_info
{
  /* Target link descriptor.  */
  WTX_TGT_LINK_DESC tgt_link_desc;
  /* Info obtained from Target.  */
  WTX_TGT_INFO tgt_info;
  /* Target Server version.  */
  char *version;
  /* Target server user name.  */
  char *user_name;
  /* Lock/authorization message.  */
  char *lock_msg;
  /* 0 iff the PDs are not initialized.  */
  int pd_initialized;
};


/* The following function are simple wrappers used to resolve the
   differences between the WTX versions. Unless specified in the
   comments, the WTX lib C documentation applies. If the return type
   is a WTX type, it should be deallocated using wtx_result_free.  If
   it is a type defined in remote-wtxapi.h, it should be deallocate
   using one of the deallocation function provided in this file.

   Note those differences:

   - when the return type of the WTX function is a STATUS, the wrapper
   returns an int; in this case, iff the WTX function returns
   WTX_ERROR, the wrapper returns 0;

   - when the return type of the WTX function is a BOOL32, the wrapper
   returns an int; in this case, iff the WTX function returns FALSE,
   the wrapper returns 0.  */

/* Note: This WTX functions are not used by GDB, and therefore no equivalent
   is provided in this file:
   wtxAsyncEventGet
   wtxAsyncNotifyEnable
   wtxAsyncNotifyDisable
   wtxPdDelete
   wtxInfo
   wtxMemDissassemble
   wtxTsInfoGet
   wtxTclHandleGrant
   wtxTclHandleRevoke
   wtxCacheTextUpdate
   wtxClientDataGet
   wtxClientDataSet
   wtxCpuInfoGet
   wtxErrDispatch
   wtxErrSet
   wtxErrToMsg
   wtxEventAdd
   wtxEventGet
   wtxEventListGet
   wtxEventToStrType
   wtxFileOpen
   wtxFileClose
   wtxFuncCall
   wtxHwBreakpointAdd
   wtxLogging
   wtxTargetBspNameGet
   wtxTargetIpAddressGet
   wtxTargetRtVersionGet
   wtxTargetToolNameGet
   wtxToolIdGet
   wtxTsKill
   wtxTsRestart
   wtxTsLock
   wtxTsTimedLock
   wtxTsUnlock
   wtxRegister
   wtxUnregister
   wtxVerify
   wtxVioChanGet
   wtxVioChanRelease
   wtxVioCtl
   wtxVioFileList
   wtxVioWrite

   Those one have been removed in WTX 3.0, and their equivalent shall not
   be added to this API:
   wtxCommandSend
   wtxConsoleCreate
   wtxConsoleKill
   wtxEventpointList
   wtxServiceAdd
   wtxTargetRtTypeGet
   wtxTargetHasAltivecGet
   wtxTargetHasDspGet
   wtxEach
   wtxDefaultDemanglerStyle
   wtxTargetToolFamilyGet
   wtxTargetHasAltivecGet (Only in VxWorks 5.5...)
*/

/* Initialize the global WTX client handle. Only one WTX connection is
   allowed.  */

extern int wtx_initialize ();

/* Terminate the use of the global WTX client handle.  */

extern int wtx_terminate ();

/* Connect the client to the target server.  */

extern int wtx_tool_attach (const char *target_name, const char *tool_name);

/* Check tool connection to the server.  */

extern int wtx_tool_connected ();

/* Detach from the target server.  */

extern int wtx_tool_detach ();

/* Clear any error for the tool.  */

extern int wtx_err_clear ();

/* Return the last error for a handle.  */

extern WTX_ERROR_T wtx_err_get ();

/* Add an error handler.  */

extern WTX_HANDLER_T wtx_err_handler_add (WTX_HANDLER_FUNC p_func,
                                          void *p_client_data);

/* Remove an error handler for WTX handle.  */

extern int wtx_err_handler_remove (WTX_HANDLER_T p_handler);

/* Fetch the last WTX API error string.  */

extern const char *wtx_err_msg_get ();

/* Get the agent mode. Return invalid_agent_mode if an error occurs.  */

extern WTX_AGENT_MODE_TYPE wtx_agent_mode_get ();

/* Set the mode of the target agent.  */

extern int wtx_agent_mode_set (WTX_AGENT_MODE_TYPE agent_mode);

/* Create a new breakpoint.  */

extern evtpt_id_t wtx_breakpoint_add (WTX_CONTEXT_TYPE context_type,
                                      WTX_CONTEXT_ID_T context_id,
                                      TGT_ADDR_T tgt_addr);

/* Create a new event point.  */

extern evtpt_id_t wtx_eventpoint_add (struct wtxapi_evtpt *p_evtpt);

/* Delete eventpoint from the target.  */

extern int wtx_eventpoint_delete (evtpt_id_t evtpt_id);

/* Get status of a context. Return invalid_context_status if the operation
   fails.  */

extern WTX_CONTEXT_STATUS wtx_context_status_get
  (WTX_CONTEXT_TYPE context_type, WTX_CONTEXT_ID_T context_id);

/* Continue execution of target context.  */

extern int wtx_context_cont (WTX_CONTEXT_TYPE context_type,
                             WTX_CONTEXT_ID_T context_id);

/* Create a context on target.
   - if P_CONTEXT_DESC is invalid, generates WTX_ERR_API_INVALID_ARG;
   - if another unidentified error occurs, return invalid_context_id;
   e.g if the operation is not permitted by the kernel.  */

extern WTX_CONTEXT_ID_T
  wtx_context_create (struct wtxapi_context_desc *p_context_desc);

/* Resume execution of a target context.  */

extern int wtx_context_resume (WTX_CONTEXT_TYPE context_type,
                               WTX_CONTEXT_ID_T context_id);

/* Add exit evpt notification.  */

extern evtpt_id_t wtx_context_exit_notify_add (WTX_CONTEXT_TYPE context_type,
                                               WTX_CONTEXT_ID_T context_id);

/* Kill a target context.  */

extern int wtx_context_kill (WTX_CONTEXT_TYPE context_type,
                             WTX_CONTEXT_ID_T context_id);

/* Single step exec of target context.  */

extern int wtx_context_step (WTX_CONTEXT_TYPE context_type,
                             WTX_CONTEXT_ID_T context_id,
                             TGT_ADDR_T step_start, TGT_ADDR_T step_end);

/* Suspend a target context.  */

extern int wtx_context_suspend (WTX_CONTEXT_TYPE context_type,
                                WTX_CONTEXT_ID_T context_id);

/* Stop (on WTX 3.0) or suspend (on WTX 2.0) a target context.  */

extern int wtx_context_stop (WTX_CONTEXT_TYPE context_type,
                             WTX_CONTEXT_ID_T context_id);

/* List event points on TS.  */

extern struct wtxapi_evtpt_list *wtx_eventpoint_list_get ();

/* Free mem allocated by WTX API call.  */

extern int wtx_result_free (void *p_result);

/* Evaluate Gopher string on target.  */

extern WTX_GOPHER_TAPE *wtx_gopher_eval (pd_id_t pd_id,
                                         const char *input_string);

/* Get info about memory pool.  */

extern WTX_MEM_INFO *wtx_mem_info_get (pd_id_t pd_id);

/* Alloc blocks in memory pool.  */

extern TGT_ADDR_T wtx_mem_alloc (pd_id_t pd_id, int num_bytes);

/* Perform checksum on target memory.  */

extern int wtx_mem_checksum (pd_id_t pd_id, TGT_ADDR_T start_addr,
                             int num_bytes);

/* Move a block of target memory.  */

extern int wtx_mem_move (pd_id_t src_pd_id, TGT_ADDR_T src_addr,
                         pd_id_t dst_pd_id, TGT_ADDR_T dest_addr,
                         int num_bytes);

/* Free a block of target memory.  */

extern int wtx_mem_free (pd_id_t pd_id, TGT_ADDR_T address);

/* Read memory from the target.  */

extern int wtx_mem_read (pd_id_t pd_id, TGT_ADDR_T from_addr, void *to_addr,
                         int num_bytes);

/* Read memory on WIDTH bytes.  */

extern int wtx_mem_width_read (pd_id_t pd_id, TGT_ADDR_T from_addr,
                               void *to_addr, int num_bytes,
                               int width);

/* Write memory on the target.  */

extern int wtx_mem_write (pd_id_t pd_id, void *from_addr,
                          TGT_ADDR_T to_addr, int num_bytes);

/* Write memory on the target, on WIDTH bytes large.  */

extern int wtx_mem_width_write (pd_id_t pd_id, void *from_addr,
                                TGT_ADDR_T to_addr, int num_bytes,
                                int width);

/* Set target memory to given value.  */

extern int wtx_mem_set (pd_id_t pd_id, TGT_ADDR_T addr, int num_bytes,
                        int val);

/* Add memory to the agent pool.  */

extern int wtx_mem_add_to_pool (pd_id_t pd_id, TGT_ADDR_T address,
                                int size);

/* Reallocate a block of target mem.  */

extern TGT_ADDR_T wtx_mem_realloc (pd_id_t pd_id, TGT_ADDR_T address,
                                   int num_bytes);

/* Allocate aligned target memory.  */

extern TGT_ADDR_T wtx_mem_align (pd_id_t pd_id, TGT_ADDR_T alignment,
                                 int num_bytes);

/* Scan target memory for pattern.  */

extern int wtx_mem_scan (pd_id_t pd_id, int match,
                         TGT_ADDR_T start_addr, TGT_ADDR_T end_addr,
                         int num_bytes, void *pattern,
                         TGT_ADDR_T *p_result);

/* Checks validity of target memory.  */

extern int wtx_obj_module_checksum (pd_id_t pd_id, module_id_t module_id,
                                    char *module_name);

/* Find obj module ID from name.  */

extern module_id_t wtx_obj_module_find_id (pd_id_t pd_id,
                                           const char *module_name);

/* Return the ID of the module which basename is equal to MODULE_NAME.
   Search the current PD first, before extending the search to all PDs.  */

extern module_id_t wtx_obj_module_in_system_find_id (const char *module_name);

/* Find module name given its ID.  */

extern const char *wtx_obj_module_find_name (pd_id_t pd_id,
                                             module_id_t module_id);

/* Give info on obj module.  */

extern struct wtxapi_module_info
  *wtx_obj_module_info_get (pd_id_t pd_id, module_id_t module_id);

/* Give info on obj modules. The complete pathname is given in
   the NAME field of the result.  */

extern struct wtxapi_module_info
  *wtx_obj_module_info_and_path_get (pd_id_t pd_id, module_id_t module_id);

/* List loaded obj modules (wrapper around wtxObjModuleList (WTX 2.0)
   or wtxObjModuleListGet (WTX 3.0)).  To search on every PD, pd_id
   should be set to WTX_MOD_FIND_IN_ALL_PD.  */

extern struct wtxapi_module_list *wtx_obj_module_list_get (pd_id_t pd_id);

/* Load a new object module.  */

extern struct wtxapi_module_info
  *wtx_obj_module_load (pd_id_t pd_id, char *filename, int load_flag);

/* Unload an obj module from target.  */

extern int wtx_obj_module_unload (pd_id_t pd_id, module_id_t mod_id);

extern int wtx_obj_module_by_name_unload (pd_id_t pd_id, char *name);

/* Send events matching expression.  */

extern int wtx_register_for_event (const char *reg_exp);

/* Read register data from the target.  */

extern int wtx_regs_get (WTX_CONTEXT_TYPE context_type,
                         WTX_CONTEXT_ID_T context_id,
                         WTX_REG_SET_TYPE reg_set,
                         int first_byte,
                         int num_bytes,
                         void *reg_memory);

/* Write to registers on the target.  */

extern int wtx_regs_set (WTX_CONTEXT_TYPE context_type,
                         WTX_CONTEXT_ID_T context_id,
                         WTX_REG_SET_TYPE reg_set,
                         int first_byte,
                         int num_bytes,
                         void *reg_memory);

/* Convert str to a TGT_ADDR_T.  */

extern TGT_ADDR_T wtx_str_to_tgt_addr (const char *str);

/* Convert str to context ID.  */

extern WTX_CONTEXT_ID_T wtx_str_to_context_id (const char *str);

/* Convert str ton context type.  */

extern WTX_CONTEXT_TYPE wtx_str_to_context_type (const char *str);

/* Convert str to an INT32.  */

extern int wtx_str_to_int32 (const char *str);

/* Convert string to event type.  */

extern WTX_EVENT_TYPE wtx_str_to_event_type (const char *str);

/* Add symbol with given params.  */

extern int wtx_sym_add (pd_id_t pd_id, const char *sym_name,
                        TGT_ADDR_T sym_value, int sym_type);

/* Find info on symbol. Based on wtxSymFind WTX 2.0, with two differences:
   _ one additional parameter: the protection domain ID (PD_ID);
   _ no filter on the type.  */

extern struct wtxapi_symbol_list *wtx_sym_find (pd_id_t pd_id,
                                                char *sym_name,
                                                TGT_ADDR_T sym_value,
                                                int exact_name);

/* Get list of symbols. Based on wtxSymListGet WTX 2.0, with three differences:
   - one additional parameter: the protection domain ID (PD_ID);
   - no filter on the unknown symbols;
   - no filter on the module name/id.  */

extern struct wtxapi_symbol_list *wtx_sym_list_get (pd_id_t pd_id,
                                                    char *substring,
                                                    TGT_ADDR_T value);

/* Get list of symbols.  */

extern struct wtxapi_symbol_list
  *wtx_sym_list_by_module_id_get (pd_id_t pd_id, const char *string,
                                  module_id_t module_id, TGT_ADDR_T value,
                                  int list_unknown);

/* Get list of symbols.  */

extern struct wtxapi_symbol_list
  *wtx_sym_list_by_module_name_get (pd_id_t pd_id,
                                    const char *string,
                                    const char *module_name,
                                    TGT_ADDR_T value,
                                    int list_unknown);

/* Remove a symbol from sym table.  */

extern int wtx_sym_remove (pd_id_t pd_id, const char *sym_name,
                           int sym_type);

/* Get the symbol table info.  */

extern struct wtxapi_sym_tbl_info *wtx_sym_tbl_info_get (pd_id_t pd_id);

/* Reset the target.  */

extern int wtx_target_reset ();

/* Get target server name.  */

extern const char *wtx_ts_name_get ();

/* Get target-runtime version.  */

extern const char *wtx_target_rt_version_get ();

/* Get the target CPU type code.  */

extern int wtx_target_cpu_type_get ();

/* Check for floating point processor.  */

extern int wtx_target_has_fpp_get ();

/* Get edianness of target.  */

extern WTX_ENDIAN_T wtx_target_endian_get ();

/* Get target boot line info.  */

extern const char *wtx_target_bootline_get ();

/* Return name of current tool.  */

extern const char *wtx_tool_name_get ();

/* Return the Tornado version.  */

extern const char *wtx_ts_version_get ();

/* Unregister for some events.  */

extern int wtx_unregister_for_event (char *reg_exp);

/* Call func on target within agent.  */

extern int wtx_direct_call (TGT_ADDR_T entry, void *p_ret_val,
                            TGT_ARG_T arg0, TGT_ARG_T arg1, TGT_ARG_T arg2,
                            TGT_ARG_T arg3, TGT_ARG_T arg4, TGT_ARG_T arg5,
                            TGT_ARG_T arg6, TGT_ARG_T arg7, TGT_ARG_T arg8,
                            TGT_ARG_T arg9);

/* Get info about target and server.  */

extern struct wtxapi_ts_info * wtx_ts_info_get ();

/* Reattach to the target.  */

extern int wtx_target_attach ();

/* Probe to see if registry is running.  */

extern int wtx_probe ();

/* Set WTX timeout.  */

extern int wtx_timeout_set (int msec);

/* Get the current timeout.  */

extern int wtx_timeout_get (int *p_msec);

/* Create a new protection domain.  */

extern TGT_ADDR_T wtx_pd_create (const char *name, int options, int heap_size,
                                 int low_priority, int high_priority,
                                 TGT_ADDR_T page_pool_list,
                                 const char *link_path);

/* Get kernel Protection Domain ID.  */

extern pd_id_t wtx_pd_kernel_get ();

/* Get the current Protection Domain.  */

extern pd_id_t wtx_pd_current_get ();

/* Set the current Protection Domain.  */

extern int wtx_pd_current_set (pd_id_t pd_id);

/* Get the list of allocated PDs.  */

extern struct wtxapi_pd_desc_q *wtx_pd_info_q_get ();

/* Simple additional functions.  */

/* Test target server availability.  */

extern int wtxapi_target_server_available_p ();

/* Symbol list handling. The symbol list contain a 'current symbol'.
   This current symbol is used to go through the list.  */

/* Get a copy of the current symbol.  */

extern struct wtxapi_symbol
  *get_current_wtxapi_symbol (struct wtxapi_symbol_list* list);


/* Copy current symbol of LIST into TO.

   OPTIONS is used to avoid uneeded dynamic allocation.

   If OPTIONS is set to wtxapi_symbol_copy_full, the fields name and
   module_name of TO are null pointers, these string fields of TO are
   allocated and fully copied from the current symbol, and should be
   deallocated by the caller.

   If OPTIONS is set to wtxapi_symbol_copy_name (resp.
   wtxapi_symbol_copy_module_name), the field name (resp. the
   module_name) of TO is copied. If not, it is is set to NULL.

   If OPTIONS is set to wtxapi_symbol_copy_none, the field name and
   module_name of TO are not copied and are set to NULL.  In this case
   and only in this case no dynamic allocation is performed by this
   function.  */

extern void copy_current_wtxapi_symbol (struct wtxapi_symbol_list* list,
                                        struct wtxapi_symbol* to,
                                        int options);

/* Change the current symbol to the first element of the list.If there
   is no next element, return 0.  */

int go_to_first_element_in_wtxapi_sym_list (struct wtxapi_symbol_list* list);

/* Change the current symbol to the next element of the list. If there
   is no next element, return 0.  */

int go_to_next_element_in_wtxapi_sym_list (struct wtxapi_symbol_list* list);

/* Deallocation functions for the structures,lists returned by the
   function defined in this file. It frees "everything", i.e. it go
   through every element of the list and deallocate it, and also
   deallocate strings and arrays referenced by a field.  */

extern void free_wtxapi_evtpt_list (struct wtxapi_evtpt_list *to_free);

extern void free_wtxapi_module_list (struct wtxapi_module_list *to_free);

extern void free_wtxapi_pd_desc_q (struct wtxapi_pd_desc_q *to_free);

extern void free_wtxapi_symbol (struct wtxapi_symbol *to_free);

extern void free_wtxapi_symbol_list (struct wtxapi_symbol_list *to_free);

extern void free_wtxapi_module_info (struct wtxapi_module_info *to_free);

extern void free_wtxapi_ts_info (struct wtxapi_ts_info *to_free);

/* Stubs for GDB cleanup functions.  */

extern void cleanup_wtxapi_evtpt_list (void *to_free);

extern void cleanup_wtxapi_module_list (void *to_free);

extern void cleanup_wtxapi_pd_desc_q (void *to_free);

extern void cleanup_wtxapi_symbol (void *to_free);

extern void cleanup_wtxapi_symbol_list (void *to_free);

extern void cleanup_wtxapi_module_info (void *to_free);

extern void cleanup_wtxapi_ts_info (void *to_free);
