/* IRIX 6 spyThread library interface. 
   Copyright 2002
   Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#ifndef _IRIX6_SPYTHREAD_H
#define _IRIX6_SPYTHREAD_H

/* Part I: This section is derived from internal definitions for the libspy
 * package, found in the files  spyBase.h, spyCmd.h, and spyThread.h.
 * The original author was SGI. */

#include <signal.h>
#include <sys/fault.h>
#include <sys/procfs.h>
#include <sys/syscall.h>
#include <sys/types.h>

#include "gregset.h"

/* Basic object: process
 *
 * A process is registered with spy by the client as a shared data
 * structure containing process information, callbacks for spy to
 * use and private handles for the client and spy to use.
 *
 * A client registers a text address and arg as callback.
 * This permits the client to register different callbacks
 * for each process it instantiates.
 */

/* Callbacks used by library. */

typedef struct spyCallBack {
  int (*spc_symbol_addr) (void*, char*, off64_t*);
  void* spc_symbol_addr_arg;
  int (*spc_memory_read) (void*, int, off64_t, char*, size_t);
  void* spc_memory_read_arg;
  int (*spc_register_read) (void*, int, void*);
  void* spc_register_read_arg;
} spyCallBack_t;


/* ABI types */

typedef enum { SP_O32 = 0, SP_N32, SP_N64 } spABI_t;

typedef struct spyProc {
  spABI_t	sp_abi;
  int		sp_procfd;	/* /proc or -1 if not live */
  spyCallBack_t* sp_vec;
  void*		sp_client;	/* private client data for proc */
  void*		sp_spy;		/* private spy data for proc */
} spyProc_t;


typedef uint32_t	spyThread_t;
#define SPYTHREADNULL	((spyThread_t)-1)

typedef int (*scan_callback_t) (spyProc_t*, spyThread_t, void*, void*);

typedef struct spyScanOp {
  uint_t	sso_dom;
  spyThread_t	sso_thd;
  scan_callback_t sso_cb;
  void*		sso_cbArg;
} spyScanOp_t;

typedef struct spyThreadCalls {
  int	stc_version;

  int	(*stc_ProcNew)(spyProc_t*);
  int	(*stc_ProcDel)(spyProc_t*);

  int	(*stc_ScanOp)(spyProc_t*, int, void*, spyScanOp_t*);

} spyThreadCalls_t;

typedef struct spyCmdList spyCmdList_t;

/* Options for the ScanOp domain.
 *
 *	STC_PROCESS
 * or	STC_THREAD			named thread
 * or	STC_SCAN_USER			all threads with user contexts
 * or	STC_SCAN_KERN			all threads with kernel contexts
 * or	STC_SCAN_STOPPED		all stopped threads
 * or	STC_SCAN_EVENTS			all threads at an event of interest
 */
#define STC_PROCESS		0x0001
#define STC_THREAD		0x0002
#define STC_SCAN_ALL		(STC_SCAN|STC_USER|STC_KERN)
#define STC_SCAN_USER		(STC_SCAN|STC_USER)
#define STC_SCAN_KERN		(STC_SCAN|STC_KERN)
#define STC_SCAN_STOPPED	(STC_SCAN|STC_KERN|STC_STOPPED)
#define STC_SCAN_EVENTS		(STC_SCAN|STC_KERN|STC_EVENTS)


/* Internal masks.
 */
#define STC_SCAN		0x0004
#define STC_USER		0x0010
#define STC_KERN		0x0020
#define	STC_STOPPED		0x0100
#define	STC_EVENTS		0x0200
#define STC_LIVE		0x1000
#define STC_DEAD		0x2000
#define STC_TGT			0x000f
#define STC_CTX			0x00f0
#define STC_LIFE		0xf000

/* Proc ioctl extensions
 * Anything below ('q'<<8) is free - so we can avoid including <sys/procfs.h>
 */
#define SPYCGINFO0	1	/* state info (char[STC_INFO0BUFLEN]) */
#define SPYCGNAME	3	/* user name (spyThread_t*) */
#define SPYCITER	4	/* No-op, for call backs (0) */


#define STC_INFO0BUFLEN	24	/* buffer for inline info0 requests */

/* Initialize the spyThread library, setting the members of CALLS to
   valid values.   The second argument is not publically documented, 
   and may be set to NULL, as GDB does. */
 
extern int spyThreadInit(spyThreadCalls_t* calls, spyCmdList_t**);

/* End of Part I and SGI-authored material */

/* Part II: A slightly more useful interface to the spyThread library. 
 */

#define IS_LWP_ID(thread) (((thread) & 0x80000000) != 0)

extern int libspy_enabled;

extern int irix_get_threads (spyProc_t* handle, prstatus_t** thread_list);

extern int irix_get_kernel_threads (spyProc_t* handle, prstatus_t** thread_list);

extern int irix_spy_open (int fd, spyProc_t** handle_p);

extern void irix_spy_close (spyProc_t* handle_p);

extern int irix_get_status (spyProc_t* handle, int fd, spyThread_t pthread, 
			    prstatus_t* prstatus);

extern int irix_wait_proc (spyProc_t* handle, int fd, prstatus_t* prstatus);

extern int irix_resume (spyProc_t* handle_p, int fd, spyThread_t lwpid,
			prrun_t* prrun_p);

extern int irix_reset_held_signals (spyProc_t* handle);

extern int irix_get_fpregs (spyProc_t* handle_p,
			    spyThread_t tid, gdb_fpregset_t* fpregset);

extern int irix_set_gregs (spyProc_t* handle_p,
			   spyThread_t tid, gdb_gregset_t* gregs);

extern int irix_set_fregs (spyProc_t* handle_p,
			   spyThread_t tid, gdb_fpregset_t* fregs);

extern void initialize_libspy (void);

#endif /* _SPYTHREAD_H */
