/* Native support code for HPUX IA64, for GDB the GNU debugger.

   Copyright 2005 Free Software Foundation, Inc.

   Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "defs.h"
#include "floatformat.h"
#include "gdb_string.h"
#include "gdbcore.h"
#include "ia64-tdep.h"
#include "inferior.h"
#include "osabi.h"
#include "regcache.h"

/* A sequence of instructions pushed on the stack when we want to perform
   an inferior function call.  The main purpose of this code is to save
   the output region of the register frame belonging to the function
   from which we are making the call.  Normally, all registers are saved
   prior to the call, but this does not include stacked registers because
   they are seen by GDB as pseudo registers.

   On Linux, these stacked registers can be saved by simply creating
   a new register frame, or in other words by moving the BSP.  But the
   HP/UX kernel does not allow this.  So we rely on this code instead,
   that makes functions calls whose only purpose is to create new
   register frames.

   The array below is the result obtained after assembling the code
   shown below. It's an array of bytes in order to make it independent
   of the host endianess, in case it ends up being used on more than
   one target.

   start:
        // Save b0 before using it (into preserved reg: r4).
        mov r4 = b0
        ;;

        br.call.dptk.few b0 = stub#
        ;;

        // Add a nop bundle where we can insert our dummy breakpoint.
        nop.m 0
        nop.i 0
        nop.i 0
        ;;

   stub:
        // Alloc a new register stack frame.  Here, we set the size
        // of all regions to zero.  Eventually, GDB will manually
        // change the instruction to set the size of the local region
        // to match size of the output region of the function from
        // which we are making the function call.  This is to protect
        // the value of the output registers of the function from
        // which we are making the call.
        alloc r6 = ar.pfs, 0, 0, 0, 0

        // Save b0 before using it again (into preserved reg: r5).
        mov r5 = b0
        ;;

        //  Now that we have protected the entire output region of the
        //  register stack frame, we can call our function that will
        //  setup the arguments, and call our target function.
        br.call.dptk.few b0 = call_dummy#
        ;;

        //  Restore b0, ar.pfs, and return
        mov b0 = r5
        mov.i ar.pfs = r6
        ;;
        br.ret.dptk.few b0
        ;;

   call_dummy:
        //  Alloc a new frame, with 2 local registers, and 8 output registers
        //  (8 output registers for the maximum of 8 slots passed by register).
        alloc r32 = ar.pfs, 2, 0, 8, 0

        //  Save b0 before using it to call our target function.
        mov r33 = b0

        // Load the argument values placed by GDB inside r14-r21 in their
        // proper registers.
        or r34 = r14, r0
        or r35 = r15, r0
        or r36 = r16, r0
        or r37 = r17, r0
        or r38 = r18, r0
        or r39 = r19, r0
        or r40 = r20, r0
        or r41 = r21, r0
        ;;

        // actual call
        br.call.dptk.few b0 = b1
        ;;

        mov.i ar.pfs=r32
        mov b0=r33
        ;;

        br.ret.dptk.few b0
        ;;

*/

static const unsigned char ia64_hpux_dummy_code[] =
{
  0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x40, 0x00,
  0x00, 0x62, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
  0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x00, 0x00, 0x20, 0x00, 0x00, 0x52,
  0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
  0x02, 0x30, 0x00, 0x00, 0x80, 0x05, 0x50, 0x00,
  0x00, 0x62, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
  0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x00, 0x00, 0x30, 0x00, 0x00, 0x52,
  0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x28,
  0x04, 0x80, 0x03, 0x00, 0x60, 0x00, 0xaa, 0x00,
  0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x00, 0x80, 0x00, 0x00, 0x84, 0x02,
  0x00, 0x00, 0x29, 0x04, 0x80, 0x05, 0x10, 0x02,
  0x00, 0x62, 0x00, 0x40, 0xe4, 0x00, 0x38, 0x80,
  0x00, 0x18, 0x3d, 0x00, 0x0e, 0x20, 0x40, 0x82,
  0x00, 0x1c, 0x40, 0xa0, 0x14, 0x01, 0x38, 0x80,
  0x00, 0x30, 0x49, 0x00, 0x0e, 0x20, 0x70, 0x9a,
  0x00, 0x1c, 0x40, 0x00, 0x45, 0x01, 0x38, 0x80,
  0x0a, 0x48, 0x55, 0x00, 0x0e, 0x20, 0x00, 0x00,
  0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
  0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x00, 0x00, 0x10, 0x00, 0x80, 0x12,
  0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x01, 0x55, 0x00, 0x00, 0x10, 0x0a, 0x00, 0x07,
  0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x00, 0x80, 0x00, 0x00, 0x84, 0x02
};

/* Return non-zero if the value of the register identified by REGNUM
   can be modified.  */

static int
ia64_hpux_can_store_ar_register (int regnum)
{
  switch (regnum)
    {
      case IA64_RSC_REGNUM:
      case IA64_RNAT_REGNUM:
      case IA64_CSD_REGNUM:
      case IA64_SSD_REGNUM:
      case IA64_CCV_REGNUM:
      case IA64_UNAT_REGNUM:
      case IA64_FPSR_REGNUM:
      case IA64_PFS_REGNUM:
      case IA64_LC_REGNUM:
      case IA64_EC_REGNUM:
         return 1;
         break;

      default:
         return 0;
         break;
    }
}

/* Return non-zero if the value of the register identified by REGNUM
   can not be modified.  */

static int
ia64_hpux_cannot_store_register (int regnum)
{
  /* General registers.  */
  
  if (regnum == IA64_GR0_REGNUM)
    return 1;

  /* FP register.  */
  
  if (regnum == IA64_FR0_REGNUM || regnum == IA64_FR1_REGNUM)
    return 1;

  /* Application registers.  */
  if (regnum >= IA64_AR0_REGNUM && regnum <= IA64_AR0_REGNUM + 127)
    return (!ia64_hpux_can_store_ar_register (regnum));
  
  /* We can store all other registers.  */
  return 0;
}

/* Implement the push_dummy_code gdbarch method.  */
/* Assume SP is already 16-byte aligned.  */

static CORE_ADDR
ia64_hpux_push_dummy_code (struct gdbarch *gdbarch, CORE_ADDR sp,
                           CORE_ADDR funaddr, int using_gcc,
                           struct value **args, int nargs, 
                           struct type *value_type, CORE_ADDR *real_pc, 
                           CORE_ADDR *bp_addr)
{
  const CORE_ADDR cfm = read_register (IA64_CFM_REGNUM);
  const int soo = ia64_size_of_output (cfm);
  char buf[16];

  /* Reserve some space on the stack to hold the dummy code.  */
  sp = sp - sizeof (ia64_hpux_dummy_code);

  /* Set the breakpoint address at the first instruction of the bundle
     in the dummy code that has only nops.  This is where the dummy code
     expects us to break.  */
  *bp_addr = sp + 0x20;

  /* Start the inferior function call from the dummy code.  The dummy
     code will then call our function.  */
  *real_pc = sp;

  /* Transfer the dummy code to the inferior.  */
  write_memory (sp, (char *) ia64_hpux_dummy_code,
                sizeof (ia64_hpux_dummy_code));

  /* Update the size of the local portion of the register frame allocated
     by ``stub'' to match the size of the output region of the current
     register frame.  This allows us to save the stacked registers.
     
     The "alloc" instruction is located at slot 0 of the bundle at +0x30.
     Update the "sof" and "sol" portion of that instruction which are
     respectively at bits 18-24 and 25-31 of the bundle.  */
  memcpy (buf, ia64_hpux_dummy_code + 0x30, sizeof (buf));

  buf[2] |= ((soo & 0x3f) << 2);
  buf[3] |= (soo << 1);
  if (soo > 63)
    buf[3] |= 1;

  write_memory (sp + 0x30, buf, sizeof (buf));

  /* Return the new (already properly aligned) SP.  */
  return sp;
}

static struct frame_id
ia64_hpux_unwind_dummy_id (struct gdbarch *gdbarch,
                           struct frame_info *next_frame)
{
  char buf[8];
  CORE_ADDR sp;
  CORE_ADDR bp_addr;

  frame_unwind_register (next_frame, IA64_GR12_REGNUM, buf);
  sp = extract_unsigned_integer (buf, 8);

  /* The call sequence is such that the address of the dummy breakpoint
     we inserted is stored in r5.  */
  frame_unwind_register (next_frame, IA64_GR5_REGNUM, buf);
  bp_addr = extract_unsigned_integer (buf, 8);

  /* We don't need to provide the original BSP addr.  All we need to do
     is to return a frame ID that matches the one that GDB stored before
     making the call, and this ID doesn't has the BSP.  */

  return frame_id_build (sp, bp_addr);
}

static CORE_ADDR
ia64_hpux_find_global_pointer (CORE_ADDR faddr)
{ 
  CORE_ADDR gp;

  /* If this function is part of one of the shared libraries, then
     the ia64-hpux solib module will find it and return the associated
     global pointer.  */
  gp = solib_ia64_hpux_get_got_by_pc (faddr);
  if (gp != 0)
    return gp;
  
  /* Try another method.  The method below doesn't work for functions
     inside shared libraries for some reason I haven't investigated,
     but does seem to work when the function is part of the executable.
     That's why we're calling it only when the above didn't find the GP.  */
  return generic_elf_find_global_pointer (faddr);
}   
      
static CORE_ADDR
ia64_hpux_allocate_new_rse_frame (int sof)
{
  /* We cannot change the value of the BSP register on HP/UX,
     so we can't allocate a new RSE frame.  */
  return 0;
}

static void
ia64_hpux_store_argument_in_slot (struct regcache *regcache, CORE_ADDR bsp,
                                  int slotnum, char *buf)
{
  /* The call sequence on this target expects us to place the arguments
     inside r14 - r21.  */
  regcache_cooked_write (regcache, IA64_GR0_REGNUM + 14 + slotnum, buf);
}

static void
ia64_hpux_set_function_addr (CORE_ADDR func_addr)
{
  /* The calling sequence calls the function whose address is placed
     in register b1.  */
  write_register (IA64_BR1_REGNUM, func_addr);
}

static const struct infcall_ops ia64_hpux_infcall_ops = {
  ia64_hpux_allocate_new_rse_frame,
  ia64_hpux_store_argument_in_slot,
  ia64_hpux_set_function_addr
};

static enum gdb_osabi
ia64_hpux_osabi_sniffer (bfd *abfd)
{
  if (strstr (bfd_get_target (abfd), "hpux") != NULL)
    return GDB_OSABI_HPUX_ELF;

  return GDB_OSABI_UNKNOWN;
}

static void
ia64_hpux_init_osabi (struct gdbarch_info info, struct gdbarch *gdbarch)
{
  struct gdbarch_tdep *tdep = gdbarch_tdep (gdbarch);

  set_gdbarch_long_double_format (gdbarch, &floatformat_ia64_quad_big);
  set_gdbarch_cannot_store_register (gdbarch, ia64_hpux_cannot_store_register);

  /* Inferior functions must be called from stack. */
  set_gdbarch_call_dummy_location (gdbarch, ON_STACK);
  set_gdbarch_push_dummy_code (gdbarch, ia64_hpux_push_dummy_code);
  tdep->infcall_ops = ia64_hpux_infcall_ops;
  set_gdbarch_unwind_dummy_id (gdbarch, ia64_hpux_unwind_dummy_id);

  tdep->find_global_pointer = ia64_hpux_find_global_pointer;
}

void
_initialize_ia64_hpux_tdep (void)
{
  gdbarch_register_osabi_sniffer (bfd_arch_ia64,
                                  bfd_target_elf_flavour,
                                  ia64_hpux_osabi_sniffer);

  gdbarch_register_osabi (bfd_arch_ia64, 0, GDB_OSABI_HPUX_ELF,
                          ia64_hpux_init_osabi);
}
