-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/gsql/gui_addrow.adb,v $
--  Description     : dialog to add a row to a table description             --
--  Author          : Michael Erdmann                                        --
--  Created         : 8.8.2001                                               --
--  Last Modified By: $Author: me $
--  Last Modified On: $Date: 2001/11/27 20:39:29 $
--  Status          : $State: Exp $
--
--  Copyright (C) 2000 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Functional description                                                   --
--  ======================                                                   --
--                                                                           --
--  Author                                                                   --
--  ======                                                                   --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Text_IO;              use Ada.Text_IO;

with Glib;                     use Glib;
with Gdk.Color;                use Gdk.Color;
with Gdk.Font;                 use Gdk.Font;
with Gtk.Box;                  use Gtk.Box;
with Gtk.Clist;                use Gtk.Clist;
with Gtk.Check_Button;         use Gtk.Check_Button;
with Gtk.Radio_Button;         use Gtk.Radio_Button;
with Gtk.Enums;                use Gtk.Enums;
with Gtk.Hbutton_Box;          use Gtk.Hbutton_Box;
with Gtk.Handlers;             use Gtk.Handlers;
with Gtk.Label;                use Gtk.Label;
with Gtk.Frame;                use Gtk.Frame;
with Gtk.Gentry;               use Gtk.Gentry;
with Gtk.Radio_Menu_Item;      use Gtk.Radio_Menu_Item;
with Gtk.Widget;               use Gtk.Widget;
with Gtk;                      use Gtk;

with Gsql_Types;               use Gsql_Types;

with GUI_Window;               use GUI_Window;
with GUI_Dialog;               use GUI_Dialog;
with GUI_Common;               use GUI_Common;
with GUI_Logging;              use GUI_Logging;
with GUI_Table;                use GUI_Table;

package body GUI_AddRow is

   Version   : constant String := "$Id: gui_addrow.adb,v 1.11 2001/11/27 20:39:29 me Exp $";

   --- ************************************************************************** ---
   --  ***                       C L A S S    D A T A                         *** ---
   --- ************************************************************************** ---

   Cmd_Apply : constant Natural := 100;
   Cmd_Ok    : constant Natural := 101;


   List_Of_Sql_Types : constant Text_Buffer := (
           To_Unbounded_String( "boolean"                 ),
           To_Unbounded_String( "character(...)"          ),
           To_Unbounded_String( "date"                    ),
           To_Unbounded_String( "decimal(p,s)"            ),
           To_Unbounded_String( "float(p)"                ),
           To_Unbounded_String( "smallint"                ),
           To_Unbounded_String( "int"                     ),
           To_Unbounded_String( "integer"                 ),
           To_Unbounded_String( "interval"                ),
           To_Unbounded_String( "decimal(p,s)"            ),
           To_Unbounded_String( "numeric(p,s)"            ),
           To_Unbounded_String( "time"                    ),
           To_Unbounded_String( "time with time zone"     ),
           To_Unbounded_String( "timestamp with time zone"),
           To_Unbounded_String( "character varying(n)"    )
      );

   SQL_Undef    : constant Positive := 1;
   SQL_NULL     : constant Positive := 2;
   SQL_Not_Null : constant Positive := 3;

   SQL_Constraints : constant Text_Buffer := (
           SQL_Undef    => To_Unbounded_String( "Empty"                   ),
           SQL_Null     => To_Unbounded_String( "NULL"                    ),
           SQL_Not_Null => To_Unbounded_String( "NOT NULL"                )
      );

   type Choice_Result_Type is
      array( Positive range <> ) of Gtk_Radio_Button;

   ------------
   -- Choice --
   ------------
   function Choice(
      Parent : in Gtk_Box;
      Text   : in Text_Buffer;
      Border : in Boolean := True ) return Choice_Result is
      -- create a set of radio buttons
      First  : Boolean := True;
      Button : Gtk_Radio_Button;
      Vbox   : Gtk_Box;
      Fr     : Gtk_Frame;
      Result : Choice_Result := new Choice_Result_Type( Text'Range );
   begin
      Gtk_New_Vbox( Vbox );
      if Border then
         Gtk_New( Fr );
         Add( Fr, Vbox );
      end if;

      for I in Text'Range loop
         if First then
            Gtk_New (Button, Widget_SList.Null_List, To_String( Text(I) ));
            First := False;
         else
            Gtk_New (Button, Group (Button), To_String( Text(I) ) );
         end if;
         Pack_Start (Vbox, Button, True, True, 0);

         Result(I) := Button;
      end loop;

      if Border then
         Pack_Start( Parent, Fr, Expand => False );
      else
         Pack_Start( Parent, Vbox, Expand => False );
      end if;

      return Result;
   end Choice;

   --------------
   -- Selected --
   --------------
   function Selected(
      Choice : in Choice_Result ) return Natural is
   begin
      for I in Choice'Range loop
         if Get_Active (Choice(I)) then
            return I;
         end if;
      end loop;

      return 0;
   end Selected;

   ---------
   -- Set --
   ---------
   procedure Set(
      Choice : in Choice_Result;
      Button : in Positive;
      Active : in Boolean := True ) is
   begin
      Gtk.Check_Button.Set_Active( Gtk_Check_Button(Choice(Button)), Active );
   end Set;

   -------------------
   -- Name_And_Type --
   -------------------
   procedure Name_And_Type(
      This           : in out Object;
      Parent         : in out Gtk_Box;
      Data           : in  Row_Information_Record ) is
      Vbox           : Gtk_Box;
      Fr             : Gtk_Frame;

      Name           : Gtk_Entry renames This.Name;
      Sql_Type       : Gtk_Combo renames This.Sql_Type;
      Sql_Primary    : GUI_Checkbutton.Handle renames This.Sql_Primary;
      Sql_Unique     : GUI_Checkbutton.Handle renames This.Sql_Unique;
   begin
      Gtk_New( Fr );
      Set_Label( Fr, "Column name & type");
      Gtk_New_Vbox( Vbox );
      Add( Fr, Vbox );

      Entry_Field( Vbox, This.Name,  "Name        " );
      Set_Text( Name, To_String( Data.Name ) );

      Combo_Field( Vbox, Sql_Type,   "Type        ", List_Of_SQL_Types  );
      Set_Text( Get_Entry(SQL_Type), To_String( Data.Sql_Type ) );

      This.Sql_Null := Choice( Vbox, SQL_Constraints, Border => True );

      if Data.Is_Null then
         Set( This.Sql_Null, SQL_NULL );
      elsif Data.Is_Not_Null then
         Set( This.Sql_Null, SQL_NOT_NULL );
      else
         Set( This.Sql_Null, SQL_Undef );
      end if;

      Add_Checkbutton( Vbox, Sql_Primary, "PRIMARY KEY", Data.Primary_Key );
      Add_Checkbutton( Vbox, Sql_Unique,  "UNIQUE", Data.Unique );

      Pack_Start( Parent, Fr );
      Show_All( Fr );
   end Name_And_Type;

   -----------------
   -- Constraints --
   -----------------
   procedure Default_Value(
      This           : in out Object;
      Parent         : in out Gtk_Box;
      Data           : in  Row_Information_Record ) is
      Vbox           : Gtk_Box;
      Fr             : Gtk_Frame;
      Sql_Default    : Gtk_Entry renames This.Sql_Default;
   begin
      Gtk_New( Fr );
      Set_Label( Fr, "Default value expression");
      Gtk_New_Vbox( Vbox );
      Add( Fr, Vbox );

      Entry_Field( Vbox, Sql_Default,"Default     " );
      Set_Text( SQL_Default, To_String( Data.Default ) );

      Pack_Start( Parent, Fr );
      Show_All( Fr );
   end Default_Value;

   -----------------
   -- Constraints --
   -----------------
   procedure Constraints(
      This           : in out Object;
      Parent         : in out Gtk_Box;
      Data           : in  Row_Information_Record ) is
      Vbox           : Gtk_Box;
      Fr             : Gtk_Frame;
      Sql_Constraint : Gtk_Entry renames This.Sql_Constraint;
      Sql_Check      : Gtk_Entry renames This.Sql_Check;
   begin
      Gtk_New( Fr );
      Set_Label( Fr, "Column constraints");
      Gtk_New_Vbox( Vbox );
      Add( Fr, Vbox );

      Entry_Field( Vbox, Sql_Constraint, "Constraint Name" );
      Set_Text( SQL_Constraint, To_String( Data.Constraint ) );
      Entry_Field( Vbox, Sql_Check,      "Check Condition" );

      Pack_Start( Parent, Fr );
      Show_All( Fr );
   end Constraints;

   --- ************************************************************************** ---
   --- ***                  P U B L I C    P R O C E D U R E S                *** ---
   --- ************************************************************************** ---

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize(
      This     : in out Object;
      Row      : in Row_Information_Record;
      Sheet    : in GUI_Worksheet.Object'Class ) is
   begin
      GUI_Dialog.Initialize( GUI_Dialog.Object(This) );
      This.Sheet := Self( Sheet );
      This.Row   := Row;
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize(
      This  : in out Object ) is
   begin
      Destroy( This.Sql_Type );
      Destroy( This.Name  );
      Destroy( This.Sql_Default );
      Destroy( This.Sql_Constraint );
      Destroy( This.Comment );

      GUI_Dialog.Finalize( GUI_Dialog.Object(This) );
   end Finalize;

   ----------------
   -- Get_Result --
   ----------------
   procedure Get_Result(
      This           : in out Object;
      Data           : out Row_Information_Record ) is
      -- this function gets all set parameters from the panel and
      -- stores it in the text buffer.
      Name           : Gtk_Entry renames This.Name;
      Sql_Type       : Gtk_Combo renames This.Sql_Type;
      Sql_Default    : Gtk_Entry renames This.Sql_Default;
      Sql_Constraint : Gtk_Entry renames This.SQL_Constraint;
      Comment        : Gtk_Entry renames This.Comment;
      Sql_Primary    : GUI_Checkbutton.Handle renames This.Sql_Primary;
      Sql_Unique     : GUI_Checkbutton.Handle renames This.Sql_Unique;
   begin
      Data.Name        := To_Unbounded_String( Get_Text( Name ) );
      Data.Sql_Type    := To_Unbounded_String( Get_Text( Get_Entry(Sql_Type)) );
      Data.Default     := To_Unbounded_String( Get_Text( Sql_Default ) );
      Data.Constraint  := To_Unbounded_String( Get_Text( Sql_Constraint ) );
      Data.Comment     := To_Unbounded_String( Get_Text( Comment ) );

      Data.Primary_Key := Is_Active( Sql_Primary );
      Data.Unique      := Is_Active( Sql_Unique  );

      case Selected( This.Sql_Null ) is
         when SQL_NULL =>
            Data.Is_Null  := True;
            Data.Is_Not_Null := False;
         when SQL_NOT_NULL =>
            Data.Is_Null  := False;
            Data.Is_Not_Null := True;
         when others =>
            Data.Is_Null  := False;
            Data.Is_Not_Null := False;
      end case;
   end Get_result;

   ---------------------
   -- Dialog_Contents --
   ---------------------
   procedure Dialog_Contents(
      This           : in out Object;
      Vbox           : in Gtk_Box ) is
      -- display the row dialog
      Textbox        : Gtk_Box;
      Ebox           : Gtk_Box;
      Fr             : Gtk_Frame;

      Sql_Type       : Gtk_Combo renames This.Sql_Type;
      Sql_Default    : Gtk_Entry renames This.Sql_Default;
      Sql_Constraint : Gtk_Entry renames This.Sql_Constraint;
      Comment        : Gtk_Entry renames This.Comment;

      Data           : Row_Information_Record;
   begin
      Gtk_New(Fr);
      Gtk_New_Vbox( Textbox );
      Add( Fr, Textbox );

      Name_And_Type( This, Textbox, This.Row );
      Default_Value( This, Textbox, This.Row);
      Constraints( This, Textbox, This.Row );

      Entry_Field( Textbox , Comment, "Comment" );

      Show_All( Fr );
      Pack_Start( Vbox, Fr );

      Add_Command( This, "Apply", Cmd_Apply );
      Add_Command( This, "OK",    Cmd_Ok );
   end Dialog_Contents;

   -------------
   -- Command --
   -------------
   procedure Command(
      This           : in out Object;
      Cmd            : in Natural ) is
      -- execute all commands in this dialog
      Name           : Gtk_Entry renames This.Name;
      Sql_Type       : Gtk_Combo renames This.Sql_Type;
      Sql_Default    : Gtk_Entry renames This.Sql_Default;
      Sql_Constraint : Gtk_Entry renames This.SQL_Constraint;
      Comment        : Gtk_Entry renames This.Comment;

      Data           : Row_Information_Record;
   begin
      case Cmd is
         when Dlg_Cmd_Dismiss =>
            Finalize( This );

         when Cmd_Apply =>
            Get_Result( This, Data);
            GUI_Table.Insert_Row( GUI_Table.Object(This.Sheet.all), Data );

         when Cmd_Ok   =>
            Get_Result( This, Data );
            GUI_Table.Insert_Row( GUI_Table.Object(This.Sheet.all), Data );
            Command( This, Dlg_Cmd_Dismiss );

         when others =>
            GUI_Logging.Error("Unknown internal command in gui_addrow.adb");
      end case;

   end Command;

end GUI_AddRow;
