/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "globus_rls_client.h"
#include "globus_rls_rpc.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdarg.h>

static int	rrpc_timeout = 30;
static int	flushbuf(globus_io_handle_t *h, BUFFER *b, char *errmsg);

void
rrpc_initbuf(BUFFER *b)

{
  b->idx = 0;
  b->len = 0;
}

int
rrpc_fillbuf(globus_io_handle_t *h, BUFFER *b, int *rcp, char *errmsg)

{
  if ((*rcp = rrpc_read(h, b->buf, RPCBSIZ, 1, &b->len,
			errmsg)) != GLOBUS_RLS_SUCCESS)
    return -1;
  b->idx = 1;
  return b->buf[0];
}

int
rrpc_getresult(globus_rls_handle_t *h, BUFFER *b, char *errmsg)

{
  int	c;
  int	rc;
  int	i;
  int	trc;

  rc = 0;
  rrpc_initbuf(b);
  while (c = NEXTC(&h->handle, b, &trc, errmsg)) {
    if (c == -1) {
      h->flags |= FH_IOERROR;
      return trc;
    } 
    if (isdigit(c))
      rc = (rc * 10) + (c - '0');
  }
  if (rc == GLOBUS_RLS_SUCCESS)
    return rc;
  for (i = 0; i < MAXERRMSG; i++) {
    if ((c = NEXTC(&h->handle, b, &trc, errmsg)) == -1) {
      h->flags |= FH_IOERROR;
      return trc;
    }
    errmsg[i] = c;
    if (!c)
      break;
  }
  return rc;
}

/*
 * Map r to error string, return GLOBUS_RLS_GLOBUSERR.
 */
int
rrpc_globuserr(char *errmsg, int len, globus_result_t r)

{
  globus_object_t	*eo;
  char			*s;

  if ((eo = globus_error_get(r)) == NULL)
    strncpy(errmsg, "Unknown globus error", len);
  else {
    s = globus_error_print_friendly(eo);
    globus_object_free(eo);
    if (s) {
      strncpy(errmsg, s, len);
      globus_libc_free(s);
    } else
      strncpy(errmsg, "Unknown globus error", len);
  }
  return GLOBUS_RLS_GLOBUSERR;
}

typedef struct {
  globus_mutex_t	mtx;
  globus_cond_t		cond;
  globus_bool_t		done;
  globus_size_t		nb;
  int			rc;
  char			*errmsg;
  int			errmsglen;
} IOMON;

static void	connectcb(void *a, globus_io_handle_t *h, globus_result_t r);
static void	readcb(void *a, globus_io_handle_t *h, globus_result_t r,
		       globus_byte_t *buf, globus_size_t nb);
static void	writecb(void *a, globus_io_handle_t *h, globus_result_t r,
			globus_byte_t *buf, globus_size_t nb);
static void	writevcb(void *a, globus_io_handle_t *h, globus_result_t r,
			 struct iovec *iov, globus_size_t iovcnt,
			 globus_size_t nb);

int
rrpc_connect(char *host, unsigned short port, globus_io_attr_t *attr,
         globus_io_handle_t *h, char *errmsg, int errmsglen)

{
  globus_result_t   r;
  IOMON             mon;
  struct timespec   ts;
  globus_bool_t     timed_out = GLOBUS_FALSE;

  globus_mutex_init(&mon.mtx, GLOBUS_NULL);
  globus_cond_init(&mon.cond, GLOBUS_NULL);
  mon.done = GLOBUS_FALSE;
  mon.rc = GLOBUS_RLS_SUCCESS;
  mon.errmsg = errmsg;
  mon.errmsglen = errmsglen;
  r = globus_io_tcp_register_connect(host, port, attr, connectcb, &mon, h);
  if (r != GLOBUS_SUCCESS) {
    mon.done = GLOBUS_TRUE;
    mon.rc = rrpc_globuserr(errmsg, errmsglen, r);
  }

  globus_mutex_lock(&mon.mtx);
  if (rrpc_timeout) {
    ts.tv_nsec = 0;
    ts.tv_sec = time(0) + rrpc_timeout;
    while (!mon.done && time(0) < ts.tv_sec)
      globus_cond_timedwait(&mon.cond, &mon.mtx, &ts);
    if (!mon.done) {
      /* Flag the timed out condition as TRUE */
      timed_out = GLOBUS_TRUE;
      /* Call async cancel, which will call the connectcb registered above */
      r = globus_io_register_cancel(h, GLOBUS_TRUE, NULL, NULL);
      if (r != GLOBUS_SUCCESS) {
        mon.done = GLOBUS_TRUE; /* Failed cancel, so either way we're done */
        mon.rc = GLOBUS_RLS_TIMEOUT;
      }
    }
  }

  /* Wait for connectcb, which is called on connection _or_ cancel */
  while (!mon.done)
    globus_cond_wait(&mon.cond, &mon.mtx);
  globus_mutex_unlock(&mon.mtx);

  if (mon.rc != GLOBUS_RLS_SUCCESS && timed_out) {
    mon.rc = GLOBUS_RLS_TIMEOUT;
    snprintf(errmsg, MAXERRMSG,
       "globus_io_tcp_register_connect() timed out after %d seconds",rrpc_timeout);
  }

  globus_cond_destroy(&mon.cond);
  globus_mutex_destroy(&mon.mtx);
  return mon.rc;
}

int
rrpc_read(globus_io_handle_t *h, globus_byte_t *buf, globus_size_t max_nb,
	  globus_size_t wait_nb, globus_size_t *nbr, char *errmsg)

{
  globus_result_t	r;
  globus_size_t		try_nb;
  IOMON			mon;
  struct timespec	ts;

  r = globus_io_try_read(h, buf, max_nb, nbr);
  if (r != GLOBUS_SUCCESS)
    return rrpc_globuserr(errmsg, MAXERRMSG, r);
  if (*nbr >= wait_nb)
    return GLOBUS_RLS_SUCCESS;

  try_nb = *nbr;
  globus_mutex_init(&mon.mtx, GLOBUS_NULL);
  globus_cond_init(&mon.cond, GLOBUS_NULL);
  mon.done = GLOBUS_FALSE;
  mon.nb = 0;
  mon.rc = GLOBUS_RLS_SUCCESS;
  mon.errmsg = errmsg;
  mon.errmsglen = MAXERRMSG;
  r = globus_io_register_read(h, buf + try_nb, max_nb - try_nb,
			      wait_nb - try_nb, readcb, &mon);
  if (r != GLOBUS_SUCCESS) {
    mon.done = GLOBUS_TRUE;
    mon.rc = rrpc_globuserr(errmsg, MAXERRMSG, r);
  }

  globus_mutex_lock(&mon.mtx);
  if (rrpc_timeout) {
    ts.tv_nsec = 0;
    ts.tv_sec = time(0) + rrpc_timeout;
    while (!mon.done && time(0) < ts.tv_sec)
      globus_cond_timedwait(&mon.cond, &mon.mtx, &ts);
  } else
    while (!mon.done)
      globus_cond_wait(&mon.cond, &mon.mtx);
  globus_mutex_unlock(&mon.mtx);
  *nbr = mon.nb + try_nb;

  if (!mon.done) {
    globus_io_cancel(h, GLOBUS_FALSE);
    mon.rc = GLOBUS_RLS_TIMEOUT;
    snprintf(errmsg, MAXERRMSG,
	"globus_io_register_read() timed out after %d seconds", rrpc_timeout);
  }

  globus_cond_destroy(&mon.cond);
  globus_mutex_destroy(&mon.mtx);
  return mon.rc;
}

int
rrpc_get_timeout()

{
  return rrpc_timeout;
}

void
rrpc_set_timeout(int timeout)

{
  rrpc_timeout = timeout;
}

int
rrpc_write(globus_io_handle_t *h, globus_byte_t *buf, globus_size_t nb,
	   globus_size_t *nbw, char *errmsg)

{
  globus_result_t	r;
  globus_size_t		try_nb;
  IOMON			mon;
  struct timespec	ts;

  r = globus_io_try_write(h, buf, nb, nbw);
  if (r != GLOBUS_SUCCESS)
    return rrpc_globuserr(errmsg, MAXERRMSG, r);
  if (*nbw == nb)
    return GLOBUS_RLS_SUCCESS;

  try_nb = *nbw;
  globus_mutex_init(&mon.mtx, GLOBUS_NULL);
  globus_cond_init(&mon.cond, GLOBUS_NULL);
  mon.done = GLOBUS_FALSE;
  mon.nb = 0;
  mon.rc = GLOBUS_RLS_SUCCESS;
  mon.errmsg = errmsg;
  mon.errmsglen = MAXERRMSG;
  r = globus_io_register_write(h, buf + try_nb, nb - try_nb, writecb, &mon);
  if (r != GLOBUS_SUCCESS) {
    mon.done = GLOBUS_TRUE;
    mon.rc = rrpc_globuserr(errmsg, MAXERRMSG, r);
  }

  globus_mutex_lock(&mon.mtx);
  if (rrpc_timeout) {
    ts.tv_nsec = 0;
    ts.tv_sec = time(0) + rrpc_timeout;
    while (!mon.done && time(0) < ts.tv_sec)
      globus_cond_timedwait(&mon.cond, &mon.mtx, &ts);
  } else
    while (!mon.done)
      globus_cond_wait(&mon.cond, &mon.mtx);
  globus_mutex_unlock(&mon.mtx);
  *nbw = mon.nb + try_nb;

  if (!mon.done) {
    globus_io_cancel(h, GLOBUS_FALSE);
    mon.rc = GLOBUS_RLS_TIMEOUT;
    snprintf(errmsg, MAXERRMSG,
	"globus_io_register_write() timed out after %d seconds", rrpc_timeout);
  }

  globus_cond_destroy(&mon.cond);
  globus_mutex_destroy(&mon.mtx);
  return mon.rc;
}

int
rrpc_writev(globus_io_handle_t *h, struct iovec *iov, globus_size_t iovcnt,
	    globus_size_t *nbw, char *errmsg)

{
  globus_result_t	r;
  IOMON			mon;
  struct timespec	ts;

  globus_mutex_init(&mon.mtx, GLOBUS_NULL);
  globus_cond_init(&mon.cond, GLOBUS_NULL);
  mon.done = GLOBUS_FALSE;
  mon.nb = 0;
  mon.rc = GLOBUS_RLS_SUCCESS;
  mon.errmsg = errmsg;
  mon.errmsglen = MAXERRMSG;
  r = globus_io_register_writev(h, iov, iovcnt, writevcb, &mon);
  if (r != GLOBUS_SUCCESS) {
    mon.done = GLOBUS_TRUE;
    mon.rc = rrpc_globuserr(errmsg, MAXERRMSG, r);
  }

  globus_mutex_lock(&mon.mtx);
  if (rrpc_timeout) {
    ts.tv_nsec = 0;
    ts.tv_sec = time(0) + rrpc_timeout;
    while (!mon.done && time(0) < ts.tv_sec)
      globus_cond_timedwait(&mon.cond, &mon.mtx, &ts);
  } else
    while (!mon.done)
      globus_cond_wait(&mon.cond, &mon.mtx);
  globus_mutex_unlock(&mon.mtx);
  *nbw = mon.nb;

  if (!mon.done) {
    globus_io_cancel(h, GLOBUS_FALSE);
    mon.rc = GLOBUS_RLS_TIMEOUT;
    snprintf(errmsg, MAXERRMSG,
	"globus_io_register_writev() timed out after %d seconds",rrpc_timeout);
  }

  globus_cond_destroy(&mon.cond);
  globus_mutex_destroy(&mon.mtx);
  return mon.rc;
}

int
rrpc_bufwrite(globus_io_handle_t *h, BUFFER *b, char *buf, int nb, int flush,
	      char *errmsg)

{
  int	rc;

  if (nb > RPCBSIZ) {
    sprintf(errmsg, "Buffer not large enough (%d) to hold %d bytes",
	    RPCBSIZ, nb);
    return GLOBUS_RLS_BADARG;
  }
  if (b->len + nb > RPCBSIZ)
    if ((rc = flushbuf(h, b, errmsg)) != GLOBUS_RLS_SUCCESS)
      return rc;
  memcpy(&b->buf[b->len], buf, nb);
  b->len += nb;
  if (flush)
    return flushbuf(h, b, errmsg);
  return GLOBUS_RLS_SUCCESS;
}

int
rrpc_bufprintf(globus_io_handle_t *h, BUFFER *b, int flush, char *errmsg,
	       char *fmt, ...)

{
  va_list	ap;
  char		buf[RPCBSIZ+1];
  int		len;

  va_start(ap, fmt);
  len = vsnprintf(buf, RPCBSIZ+1, fmt, ap);
  va_end(ap);
  return rrpc_bufwrite(h, b, buf, len, flush, errmsg);
}

static int
flushbuf(globus_io_handle_t *h, BUFFER *b, char *errmsg)

{
  int		rc;
  globus_size_t	nb;

  rc = rrpc_write(h, b->buf, b->len, &nb, errmsg);
  if (rc != GLOBUS_RLS_SUCCESS)
    return rc;
  if (nb != b->len) {
    sprintf(errmsg, "Attempted to write %d bytes, only wrote %d", b->len, nb);
    return GLOBUS_RLS_GLOBUSERR;
  }
  b->len = 0;
  return GLOBUS_RLS_SUCCESS;
}

static void
connectcb(void *a, globus_io_handle_t *h, globus_result_t r)

{
  IOMON	*monp = (IOMON *) a;

  globus_mutex_lock(&monp->mtx);
  if (r != GLOBUS_SUCCESS)
    monp->rc = rrpc_globuserr(monp->errmsg, monp->errmsglen, r);
  monp->done = GLOBUS_TRUE;
  globus_cond_signal(&monp->cond);
  globus_mutex_unlock(&monp->mtx);
}

static void
readcb(void *a, globus_io_handle_t *h, globus_result_t r,
	globus_byte_t *buf, globus_size_t nb)

{
  IOMON	*monp = (IOMON *) a;

  globus_mutex_lock(&monp->mtx);
  monp->nb = nb;
  if (r != GLOBUS_SUCCESS)
    monp->rc = rrpc_globuserr(monp->errmsg, monp->errmsglen, r);
  monp->done = GLOBUS_TRUE;
  globus_cond_signal(&monp->cond);
  globus_mutex_unlock(&monp->mtx);
}

static void
writecb(void *a, globus_io_handle_t *h, globus_result_t r,
	globus_byte_t *buf, globus_size_t nb)

{
  IOMON	*monp = (IOMON *) a;

  globus_mutex_lock(&monp->mtx);
  monp->nb = nb;
  if (r != GLOBUS_SUCCESS)
    monp->rc = rrpc_globuserr(monp->errmsg, monp->errmsglen, r);
  monp->done = GLOBUS_TRUE;
  globus_cond_signal(&monp->cond);
  globus_mutex_unlock(&monp->mtx);
}

static void
writevcb(void *a, globus_io_handle_t *h, globus_result_t r,
	 struct iovec *iov, globus_size_t iovcnt, globus_size_t nb)

{
  IOMON	*monp = (IOMON *) a;

  globus_mutex_lock(&monp->mtx);
  monp->nb = nb;
  if (r != GLOBUS_SUCCESS)
    monp->rc = rrpc_globuserr(monp->errmsg, monp->errmsglen, r);
  monp->done = GLOBUS_TRUE;
  globus_cond_signal(&monp->cond);
  globus_mutex_unlock(&monp->mtx);
}
