/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * util.c
 *
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-packages/nexus/source/nexus/util.c,v 1.37 2006/01/19 05:57:06 mlink Exp $";

#include "internal.h"

/* This is for the access() call */
#include <sys/file.h>

static void blocking_error_callback(
     void *                        arg,
     globus_io_handle_t *          handle,
     globus_result_t               result,
     globus_byte_t *               buf,
     globus_size_t                 nbytes);
/*
 * _nx_copy_string()
 *
 * Copy the string into malloced space and return it.
 *
 * Reminder: This function returns a pointer to malloced memory, so
 * don't forget to use NexusFree() on it...
 */
char *_nx_copy_string(char *s)
{
    char *rc;

    NexusMalloc(_nx_copy_string(), rc, char *, (strlen(s) + (size_t)(1)));
    strcpy(rc, s);
    return (rc);
} /* _nx_copy_string() */


/*
 * nexus_ids()
 *
 * Return context_id, and thread_id for the calling thread.
 */
void nexus_ids(int *context_id, int *thread_id)
{
    _nx_context_id(context_id);
    /*
    _nx_thread_id(thread_id);
    */
} /* nexus_ids() */


/*
 * _nx_find_attribute()
 */
char *_nx_find_attribute(char *attr, char *search_string, char separator)
{
    char *pos;
    char *tmp;

    tmp = search_string;
    while ((pos = strstr(tmp, attr)))
    {
        if (   (*(pos-1)==separator)
	    && (   (*(pos+strlen(attr))=='=')
		|| (*(pos+strlen(attr))==separator) ) )
        {
	    char *value;
	    char *i;
	    int j;

            /* got right place */
	    NexusMalloc(_nx_find_attribute(),
		        value,
		        char *,
		        sizeof(char) * /* some arbitrary size */ 100);
	    for (j = 0, i = pos+strlen(attr); *i; i++)
	    {
		if (*i == separator)
		{
		    break;
		}
	        if (!isspace(*i) && *i != '=')
	        {
		    value[j++] = *i;
	        }
	    }
	    value[j] = '\0';
	    return (value);
        }
	else
	{
	    tmp = pos + strlen(attr);
	}
    }
    
    return NULL;
} /* _nx_find_attribute() */


/*
 * _nx_get_next_value()
 */
void _nx_get_next_value(char *string,
			char separator,
			char **next,
			char **value)
{
    if (!*string)
    {
	*value = NULL;
	*next = NULL;
    }
    *value = _nx_copy_string(string);
    if ((*next = strchr(*value, separator)) != (char *) NULL)
    {
	**next = '\0';
	*next = strchr(string, separator);
	(*next)++;
    }
    else if ((*next = strchr(*value, '\0')) != (char *) NULL)
    {
	*next = NULL;
    }
} /* _nx_get_next_value() */


/*
 * _nx_hex_encode_byte_array()
 *
 * Encode the first 'length' bytes of 'bytes' as hex digits, placing
 * the result in 'hex'.
 *
 * 'hex' must be an array of at least length: ((2 * length) + 1)
 */
void _nx_hex_encode_byte_array(nexus_byte_t *bytes,
			       int length,
			       char *hex)
{
    int i;
    char buf[4];
    hex[0] = '\0';
    nexus_stdio_lock();
    for (i = 0; i < length; i++)
    {
	if (bytes[i] <= 0xF)
	{
	    sprintf(buf, "0%1x", (int) bytes[i]);
	}
	else
	{
	    sprintf(buf, "%2x", (int) bytes[i]);
	}
	strcat(hex, buf);
    }
    nexus_stdio_unlock();
} /* _nx_hex_encode_byte_array() */


/*
 * _nx_hex_decode_byte_array()
 *
 * Decode 'length' bytes from the 'hex' array (encoded in hex values)
 * into the 'bytes' array.
 *
 * 'bytes' must be an array of at least length: (length)
 */
void _nx_hex_decode_byte_array(char *hex,
			       int length,
			       nexus_byte_t *bytes)
{
    int i, j;
    char *h = hex;
    nexus_stdio_lock();
    for (i = 0; i < length; i++)
    {
	sscanf(h, "%2x", &j);
	bytes[i] = (nexus_byte_t) j;
	h += 2;
    }
    nexus_stdio_unlock();
} /* _nx_hex_decode_byte_array() */


#ifndef HAVE_STRTOUL
/*
 * SunOS 4.1.x does not have strtoul().
 * This version was snarfed from FreeBSD.
 */

/*
 * Copyright (c) 1990, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#include <limits.h>

/*
 * Convert a string to an unsigned long integer.
 *
 * Ignores `locale' stuff.  Assumes that the upper and lower case
 * alphabets and digits are each contiguous.
 */
unsigned long strtoul(const char *nptr,
		      char **endptr,
		      int base)
{
	register const char *s = nptr;
	register unsigned long acc;
	register int c;
	register unsigned long cutoff;
	register int neg = 0, any, cutlim;

	/*
	 * See strtol for comments as to the logic used.
	 */
	do {
		c = *s++;
	} while (isspace(c));
	if (c == '-') {
		neg = 1;
		c = *s++;
	} else if (c == '+')
		c = *s++;
	if ((base == 0 || base == 16) &&
	    c == '0' && (*s == 'x' || *s == 'X')) {
		c = s[1];
		s += 2;
		base = 16;
	}
	if (base == 0)
		base = c == '0' ? 8 : 10;
	cutoff = (unsigned long)ULONG_MAX / (unsigned long)base;
	cutlim = (unsigned long)ULONG_MAX % (unsigned long)base;
	for (acc = 0, any = 0;; c = *s++) {
		if (isdigit(c))
			c -= '0';
		else if (isalpha(c))
			c -= isupper(c) ? 'A' - 10 : 'a' - 10;
		else
			break;
		if (c >= base)
			break;
		if (any < 0 || acc > cutoff || (acc == cutoff && c > cutlim))
			any = -1;
		else {
			any = 1;
			acc *= base;
			acc += c;
		}
	}
	if (any < 0) {
		acc = ULONG_MAX;
		errno = ERANGE;
	} else if (neg)
		acc = -acc;
	if (endptr != 0)
		*endptr = (char *)(any ? s - 1 : nptr);
	return (acc);
} /* strtoul() */

#endif /* !HAVE_STRTOUL */


typedef struct _nx_barrier_t
{
    nexus_mutex_t    mutex;
    nexus_cond_t     cond;
    nexus_bool_t     done;
    globus_result_t  rc;
} nx_barrier_t;

/*
 * blocking_write_callback()
 */
static void blocking_write_callback(
     void *                        arg,
     globus_io_handle_t *          handle,
     globus_result_t               result,
     globus_byte_t *               buf,
     globus_size_t                 nbytes)
{
    nx_barrier_t *barrier = (nx_barrier_t *)arg;
    
    if(result != GLOBUS_SUCCESS)
    {
        blocking_error_callback(arg, handle, result, buf, nbytes); 
	return;
    }

    nexus_mutex_lock(&barrier->mutex);
    barrier->done = NEXUS_TRUE;
    nexus_cond_signal(&barrier->cond);
    nexus_mutex_unlock(&barrier->mutex);
} /* blocking_write_callback() */


/*
 * blocking_read_callback()
 */
static void 
blocking_read_callback(
     void *                        arg,
     globus_io_handle_t *          handle,
     globus_result_t               result,
     globus_byte_t *               buf,
     globus_size_t                 nbytes)
{
    nx_barrier_t *barrier = (nx_barrier_t *)arg;

    if(result != GLOBUS_SUCCESS)
    {
        blocking_error_callback(arg, handle, result, buf, nbytes); 
	return;
    }

    nexus_mutex_lock(&barrier->mutex);
    barrier->done = NEXUS_TRUE;
    nexus_cond_signal(&barrier->cond);
    nexus_mutex_unlock(&barrier->mutex);
} /* blocking_read_callback() */


/*
 * blocking_error_callback()
 */
static void blocking_error_callback(
     void *                        arg,
     globus_io_handle_t *          handle,
     globus_result_t               result,
     globus_byte_t *               buf,
     globus_size_t                 nbytes)
{
    nx_barrier_t *barrier = (nx_barrier_t *)arg;
    nexus_mutex_lock(&barrier->mutex);
    barrier->done = NEXUS_TRUE;
    barrier->rc = result;
    nexus_cond_signal(&barrier->cond);
    nexus_mutex_unlock(&barrier->mutex);
} /* blocking_error_callback() */


/*
 * _nx_write_blocking()
 */
globus_result_t 
_nx_write_blocking(globus_io_handle_t * handle, 
		       void * buf, 
		       globus_size_t size)
{
    nx_barrier_t write_barrier;

    nexus_mutex_init(&write_barrier.mutex, (nexus_mutexattr_t *)NULL);
    nexus_cond_init(&write_barrier.cond, (nexus_condattr_t *)NULL);
    write_barrier.done = NEXUS_FALSE;
    write_barrier.rc = 0;
/*    
    nexus_fd_register_for_write(fd,
				buf,
				size,
				blocking_write_callback,
				blocking_error_callback,
				(void *) &write_barrier);
*/

    globus_io_register_write(handle,
			     (globus_byte_t *)buf,
			     size,
			     blocking_write_callback,
			     (void *) &write_barrier);

    nexus_mutex_lock(&write_barrier.mutex);
    while(!write_barrier.done)
    {
        nexus_cond_wait(&write_barrier.cond, &write_barrier.mutex);
    }
    nexus_mutex_unlock(&write_barrier.mutex);

    nexus_mutex_destroy(&write_barrier.mutex);
    nexus_cond_destroy(&write_barrier.cond);

    return write_barrier.rc;
} /* _nx_write_blocking() */

/*
 * _nx_read_blocking()
 */
globus_result_t
_nx_read_blocking(globus_io_handle_t * handle, void *buf, size_t size)
{
    nx_barrier_t read_barrier;

    nexus_mutex_init(&read_barrier.mutex, (nexus_mutexattr_t *)NULL);
    nexus_cond_init(&read_barrier.cond, (nexus_condattr_t *)NULL);
    read_barrier.done = NEXUS_FALSE;
    read_barrier.rc = 0;

/*
    nexus_fd_register_for_read(fd,
			       buf,
			       size,
			       size,
			       blocking_read_callback,
			       blocking_error_callback,
			       (void *) &read_barrier);
*/

/* read_barrier is local ???*/
    globus_io_register_read(handle,
			    buf,
			    size,
			    size,
			    blocking_read_callback,
			    (void *) &read_barrier);

    nexus_mutex_lock(&read_barrier.mutex);
    while(!read_barrier.done)
    {
        nexus_cond_wait(&read_barrier.cond, &read_barrier.mutex);
    }
    nexus_mutex_unlock(&read_barrier.mutex);

    nexus_mutex_destroy(&read_barrier.mutex);
    nexus_cond_destroy(&read_barrier.cond);

    return read_barrier.rc;
} /* _nx_blocking_read() */

