/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Nexus
 * Authors:     James Stepanek
 *              Aerospace
 *
 * qoslib.c	- QoSlib module
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-packages/nexus/source/nexus/qoslib.c,v 1.8 2006/01/19 05:57:06 mlink Exp $";

#include "internal.h"

#include <sys/socket.h>
#include <netinet/in.h>

#include "nexus_resource.h"

#include "qoslib.h"

static int _sizeof_socket(qos_request_obj*, int);
static int _sizeof_process(qos_request_obj*, int);
static int _sizeof_thread(qos_request_obj*, int);
static int _sizeof_memory(qos_request_obj*, int);

static void _put_socket(nexus_buffer_t*,  qos_request_obj*);
static void _put_process(nexus_buffer_t*,  qos_request_obj*);
static void _put_thread(nexus_buffer_t*,  qos_request_obj*);
static void _put_memory(nexus_buffer_t*,  qos_request_obj*);

static void _get_socket(nexus_buffer_t*,  qos_request_obj*);
static void _get_process(nexus_buffer_t*,  qos_request_obj*);
static void _get_thread(nexus_buffer_t*,  qos_request_obj*);
static void _get_memory(nexus_buffer_t*,  qos_request_obj*);

static struct _request_funcs {
  int (*size)	(qos_request_obj*, int);
  void (*put)	(nexus_buffer_t*, qos_request_obj*);
  void (*get)	(nexus_buffer_t*, qos_request_obj*);
} request_funcs[NEXUS_RESOURCE_NTYPES] = {
  /* size */        /* put */	  /* get */
  {_sizeof_socket,  _put_socket,  _get_socket,  }, /* SOCKET */
  {_sizeof_process, _put_process, _get_process, }, /* PROCESS */
  {_sizeof_thread,  _put_thread,  _get_thread,  }, /* THREAD */
  {_sizeof_memory,  _put_memory,  _get_memory,  }, /* MEMORY */
};

static int sizeof_spec(qos_spec*, int);
static void spec_put(nexus_buffer_t*, qos_spec*);
static void spec_get(nexus_buffer_t*, qos_spec*);

static int _sizeof_spec_is(qos_spec*, int);
static void _spec_put_is(nexus_buffer_t*, qos_spec*);
static void _spec_get_is(nexus_buffer_t*, qos_spec*);

static struct _spec_funcs {
  int (*size)	(qos_spec*, int);
  void (*put)	(nexus_buffer_t*, qos_spec*);
  void (*get)	(nexus_buffer_t*, qos_spec*);
} spec_funcs[QOS_SPEC_NTYPES] = {
  /* size */        /* put */     /* get */
  {NULL,            NULL,         NULL,          }, /* NULL */
  {NULL,            NULL,         NULL,          }, /* DEFAULT */
  {_sizeof_spec_is, _spec_put_is, _spec_get_is,  }, /* IS */
};


/*
 * qos_sizeof_request()
 */
int qos_sizeof_request(qos_request_obj* request_obj, int count) {
  if (request_funcs[request_obj->who->type].size) 
    return ((request_funcs[request_obj->who->type].size)(request_obj, count));
  return (0);
}

/*
 * qos_request_put()
 */
void qos_request_put( nexus_buffer_t *buffer, qos_request_obj *request_obj ) {
  if  ( request_funcs[request_obj->who->type].put )
    (request_funcs[request_obj->who->type].put)( buffer, request_obj );
}

/*
 * qos_request_get()
 */
void qos_request_get( nexus_buffer_t *buffer, qos_request_obj *request_obj ) {
  if  ( request_funcs[request_obj->who->type].get )
    (request_funcs[request_obj->who->type].get)( buffer, request_obj );
}

/* SOCKET */

static int _sizeof_socket(qos_request_obj *request_obj, int count) {
  qos_request_socket * req = (qos_request_socket *)request_obj;

  return ((count * (nexus_sizeof_byte(sizeof(qos_callback *))
		    + nexus_sizeof_int(1)))
	  + (spec_funcs[req->spec_type].size ? 
	     spec_funcs[req->spec_type].size(&req->spec, count) : 0));

}

static void _put_socket(nexus_buffer_t *buffer, qos_request_obj *request_obj) {
  qos_request_socket * req = (qos_request_socket *)request_obj;

  nexus_put_u_long( buffer, (unsigned long*)&req->proc, 1 );
  nexus_put_int( buffer, (int *)&req->spec_type, 1);
  if  ( spec_funcs[req->spec_type].put ) {
    (spec_funcs[req->spec_type].put)( buffer, &req->spec );
  }

}

static void _get_socket(nexus_buffer_t *buffer, qos_request_obj *request_obj) {
  qos_request_socket * req = (qos_request_socket *)request_obj;

  nexus_get_u_long( buffer, (unsigned long*)&req->proc, 1 );
  nexus_get_int( buffer, (int *)&req->spec_type, 1);

  if  ( spec_funcs[req->spec_type].get ) {
    (spec_funcs[req->spec_type].get)( buffer, &req->spec );
  }

}


/* PROCESS */

static int _sizeof_process(qos_request_obj *request_obj, int count) {

  return (count * ( nexus_sizeof_byte(sizeof(qos_callback *))

#ifdef QOS_USE_POSIX_SCHED
		    + nexus_sizeof_byte( sizeof(qos_sched_class_t) )
		    + nexus_sizeof_byte( sizeof(qos_sched_priority_t) )
#endif /* QOS_USE_POSIX_SCHED */

    ));
}

static void _put_process(nexus_buffer_t *buffer, 
			 qos_request_obj *request_obj) {
  qos_request_process * req = (qos_request_process *)request_obj;

  nexus_put_u_long( buffer, (unsigned long*)&req->proc, 1 );

#ifdef QOS_USE_POSIX_SCHED
  nexus_put_byte( buffer, (nexus_byte_t*)&req->class,
		  sizeof(qos_sched_class_t) );

  nexus_put_byte( buffer, (nexus_byte_t*)&req->priority,
		  sizeof(qos_sched_priority_t) );
#endif /* QOS_USE_POSIX_SCHED */

}

static void _get_process(nexus_buffer_t *buffer, 
			 qos_request_obj *request_obj) {
  qos_request_process * req = (qos_request_process *)request_obj;

  nexus_get_u_long( buffer, (unsigned long*)&req->proc, 1 );

#ifdef QOS_USE_POSIX_SCHED
  nexus_get_byte( buffer, (nexus_byte_t*)&req->class,
		  sizeof(qos_sched_class_t) );

  nexus_get_byte( buffer, (nexus_byte_t*)&req->priority,
		  sizeof(qos_sched_priority_t) );
#endif /* QOS_USE_POSIX_SCHED */

}

/* THREAD */

static int _sizeof_thread(qos_request_obj *request_obj, int count) {

  return ( count * ( nexus_sizeof_byte(sizeof(qos_callback *))

#ifdef QOS_USE_POSIX_SCHED
		    + nexus_sizeof_byte( sizeof(qos_sched_class_t) )
		    + nexus_sizeof_byte( sizeof(qos+sched_priority_t) )
#endif /* QOS_USE_POSIX_SCHED */
    ));
}

static void _put_thread(nexus_buffer_t *buffer, qos_request_obj *request_obj) {
  qos_request_thread * req = (qos_request_thread *)request_obj;

  nexus_put_u_long( buffer, (unsigned long*)&req->proc, 1 );

#ifdef QOS_USE_POSIX_SCHED
  nexus_put_byte( buffer, (nexus_byte_t*)&req->class,
		  sizeof(qos_sched_class_t) );

  nexus_put_byte( buffer, (nexus_byte_t*)&req->priority, 
		  sizeof(qos_sched_priority_t) );
#endif /* QOS_USE_POSIX_SCHED */

}

static void _get_thread(nexus_buffer_t *buffer, qos_request_obj *request_obj) {
  qos_request_thread * req = (qos_request_thread *)request_obj;

  nexus_get_u_long( buffer, (unsigned long*)&req->proc, 1 );

#ifdef QOS_USE_POSIX_SCHED
  nexus_get_byte( buffer, (nexus_byte_t*)&req->class
		  sizeof(qos_sched_class_t) );

  nexus_get_byte( buffer, (nexus_byte_t*)&req->priority,
		  sizeof(qos_sched_priority_t) );
#endif /* QOS_USE_POSIX_SCHED */

}


/* MEMORY */

static int _sizeof_memory(qos_request_obj *request_obj, int count) {

  return (count * ( nexus_sizeof_byte( sizeof(qos_callback *) )
		    + nexus_sizeof_byte( sizeof(nexus_bool_t) ) ) );
}

static void _put_memory(nexus_buffer_t *buffer, qos_request_obj *request_obj) {
  qos_request_memory * req = (qos_request_memory *)request_obj;

  nexus_put_u_long( buffer, (unsigned long*)&req->proc, 1 );

  nexus_put_byte( buffer, (nexus_byte_t*)&req->lock,  sizeof(nexus_bool_t) );
}

static void _get_memory(nexus_buffer_t *buffer, qos_request_obj *request_obj) {
  qos_request_memory * req = (qos_request_memory *)request_obj;

  nexus_get_u_long( buffer, (unsigned long*)&req->proc, 1);

  nexus_get_byte( buffer, (nexus_byte_t*)&req->lock, sizeof(nexus_bool_t) );
}


/* SPEC_IS */

static int _sizeof_spec_is(qos_spec *spec, int count) {

  return (nexus_sizeof_u_int(count) 
	  + nexus_sizeof_float(count * 3)
	  + nexus_sizeof_u_long(count * 2));
}


static void _spec_put_is( nexus_buffer_t *buffer, qos_spec *spec ) {

  nexus_put_u_int(buffer, &spec->is.service, 1);
  nexus_put_float(buffer, &spec->is.r, 1);
  nexus_put_float(buffer, &spec->is.b, 1);
  nexus_put_float(buffer, &spec->is.p, 1);
  nexus_put_u_long(buffer, &spec->is.m, 1);
  nexus_put_u_long(buffer, &spec->is.M, 1);
}


static void _spec_get_is( nexus_buffer_t *buffer, qos_spec *spec ) {

  nexus_get_u_int(buffer, &spec->is.service, 1);
  nexus_get_float(buffer, &spec->is.r, 1);
  nexus_get_float(buffer, &spec->is.b, 1);
  nexus_get_float(buffer, &spec->is.p, 1);
  nexus_get_u_long(buffer, &spec->is.m, 1);
  nexus_get_u_long(buffer, &spec->is.M, 1);
}
