/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/******************************************************************************
Description:

  UDP/IP protocol module

CVS Information:

  $Source: /home/globdev/CVS/globus-packages/nexus/source/nexus/pr_udp.c,v $
  $Date: 2006/01/19 05:57:06 $
  $Revision: 1.101 $
  $State: Exp $
  $Author: mlink $
******************************************************************************/

/******************************************************************************
		  Define pre-include configuration parameters
******************************************************************************/
/*
#define USE_TRACEPOINT
*/

/******************************************************************************
			     Include header files
******************************************************************************/
#include "internal.h"
#include "globus_io.h"
#include <assert.h>

#if defined(TARGET_ARCH_AIX)
#define netlen_t size_t
#else
#define netlen_t int
#endif

#if defined(HAVE_UDP_PROTO)

/*
 * Some OS builds don't have the definitions necessary to build the multicast
 * code; others, like Cygwin, have the correct defintions but fail if you
 * attempt to use them :-(.
 */
#if defined(IP_MULTICAST_TTL) && !defined(TARGET_ARCH_CYGWIN)
#    define IP_MULTICAST_ENABLED 1
#else
#    define IP_MULTICAST_ENABLED 0
#endif

#if defined(TARGET_ARCH_LINUX)
#   include <sys/socket.h>
#   include <netinet/in.h>
#   if !defined(IP_DEFAULT_MULTICAST_TTL)
#       define IP_DEFAULT_MULTICAST_TTL 1
#   endif
#else
#   include <sys/socket.h>
#   include <netinet/in.h>
#   if defined(HAVE_NETINET_TCP)
#       include <netinet/tcp.h>
#   endif
#endif
#include <netdb.h>

#include <fcntl.h>

#ifdef BUILD_RESOURCE
#include "nexus_resource.h"
#endif 

#ifdef USE_TRACEPOINT
#   include "tracepoint.h"
#else
#   define tracepoint(str)
#endif /* USE_TRACEPOINT */


/******************************************************************************
		       Define module specific constants
******************************************************************************/

/*
 * NEXUS_UDP_DIRECT_CUSTOM_MAX_SIZE
 */
#ifndef NEXUS_UDP_DIRECT_CUSTOM_MIN_SIZE
#define NEXUS_UDP_DIRECT_CUSTOM_MIN_SIZE 2001
#endif


/*
 * GLOBUS_L_NEXUS_UDP_PROTOCOL_VERSION
 *
 * The version of this protocol module's wire protocol.  If you change
 * this module wire protocol, bump this version number.
 */
#define GLOBUS_L_NEXUS_UDP_PROTOCOL_VERSION (1 + GLOBUS_I_NEXUS_BUFFER_VERSION)

/*
 * GLOBUS_L_NEXUS_UDP_MI_PROTO_VERSION
 *
 * The version of this protocol module's mi_proto.  If you change
 * the contents of this module's mi_proto, bump this version number.
 */
#define GLOBUS_L_NEXUS_UDP_MI_PROTO_VERSION  1

/*
 * Only one thread is allowed to be in the udp code (and thus
 * mucking with data structures) at a time.
 */
#ifdef BUILD_LITE
#define udp_enter()
#define udp_exit()
#else  /* BUILD_LITE */
#define udp_enter()	nexus_mutex_lock(&udp_mutex);
#define udp_exit()	nexus_mutex_unlock(&udp_mutex);
#endif /* BUILD_LITE */

static nexus_bool_t	udp_done;
static nexus_mutex_t	udp_mutex;

#define udp_fatal udp_exit(); globus_fatal

/*
 * Other useful defines
 */

#define UDP_HDR_SIZE             4

/*
 * TODO: Figure out if there is a more dynamic way to determine
 * what this should be.  For now keep it at 1024
 */
#define UDP_MAX_TRANSMISSION_UNIT	8192

/*
 * The largest message size is maximum packet sequence number (as packed
 * into the udp message header), times the maximum payload per packet.
 * The header currently uses a 2 byte sequence number.
 */
#define FLAG_MASK	        0xf0
#define OPEN_FLAG		0x80

#define NEXUS_PROTO_NAME_UDP "udp"
#define UDP_PROTOCOL_INFO _nx_pr_udp_info
#define UDP_INTERFACE_NAME_KEY "udp_interface"

#ifndef SO_REUSEPORT
#   define SO_REUSEPORT SO_REUSEADDR
#endif

/******************************************************************************
			       Type definitions
******************************************************************************/

/*
 * udp_outgoing_t
 *
 * This is an overload of nexus_proto_t.  It adds the
 * udp specific information to that structure.
 */
struct udp_outgoing_s
{
    nexus_proto_type_t		type;
    nexus_proto_funcs_t *	funcs;
    int				version;
    unsigned long		direct_custom_min_size;
    unsigned long		direct_custom_max_size;
    unsigned long		direct_pointer_min_size;
    unsigned long		direct_pointer_max_size;
    nexus_bool_t		can_use_iovec;
    unsigned long		reserved_header_size;
    struct udp_outgoing_s *	next;
    char *			host;
    unsigned short		port;
    globus_bool_t		multi_sender;
    globus_bool_t		local_hack;
    struct sockaddr_in		addr;
    globus_io_handle_t          handle;
    unsigned short              l_port;
    globus_io_attr_t            attr;
    int				state;
    int				new_state;
    int				fault_code;
    int				reference_count;
    char			read_buf[1];
#ifdef BUILD_RESOURCE
    nexus_resource_name_t	resource_name;
#endif /* BUILD_RESOURCE */
};

enum
{
  OUTGOING_STATE_UNOPENED                = 0,
  OUTGOING_STATE_OPEN                    = 1, 
  OUTGOING_STATE_CLOSING                 = 2,
  OUTGOING_STATE_CLOSED                  = 3,
  OUTGOING_STATE_WRITE_ERROR             = 4,
  OUTGOING_STATE_FAULT                   = 5
};


/*
 * udp_incoming_t
 *
 * One of these is allocated for each incoming file descriptor.
 * It stores partial incoming messages until they are complete
 * and can be handled or queued.
 */
struct udp_incoming_s
{
    globus_io_handle_t          handle;
    globus_io_attr_t            attr;
    unsigned short              port;
    char *                      host;
    int				state;
    int				format;
    unsigned long		msg_size;
    unsigned int                expected_seq_num;
    char                        tmp_storage[UDP_HDR_SIZE];
    size_t			nbytes_read;
    unsigned long		nbytes_parsed;
    unsigned long		nbytes_unparsed;
    unsigned long		storage_size;
    nexus_byte_t *		storage;
    nexus_byte_t *		current;
    unsigned long		save_storage_size;
    nexus_byte_t *		save_storage;
    struct sockaddr             from_addr;
    int                         from_len;
    nexus_bool_t		dispatch_in_progress;
    struct globus_nexus_buffer_s *	dispatch_q_head;
    struct globus_nexus_buffer_s *	dispatch_q_tail;
    nexus_endpoint_t *		endpoint;
#ifdef BUILD_RESOURCE
    nexus_resource_name_t	resource_name;
#endif /* BUILD_RESOURCE */
};

typedef struct udp_outgoing_s	udp_outgoing_t;
typedef struct udp_incoming_s	udp_incoming_t;

/******************************************************************************
			    Module specific macros
******************************************************************************/

/*
 * Some useful queue macros
 */
#define Enqueue(Qhead, Qtail, Item) \
{ \
    if (Qhead) \
    { \
	(Qtail)->next = (Item); \
	(Qtail) = (Item); \
    } \
    else \
    { \
	(Qhead) = (Qtail) = (Item); \
    } \
}

#define Dequeue(Qhead, Qtail, Item) \
{ \
    (Item) = (Qhead); \
    (Qhead) = (Qhead)->next; \
}

#define QueueNotEmpty(Qhead)	(Qhead)

typedef struct globus_l_nexus_udp_outgoing_table_entry_s
{
    udp_outgoing_t *                                     outgoing;
    struct globus_l_nexus_udp_outgoing_table_entry_s *   next;
} globus_l_nexus_udp_outgoing_table_entry_t;

#define OUTGOING_TABLE_SIZE 149

static globus_l_nexus_udp_outgoing_table_entry_t  outgoing_table[OUTGOING_TABLE_SIZE];
/******************************************************************************
			  Module specific prototypes
******************************************************************************/
static udp_incoming_t *
globus_l_nexus_udp_incoming_find(char * host,
				 unsigned short port);
static void 
globus_l_nexus_udp_outgoing_table_insert(udp_outgoing_t *outgoing);

static void
globus_l_nexus_udp_outgoing_table_remove(
     udp_outgoing_t *                               outgoing);

static int 
globus_l_nexus_udp_outgoing_table_hash(char *host, unsigned short port);

static udp_outgoing_t *
globus_l_nexus_udp_outgoing_table_lookup(char *host, unsigned short port);

static void
nexusl_pr_udp_init(nexus_bool_t * add_to_my_mi_proto);

static void
nexusl_pr_udp_shutdown(void);

static int
nexusl_pr_udp_send_rsr(struct globus_nexus_buffer_s *buffer);

static globus_bool_t
nexusl_pr_udp_send_rsr_outstanding(globus_nexus_proto_t *nproto);

static void
nexusl_pr_udp_increment_reference_count(nexus_proto_t *nproto);

static nexus_bool_t
nexusl_pr_udp_decrement_reference_count(nexus_proto_t *nproto);

static int
nexusl_pr_udp_get_my_mi_proto(nexus_byte_t **array,
			      int *size,
			      void *proto_info,
			      nexus_endpoint_t *endpoint);

static nexus_bool_t
nexusl_pr_udp_construct_from_mi_proto(nexus_proto_t **proto,
				      nexus_mi_proto_t *mi_proto,
				      nexus_byte_t *proto_array,
				      int size);

static int
nexusl_pr_udp_destroy_my_mi_proto(nexus_endpoint_t * endpoint,
				  nexus_byte_t *proto_array,
				  int size);

static int
nexusl_pr_udp_test_proto(nexus_proto_t *proto);

static int
nexusl_pr_udp_direct_info_size(void);


static int
nexusl_pr_udp_outgoing_construct(globus_nexus_proto_info_udp_t * pi,
				 udp_outgoing_t ** outgoing_ret);

static void
nexusl_pr_udp_outgoing_free(udp_outgoing_t *outgoing);

static void
nexusl_pr_udp_outgoing_close(udp_outgoing_t *outgoing,
			     int new_state);

static udp_incoming_t *
nexusl_pr_udp_incoming_construct(
				 nexus_endpoint_t * endpoint);

static void
nexusl_pr_udp_incoming_close(udp_incoming_t *incoming);

static void
nexusl_pr_udp_incoming_free(udp_incoming_t *incoming);

static void
nexusl_pr_udp_process_fragment(void *arg, long n_read);

/* callbacks */
static void
nexusl_pr_udp_read_callback(
    void *                           arg,
    globus_io_handle_t *             handle,
    globus_result_t                  result,
    globus_byte_t *                  buf,
    globus_size_t                    nbytes_recvd,
    char *                           host,
    unsigned short                   port);

static void
globus_l_nexus_udp_incoming_close_callback(
    void *                                 arg,
    globus_io_handle_t *                   handle,
    globus_result_t                        result);

static void
globus_l_nexus_udp_outgoing_close_callback(
    void *                                 arg,
    globus_io_handle_t *                   handle,
    globus_result_t                        result);

static void
globus_l_nexus_udp_sento_callback(
    void *                           arg,
    globus_io_handle_t *             handle,
    globus_result_t                  result,
    globus_byte_t *                  buf,
    globus_size_t                    nbytes);
/******************************************************************************
		       Define module specific variables
******************************************************************************/

static nexus_byte_t	udp_seq_num
;

static int              globus_l_nexus_udp_n_outgoing_handles_open;
static int              globus_l_nexus_udp_n_incoming_handles_open;

/*
 * My local udp connection information
 */
static char		     udp_local_host[MAXHOSTNAMELEN];
static globus_list_t *       globus_l_nexus_udp_outgoing_list;
static globus_list_t *       globus_l_nexus_udp_incoming_list;
static globus_mutex_t        outgoing_mutex;
static globus_mutex_t        incoming_mutex;
static globus_cond_t         outgoing_cond;
static globus_cond_t         incoming_cond;

#ifdef BUILD_RESOURCE

static nexus_resource_name_t *udp_get_resource_name_sp( nexus_startpoint_t *);
static nexus_resource_name_t *udp_get_resource_name_ep( nexus_endpoint_t * );
static unsigned int udp_lookup_local_addr();

#endif

#endif  /* HAVE_UDP_PROTO */

#define GLOBUS_L_UDP_PROTO_COUNT             1

static nexus_proto_type_t
nexusl_pr_udp_proto_type(void);

static globus_bool_t
udp_startpoint_proto_match(globus_nexus_mi_proto_t * mi_proto0,
			   int                       offset0,
			   nexus_byte_t *            subarray0,
		           int                       sub_length0,
		           globus_nexus_mi_proto_t * mi_proto1,
			   int                       offset1,
			   nexus_byte_t *            subarray1,
			   int                       sub_length1);

static int
udp_proto_count(void);

#if defined(HAVE_UDP_PROTO)

/*
 * Udp specific stuff 
 */
static nexus_proto_funcs_t udp_proto_funcs =
{
    nexusl_pr_udp_proto_type,
    nexusl_pr_udp_init,
    nexusl_pr_udp_shutdown,
    nexusl_pr_udp_increment_reference_count,
    nexusl_pr_udp_decrement_reference_count,
    nexusl_pr_udp_get_my_mi_proto,
    nexusl_pr_udp_construct_from_mi_proto,
    nexusl_pr_udp_destroy_my_mi_proto,
    nexusl_pr_udp_test_proto,
    nexusl_pr_udp_send_rsr,
    nexusl_pr_udp_send_rsr_outstanding,
    nexusl_pr_udp_direct_info_size,
    NULL /* nexusl_pr_udp_direct_get */,
    udp_startpoint_proto_match,
    udp_proto_count,
#ifdef BUILD_RESOURCE
    udp_get_resource_name_sp,
    udp_get_resource_name_ep,
#endif
};

#else

static nexus_proto_funcs_t udp_proto_funcs =
{
    nexusl_pr_udp_proto_type,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    udp_startpoint_proto_match,
    udp_proto_count,
#ifdef BUILD_RESOURCE
    GLOBUS_NULL,
    GLOBUS_NULL,
#endif
};

#endif /* HAVE_UDP_PROTO */

static globus_bool_t
udp_startpoint_proto_match(globus_nexus_mi_proto_t * mi_proto0,
			   int                       offset0,
			   nexus_byte_t * subarray0,
			   int             sub_length0,
			   globus_nexus_mi_proto_t * mi_proto1,
			   int                       offset1,
			   nexus_byte_t * subarray1,
			   int             sub_length1)
{
   int           i = 0;
   int           tmp_int;
   int           version;
   char *        hostname0;
   char *        hostname1;


   UnpackMIProtoHeader((mi_proto0->array),
		       i,
		       tmp_int,
		       hostname0,
		       version);
   i = 0;
   UnpackMIProtoHeader((mi_proto1->array),
		       i,
		       tmp_int,
		       hostname1,
		       version);

    if(offset0 != offset1)
    {
        return GLOBUS_FALSE;
    }
    else if(offset0 == 0)
    {
        return globus_i_is_local_area_network(hostname0, hostname1);
    }
    else
    {
       return GLOBUS_TRUE;
    }
}

static int
udp_proto_count(void)
{
    return GLOBUS_L_UDP_PROTO_COUNT;
}

/*
 * _nx_pr_udp_info()
 *
 * Return the nexus_proto_funcs_t function table for this protocol module.
 *
 * This procedure is used for bootstrapping the protocol module.
 * The higher level Nexus code needs to call this routine to
 * retrieve the functions it needs to use this protocol module.
 */
void *
_nx_pr_udp_info(void)
{
    return((void *) (&udp_proto_funcs));
}
/* _nx_pr_udp_info() */

/*
 * nexusl_pr_udp_proto_type()
 *
 * Return the nexus_proto_type_t for this protocol module.
 */
static nexus_proto_type_t
nexusl_pr_udp_proto_type(void)
{
    return (NEXUS_PROTO_TYPE_UDP);
}
/* nexusl_pr_udp_proto_type() */

#if defined(HAVE_UDP_PROTO)

/*
 * globus_nexus_init_udp_proto_info()
 */
void
globus_nexus_proto_info_udp_init(
    globus_nexus_proto_info_udp_t *	pi)
{
    pi->pi_id = 0x0BAD0BAD;
    pi->host = NULL;
    pi->port = 0;
    pi->ttl = 1;
    pi->loopback = GLOBUS_TRUE;
    pi->multi_sender = GLOBUS_TRUE;
}
/* globus_nexus_init_udp_proto_info() */




/*
 * nexusl_pr_udp_init()
 *
 * Initialize the UDP protocol.
 */
static void
nexusl_pr_udp_init(nexus_bool_t * add_to_my_mi_proto)
{
    globus_mutex_init(&udp_mutex, (globus_mutexattr_t *) NULL);
    globus_mutex_init(&outgoing_mutex, (globus_mutexattr_t *) NULL);
    globus_mutex_init(&incoming_mutex, (globus_mutexattr_t *) NULL);
    globus_cond_init(&outgoing_cond, (globus_condattr_t *) NULL);
    globus_cond_init(&incoming_cond, (globus_condattr_t *) NULL);
    
    udp_done = GLOBUS_FALSE;
    udp_seq_num = 0;

    globus_l_nexus_udp_outgoing_list = GLOBUS_NULL;
    globus_l_nexus_udp_incoming_list = GLOBUS_NULL;
    globus_l_nexus_udp_n_outgoing_handles_open = 0;
    globus_l_nexus_udp_n_incoming_handles_open = 0;
    
    globus_libc_gethostname(udp_local_host, MAXHOSTNAMELEN);

    *add_to_my_mi_proto = NEXUS_FALSE;
}
/* nexusl_pr_udp_init() */


/*
 * nexusl_pr_udp_shutdown()
 *
 * This routine is called during normal shutdown of a process.
 *
 * Shutdown the udp protocol module by closing all off of the
 * outgoing's that have open fd's.  This will result in an orderly
 * shutdown of fd's, so that other contexts will not detect an
 * abnormal eof on the other ends of these fd's.
 *
 * It is ok if this procedure takes a little while to complete -- for
 * example to wait for another process to read off its fd and
 * free up some socket buffer space.
 */
static void
nexusl_pr_udp_shutdown(void)
{
    udp_outgoing_t *outgoing;
    udp_incoming_t *incoming;
    int rc;
    globus_list_t * list;

    udp_enter();

    nexus_debug_printf(1, ("udp_shutdown(): inside\n"));

    udp_done = NEXUS_TRUE;

    globus_mutex_lock(&outgoing_mutex);
    {
        list = globus_l_nexus_udp_outgoing_list;
        while(!globus_list_empty(list))
        {
            outgoing = (udp_outgoing_t *)globus_list_first(list); 
	    list = globus_list_rest(list);
	    if (outgoing->state == OUTGOING_STATE_OPEN)
	    {
		nexusl_pr_udp_outgoing_close(outgoing, OUTGOING_STATE_CLOSED);
	    }
            /* close outgoing */
        }
    }
    globus_mutex_unlock(&outgoing_mutex);

    globus_mutex_lock(&incoming_mutex);
    {
        list = globus_l_nexus_udp_incoming_list;
        while(!globus_list_empty(list))
        {
            incoming = (udp_incoming_t *)globus_list_first(list); 
	    list = globus_list_rest(list);
	    nexusl_pr_udp_incoming_close( incoming );
            /* close outgoing */
        }
    }
    globus_mutex_unlock(&incoming_mutex);

    udp_exit();

    globus_mutex_lock(&outgoing_mutex);
    {
        while(globus_l_nexus_udp_n_outgoing_handles_open > 0)
	{
            globus_cond_wait(&outgoing_cond,
			     &outgoing_mutex);
	}
    }
    globus_mutex_unlock(&outgoing_mutex);

    globus_mutex_lock(&incoming_mutex);
    {
        while(globus_l_nexus_udp_n_incoming_handles_open > 0)
	{
            globus_cond_wait(&incoming_cond,
			     &incoming_mutex);
	}
    }
    globus_mutex_unlock(&incoming_mutex);

    globus_mutex_destroy(&outgoing_mutex);
    globus_mutex_destroy(&incoming_mutex);
}
/* nexusl_pr_udp_shutdown() */

/*
 * nexusl_pr_udp_send_rsr()
 *
 * Generate a remote service request message to the context
 * saved in the 'nexus_buffer'.
 */
static int
nexusl_pr_udp_send_rsr(struct globus_nexus_buffer_s *buffer)
{
    size_t				rc;
    udp_outgoing_t *				outgoing;
    size_t					total_size;
    size_t					size;
    nexus_byte_t *				buf;
    nexus_bool_t				done;
    int						send_rsr_rc = GLOBUS_SUCCESS;
    long					bytes_sent;
    int						frag0;
    globus_result_t                             result; 
    
    
    nexus_debug_printf(1, ("nexusl_pr_udp_send_rsr(): entering, "
			   "buffer=0x%08x\n", (unsigned) buffer));

    if (buffer->n_direct > 0)
    {
	nexus_mutex_init(&(buffer->barrier.mutex), (nexus_mutexattr_t *) NULL);
	nexus_cond_init(&(buffer->barrier.cond), (nexus_condattr_t *) NULL);
	buffer->barrier.count = buffer->n_direct;
	buffer->using_barrier = NEXUS_TRUE;
    }

    udp_enter();
    
    outgoing = (udp_outgoing_t *) buffer->proto;

    if (outgoing->fault_code != NEXUS_ERROR_NONE)
    {
	goto fn_exit1;
    }

    /* By the time we get here, everything is already in the proto
     * including the host and port combination we have to send to.
     * this is connectionless now so use sendto.
     */

    /* don't worry about iovec or direct segments right now */
    
    total_size = buffer->base_segments->size_used;

    /*
     * If we are dealing with a multi-sender scenerio, then the current
     * implementation restricts the message to a single packet.  Messages
     * larger than a packet result in an error being propagated back to the
     * application and the message being discarded.
     */
    if (outgoing->multi_sender &&
	total_size + UDP_HDR_SIZE > UDP_MAX_TRANSMISSION_UNIT)
    {
	send_rsr_rc = GLOBUS_NEXUS_ERROR_BUFFER_TOO_LONG;
	goto fn_exit2;
    }
    
    /*
     * for the first packet, move current back to make room for
     * the header
     *
     * udp_outgoing_t->reserved_header_size is set to UDP_HDR_SIZE
     * to guarantee us that there is room saved in front of current
     * so that this operation does not underflow the array.
     */
    buffer->base_segments->current -= UDP_HDR_SIZE;

    bytes_sent = 0;
    done = NEXUS_FALSE;
    frag0 = 1;
    while(!done)
    {
	buf = buffer->base_segments->current;
	
	/* figure out the size of the packet we're sending */
	if (total_size - bytes_sent + UDP_HDR_SIZE
	    <= UDP_MAX_TRANSMISSION_UNIT)
	{
	    size = total_size - bytes_sent + UDP_HDR_SIZE;
	}
	else
	{
	    size = UDP_MAX_TRANSMISSION_UNIT;
	}

	/* put on the header */
	buf[0] = (nexus_byte_t) GLOBUS_L_NEXUS_UDP_PROTOCOL_VERSION;
	buf[1] = (nexus_byte_t) (frag0 << 7) | ((udp_seq_num >> 16) & 0x7f);
	buf[2] = (nexus_byte_t) ((udp_seq_num >> 8) & 0xff);
	buf[3] = (nexus_byte_t) (udp_seq_num & 0xff);

	frag0 = 0;

	/* send the packet */
	result= globus_io_udp_sendto(
	     	       &outgoing->handle,
		       (globus_byte_t*) buf,
		       0,
		       size,
		       outgoing->host,
		       outgoing->port,
		       &rc);

	if (result == GLOBUS_SUCCESS)
	{
	    /*
	     *  don't count the header as part of
	     *  the total number of bytes sent
	     */
	    bytes_sent += size - UDP_HDR_SIZE;

	    nexus_debug_printf(1, ("tot_size[%d] pkt-size[%d] "
				   "real-bytes-sent[%d] seq[%d]\n",
				   total_size, size, bytes_sent,
				   udp_seq_num));
		
	    /* are we done */
	    if( bytes_sent < total_size )
	    {
		/* move current forward */	  
		buffer->base_segments->current +=
		    (UDP_MAX_TRANSMISSION_UNIT - UDP_HDR_SIZE);
	    }
	    else
	    {
		buffer->current_base_segment
		    = (nexus_base_segment_t *) NULL;
		done = NEXUS_TRUE;
	    }
	}
	else
	{
	    int save_errno;

	    save_errno = errno;
	    
	    if (save_errno == ECONNREFUSED)
	    {
		nexus_debug_printf(1, ("nexusl_pr_udp_send_rsr(): "
				       "sendto() connection refused\n"));
		outgoing->fault_code = NEXUS_ERROR_PROCESS_DIED;
		nexusl_pr_udp_outgoing_close(outgoing, OUTGOING_STATE_FAULT);
		goto fn_exit1;
	    }
	    else if (save_errno != ENOBUFS)
	    {
		nexus_debug_printf(1, ("nexusl_pr_udp_send_rsr(): "
				       "sendto() failed (errno=%d)\n",
				       save_errno));
		outgoing->fault_code = NEXUS_ERROR_BAD_PROTOCOL;
		nexusl_pr_udp_outgoing_close(outgoing, OUTGOING_STATE_FAULT);
		goto fn_exit1;
	    }
	    else /* save_errno == ENOBUFS */
	    {
		/*
		 * We are out of buffers; just drop the rest of the message
		 * on the floor and return
		 */
		nexus_debug_printf(3, ("nexusl_pr_udp_send_rsr(): "
				       "sendto() ran out of buffer space, "
				       "dropping message...\n"));
		buffer->current_base_segment = (nexus_base_segment_t *) NULL;
                done = NEXUS_TRUE;
	    }
	}

	udp_seq_num++;
    }

  fn_exit1:
    send_rsr_rc = outgoing->fault_code;
    
  fn_exit2:
    udp_exit();

    if (buffer->using_barrier)
    {
	/* Signal the thread waiting in tcp_send_rsr() */
	nexus_mutex_lock(&(buffer->barrier.mutex));
	buffer->barrier.count--;
	nexus_cond_signal(&(buffer->barrier.cond));
	nexus_mutex_unlock(&(buffer->barrier.mutex));
    }
    else
    {
	/* Destroy the buffer */
	nexus_buffer_destroy(&buffer);
    }
    
    nexus_debug_printf(1, ("nexusl_pr_udp_send_rsr(): exiting\n"));

    return (send_rsr_rc);
}
/* nexusl_pr_udp_send_rsr() */

/*
 * nexusl_pr_udp_send_rsr_outstanding()
 *
 * Return true if there are any sends outstanding for this proto,
 * otherwise false.
 */
static globus_bool_t
nexusl_pr_udp_send_rsr_outstanding(globus_nexus_proto_t *nproto)
{
    return(GLOBUS_FALSE);
} /* nexusl_pr_udp_send_rsr_outstanding() */


/*
 * nexusl_pr_udp_outgoing_construct()
 *
 * Construct a udp_outgoing_t for the given host and port. Look up in the
 * outgoing table to see if one already exists. If it does, bump its reference
 * count and return that one. Otherwise create one, insert into the
 * table with a reference count of 1 and return it.
 */
static int
nexusl_pr_udp_outgoing_construct(
    globus_nexus_proto_info_udp_t *	pi,
    udp_outgoing_t **			outgoing_ret)
{
    int					rc;
    globus_result_t                     result;
    udp_outgoing_t *			outgoing;

    nexus_debug_printf(3,("nexusl_pr_udp_outgoing_construct(): enter, "
			  "host=%s, port=%d, ttl=%d\n",
			  pi->host, pi->port, pi->ttl));
    rc = 0;

    outgoing = NULL;
    if(outgoing == NULL)
    {
	char				ip_mc_loop;
	char				ip_mc_ttl;
	unsigned			ip_daddr;
	unsigned			tmp_u;

	NexusMalloc(outgoing_construct(), outgoing, udp_outgoing_t *,
		    sizeof(udp_outgoing_t));

	outgoing->type = NEXUS_PROTO_TYPE_UDP;
	outgoing->funcs = &udp_proto_funcs;
	outgoing->version = GLOBUS_L_NEXUS_UDP_PROTOCOL_VERSION;
	outgoing->direct_custom_min_size = NEXUS_UDP_DIRECT_CUSTOM_MIN_SIZE;
	outgoing->direct_custom_max_size = NEXUS_DC_MAX_U_LONG;
	outgoing->direct_pointer_min_size = NEXUS_DC_MAX_U_LONG;
	outgoing->direct_pointer_max_size = NEXUS_DC_MAX_U_LONG;
	outgoing->can_use_iovec = NEXUS_FALSE;
	outgoing->reserved_header_size = UDP_HDR_SIZE;
	outgoing->host = pi->host;
	outgoing->port = pi->port;
	outgoing->l_port = 0;
	outgoing->multi_sender = pi->multi_sender;
	outgoing->state = OUTGOING_STATE_UNOPENED;
	outgoing->fault_code = NEXUS_ERROR_NONE;
	outgoing->reference_count = 1;

	globus_io_udpattr_init(&outgoing->attr);
	/* check to see if we've got a multicast address, if so
	 * we have to join the group
	 */
	if((sscanf(pi->host, "%u.%*u.%*u.%u", &ip_daddr, &tmp_u) == 2)
	   && (ip_daddr >= 224 && ip_daddr <= 239))
	{
#	    if (IP_MULTICAST_ENABLED)
	    {
		globus_io_attr_set_udp_multicast_membership(
		    &outgoing->attr,
		    pi->host,
		    INADDR_ANY);
		globus_io_attr_set_udp_multicast_loop(
		    &outgoing->attr,
		    pi->loopback);
                ip_mc_ttl = (char) ((pi->ttl > 0) ? pi->ttl : 1);
		globus_io_attr_set_udp_multicast_ttl(
		    &outgoing->attr,
		    ip_mc_ttl);
	    }
#	    else
	    {
		rc = GLOBUS_NEXUS_ERROR_BAD_PROTOCOL;
		nexus_debug_printf(1,
		("nexusl_pr_udp_outgoing_construct(): "
		 "multicast protocol not support on this machine\n"));
		
	        return rc;
	    }
#	    endif
	}

#ifndef BUILD_LITE        
        globus_io_attr_set_callback_space(&outgoing->attr, _nx_serial_space);
#endif

        result = globus_io_udp_bind(
                      &outgoing->l_port,
	              &outgoing->attr,
	              &outgoing->handle);
        assert(result == GLOBUS_SUCCESS);

	if(result != GLOBUS_SUCCESS)
	{
	    nexus_debug_printf(1,
			       ("nexusl_pr_udp_outgoing_construct(): "
				"address_setup failed\n"));
	    return -1;
	}

	globus_mutex_lock(&outgoing_mutex);
	{
            globus_list_insert(&globus_l_nexus_udp_outgoing_list,
			       (void *) outgoing);
        }
	globus_mutex_unlock(&outgoing_mutex);
	/*
	 * socket creation notification
	 */
#       ifdef BUILD_RESOURCE
	{
	  nexus_rusage_socket current_socket;
	  int len;

	  nexus_resource_name(&outgoing->resource_name, RESOURCE_SOCKET);

	  current_socket.who = &outgoing->resource_name;
	  current_socket.pid = globus_libc_getpid();
	  current_socket.fd = outgoing->handle.fd;
	  current_socket.mode = RESOURCE_SOCKET_WRITE;
	  len = sizeof(current_socket.addr);
	  memcpy(&current_socket.addr, &outgoing->addr, 
		 sizeof(current_socket.addr));
	  current_socket.protocol = IPPROTO_UDP;
	  current_socket.op = RUSAGE_CREATE;

	  nexus_overseer_rusage((nexus_rusage_obj*)&current_socket);
	}
#       endif /* BUILD_RESOURCE */

	/*globus_l_nexus_udp_outgoing_table_insert(outgoing);*/
    }

    *outgoing_ret = outgoing;
    NexusAssert(rc == 0);

    return(rc);
}
/* nexusl_pr_udp_outgoing_construct() */

static udp_incoming_t *
globus_l_nexus_udp_incoming_find(
    char *                         host,
    unsigned short                 port)
{
    globus_list_t *        list;
    udp_incoming_t *       incoming; 
    udp_incoming_t *       found = GLOBUS_NULL;

    globus_mutex_lock(&incoming_mutex);
    {
        list = globus_l_nexus_udp_incoming_list;

	while(!globus_list_empty(list) && found == GLOBUS_NULL)
	{
            incoming = (udp_incoming_t *) globus_list_first(list);
            list = globus_list_rest(list);

	    if(incoming->port == port &&
	       strcmp(incoming->host, host) == 0)
            {
		found = incoming;
	    }
	}
    }
    globus_mutex_unlock(&incoming_mutex);

    return found;
}

/*
 * outgoing_table_insert()
 *
 * Insert the given outgoing into the table, hashing on its internal
 * hostname and port number.
 *
 * We assume that the entry is not present in the table.
 */
static void 
globus_l_nexus_udp_outgoing_table_insert(udp_outgoing_t *outgoing)
{
    int bucket;
    globus_l_nexus_udp_outgoing_table_entry_t *new_ent;

    bucket = globus_l_nexus_udp_outgoing_table_hash(outgoing->host, outgoing->port);
    if (outgoing_table[bucket].outgoing == (udp_outgoing_t *) NULL)
    {
        /* Drop it into the preallocated table entry */
        outgoing_table[bucket].outgoing = outgoing;
    }
    else
    {
	/*
         * Need to allocate a new outgoing_table_entry_t and add it
         * to the bucket
         */
        NexusMalloc(outgoing_table_insert(), 
		    new_ent, 
		    globus_l_nexus_udp_outgoing_table_entry_t *,
		    sizeof(globus_l_nexus_udp_outgoing_table_entry_t));
        new_ent->outgoing = outgoing;
	new_ent->next = outgoing_table[bucket].next;

	outgoing_table[bucket].next = new_ent;
    }
} /* outgoing_table_insert() */


/*
 * outgoing_table_remove()
 *
 * Remove the given outgoing from the table.
 *
 * We assume that the entry is present in the table.
 */
static void
globus_l_nexus_udp_outgoing_table_remove(
     udp_outgoing_t *                               outgoing)
{
    int                                              bucket;
    globus_l_nexus_udp_outgoing_table_entry_t *      ent;
    globus_l_nexus_udp_outgoing_table_entry_t *      remove_ent;

    bucket = globus_l_nexus_udp_outgoing_table_hash(outgoing->host, outgoing->port);

    if (outgoing_table[bucket].outgoing == outgoing)
    {
        if (outgoing_table[bucket].next)
	{
            outgoing_table[bucket].outgoing
                = outgoing_table[bucket].next->outgoing;
            outgoing_table[bucket].next
                = outgoing_table[bucket].next->next;
	}
        else
	{
            outgoing_table[bucket].outgoing = (udp_outgoing_t *) NULL;
            outgoing_table[bucket].next = (globus_l_nexus_udp_outgoing_table_entry_t *) NULL;
	}
    }
    else
    { 
        for (ent = &(outgoing_table[bucket]);
             ent->next->outgoing != outgoing;
             ent = ent->next)
            ;
        remove_ent = ent->next;
        ent->next = remove_ent->next;
        NexusFree(remove_ent);
    }
} /* outgoing_table_remove() */


/*
 * outgoing_table_hash()
 *
 * Hash the hostname and port for the outgoing table.
 */
static int 
globus_l_nexus_udp_outgoing_table_hash(char *host, unsigned short port)
{
    unsigned int hval = 0;
    char *s;

    for (s = host; *s != '\0'; s++)
    {
        hval <<= 1;
        hval += *s;
    }
    hval <<= 1;
    hval += port;
    return (hval % OUTGOING_TABLE_SIZE);
} /* outgoing_table_hash() */

static udp_outgoing_t *
globus_l_nexus_udp_outgoing_table_lookup(char *host, unsigned short port)
{
    globus_l_nexus_udp_outgoing_table_entry_t *ent;
    int bucket;

    bucket = globus_l_nexus_udp_outgoing_table_hash(host, port);

    for (ent = &(outgoing_table[bucket]);
         ent != (globus_l_nexus_udp_outgoing_table_entry_t *) NULL;
         ent = ent->next)
      {
        if (   (ent->outgoing != (udp_outgoing_t *) NULL)
            && (ent->outgoing->port == port)
            && (strcmp(ent->outgoing->host, host) == 0) )
	  {
            return (ent->outgoing);
	  }
      }


    return ((udp_outgoing_t *) NULL);
} /* outgoing_table_lookup() */

/*
 * nexusl_pr_udp_outgoing_free()
 *
 * Free the passed 'outgoing'.
 */
static void
nexusl_pr_udp_outgoing_free(udp_outgoing_t *outgoing)
{
    if (outgoing->host != (char *) NULL)
    {
	NexusFree(outgoing->host);
    }
    NexusFree(outgoing);
}
/* nexusl_pr_udp_outgoing_free() */


/*
 * nexusl_pr_udp_outgoing_close()
 *
 * Close a outgoing:
 *   1) Remove the outgoing from the fd_table table
 *   2) Close the fd
 *   3) Modify outgoing data structure
 */
static void
nexusl_pr_udp_outgoing_close(udp_outgoing_t *outgoing,
			     int new_state)
{
    nexus_debug_printf(1, ("outgoing_close(): inside\n"));

#ifdef BUILD_DEBUG
    if (NexusDebug(2))
    {
        struct sockaddr_in local, remote;
        globus_netlen_t len;
	len = sizeof(local);
        getsockname(outgoing->handle.fd, (struct sockaddr *) &local, &len);
	len = sizeof(remote);
        getpeername(outgoing->handle.fd, (struct sockaddr *) &remote, &len);

	nexus_printf("outgoing_close(): closing outgoing %x %s/%hu fd=%d local=%hu remote=%hu\n",
		     outgoing,
		     outgoing->host,
		     outgoing->port,
		     outgoing->handle.fd,
		     htons(local.sin_port),
		     htons(remote.sin_port));
    }
#endif

    /*
     * socket destruction notification
     */
#ifdef BUILD_RESOURCE
    {
      nexus_rusage_socket current_socket;
      int len;

      current_socket.who = &outgoing->resource_name;
      current_socket.op = RUSAGE_DESTROY;
      current_socket.pid = globus_libc_getpid();
      current_socket.fd = outgoing->handle.fd;
      current_socket.mode = RESOURCE_SOCKET_WRITE;
      len = sizeof(current_socket.addr);
      memcpy(&current_socket.addr, &outgoing->addr, 
	     sizeof(current_socket.addr));
      current_socket.protocol = IPPROTO_UDP;

      nexus_overseer_rusage((nexus_rusage_obj*)&current_socket);
    }
#endif /* BUILD_RESOURCE */

    outgoing->state = OUTGOING_STATE_CLOSING;
    outgoing->new_state = new_state;

    globus_io_register_close(
              	  &outgoing->handle,
		  globus_l_nexus_udp_outgoing_close_callback,
		  (void *) outgoing);

}


static void
globus_l_nexus_udp_outgoing_close_callback(
    void *                                 arg,
    globus_io_handle_t *                   handle,
    globus_result_t                        result)
{
    udp_outgoing_t *                     outgoing;
    globus_list_t *                      list;

    outgoing = (udp_outgoing_t *)arg;
    outgoing->state = outgoing->new_state;

    globus_mutex_lock(&outgoing_mutex);
    {
        list = globus_list_search(globus_l_nexus_udp_outgoing_list, 
				  (void *) outgoing);
        globus_list_remove(&globus_l_nexus_udp_outgoing_list,
			   list);
        globus_l_nexus_udp_n_outgoing_handles_open--;

	if(globus_l_nexus_udp_n_outgoing_handles_open <= 0)
	{
            globus_cond_signal(&outgoing_cond);
	}
    }
    globus_mutex_unlock(&outgoing_mutex);
   
    if(outgoing->state == OUTGOING_STATE_CLOSED)
    {
        nexusl_pr_udp_outgoing_free(outgoing);
    }
}
/* nexusl_pr_udp_outgoing_close() */


/*
 * nexusl_pr_udp_increment_reference_count()
 *
 * Increase the reference count on the associated proto and copy the
 * pointer to the nexus_proto_t
 *
 */
static void
nexusl_pr_udp_increment_reference_count(nexus_proto_t *nproto)
{
    udp_outgoing_t *outgoing = (udp_outgoing_t *) nproto;
    udp_enter();
    outgoing->reference_count++;
    udp_exit();
}
/* nexusl_pr_udp_increment_reference_count() */


/*
 * nexusl_pr_udp_decrement_reference_count()
 *
 * Decrement the reference count for this proto.  If it goes to 0
 * then close the fd used by this outgoing.
 *
 * Return NEXUS_TRUE if this functioin frees the proto.
 */
static nexus_bool_t
nexusl_pr_udp_decrement_reference_count(nexus_proto_t *nproto)
{
    udp_outgoing_t *outgoing = (udp_outgoing_t *) nproto;
    nexus_bool_t proto_freed = NEXUS_FALSE;
    udp_enter();
    outgoing->reference_count--;
    
    NexusAssert2((outgoing->reference_count >= 0),
		 ("nexusl_pr_udp_decrement_reference_count(): "
		  "Internal error: Reference count < 0\n"));
	       
    if (outgoing->reference_count == 0)
    {
	if (outgoing->state == OUTGOING_STATE_OPEN)
	{
	    nexusl_pr_udp_outgoing_close(outgoing, OUTGOING_STATE_CLOSED);
	}
        else
	{
	    nexusl_pr_udp_outgoing_free(outgoing);
        }
	proto_freed = NEXUS_TRUE;
    }
    udp_exit();
    return(proto_freed);
    
}
/* nexusl_pr_udp_decrement_reference_count() */


/*
 * nexusl_pr_udp_get_my_mi_proto()
 *
 * Return in 'array' and 'size' a byte array containing
 * enough information to enable another process to connect
 * to this one.  This byte array will be assembled with those
 * of the other protocol modules and placed in a nexus_mi_proto_t.
 *
 * Fill in 'array' with:
 *	version (1 byte)
 *      file descriptor ID (4 bytes, big endian integer)
 *	port (2 byte, big endian unsigned short)
 *	tty (1 byte)
 *      loopback flag (1 byte)
 *      multi-sender flag (1 byte)
 *	hostname (null terminated string)
 *
 * If the hostname is the same as my hostname, then just place an
 * empty string in the hostname field.
 */
static int
nexusl_pr_udp_get_my_mi_proto(nexus_byte_t **array,
			      int *size,
			      void *proto_info,
			      nexus_endpoint_t *endpoint)
{
    int					rc;
    int					i;
    int					tmp_int;
    char *				host;
    int					host_length;
    unsigned short			port;
    int					len;
    struct sockaddr_in			my_addr;
    int					save_error;
    udp_incoming_t *			incoming;
    nexus_proto_info_udp_t *		pi;
    nexus_proto_info_udp_t		tmp_pi;
    unsigned				ip_daddr;
    unsigned				tmp_u;
    globus_result_t                     result;

    nexus_debug_printf(1, ("udp_get_my_mi_proto(): inside\n"));

    rc = 0;

    /*
     * If the user has supplied specific protocol information, then use it;
     * otherwise, set up a temporary proto_info structure with the defaults
     */
    pi = (nexus_proto_info_udp_t *) proto_info;
    if (pi == NULL)
    {
	globus_nexus_proto_info_udp_init(&tmp_pi);
	pi = &tmp_pi;
    }
    else
    {
	if (pi->pi_id != 0x0BAD0BAD)
	{
	    rc = GLOBUS_NEXUS_ERROR_INVALID_PARAMETER;
	    nexus_debug_printf(
		1,
		("nexusl_pr_udp_get_my_mi_proto(): "
		 "proto_info structure not initialized\n"));
	    goto fn_exit;
	}
    }

    port = pi->port;

    incoming = nexusl_pr_udp_incoming_construct(endpoint);
    incoming->host = pi->host;
    incoming->port = pi->port;

    globus_io_udpattr_init(&incoming->attr);

    if((pi->host != GLOBUS_NULL) &&
       (sscanf(pi->host, "%u.%*u.%*u.%u", &ip_daddr, &tmp_u) == 2) &&
       (ip_daddr >= 224 && ip_daddr <= 239))
    {
	/*
	 * Multicast address specified
	 */
#	if (IP_MULTICAST_ENABLED)
	{
	    int					so_reuse = 1;
	    char				ip_mc_loop;

	    ip_mc_loop = (char) pi->loopback;
	    host = pi->host;
	    port = pi->port;

	    globus_io_attr_set_udp_multicast_membership(
		    &incoming->attr,
		    pi->host,
		    INADDR_ANY);
	    globus_io_attr_set_udp_multicast_loop(
		    &incoming->attr,
		    ip_mc_loop);
            globus_io_attr_set_socket_reuseaddr(
		    &incoming->attr,
                    so_reuse);

            /* TODO: check return codes */

	}
#       else
	{
	    rc = GLOBUS_NEXUS_ERROR_BAD_PROTOCOL;
	    nexus_debug_printf(
		1,
		("nexusl_pr_udp_outgoing_construct(): "
		 "multicast protocol not support on this machine\n"));
	    goto fn_exit;
	}
#       endif
    }
    else
    {
        if (pi->host != GLOBUS_NULL && strcmp(pi->host, "") != 0)
	{
	    host = pi->host;
        }
        else if (strcmp(udp_local_host, _nx_my_hostname) == 0)
	{
            host = "";
        }
        else
        {
            host = udp_local_host;
        }
    } 

#ifndef BUILD_LITE        
    globus_io_attr_set_callback_space(&incoming->attr, _nx_serial_space);
#endif

    result = globus_io_udp_bind(
		       &port,
		        &incoming->attr,
		       &incoming->handle);
    assert(result == GLOBUS_SUCCESS);

    incoming->port = port;
    if(result != GLOBUS_SUCCESS)
    {
	rc = GLOBUS_NEXUS_ERROR_RESOURCE_UNAVAILABLE;
	nexus_debug_printf(1,
			   ("nexusl_pr_udp_get_my_mi_proto(): "
			    "bind() failed: %s\n",
			    globus_libc_system_error_string(errno)));
	goto fn_exit;
    }
   

    result = globus_io_udp_register_recvfrom(
	          &incoming->handle,
	          incoming->current,
	          UDP_MAX_TRANSMISSION_UNIT,
                  0,
    	          nexusl_pr_udp_read_callback,
	          (void *) incoming);
    assert(result == GLOBUS_SUCCESS);
    
    host_length = (strlen(host) + 1);
    *size = 10 + host_length;
    NexusMalloc(udp_get_my_mi_proto(),
		*array,
		nexus_byte_t *,
		*size);

    i = 0;

    (*array)[i++] = GLOBUS_L_NEXUS_UDP_MI_PROTO_VERSION;
    PackInt4(*array, i, incoming->handle.fd);

    tmp_int = (int) port;
    PackInt2(*array, i, tmp_int);

    tmp_int = pi->ttl;
    PackInt1(*array, i, tmp_int);
    
    tmp_int = pi->loopback;
    PackInt1(*array, i, tmp_int);

    tmp_int = pi->multi_sender;
    PackInt1(*array, i, tmp_int);
    
    memcpy(&((*array)[i]), host, host_length);
    

  fn_exit:
    if (rc != 0)
    {
	*size = 0;
	*array = NULL;
    }

    nexus_debug_printf(1, ("udp_get_my_mi_proto(): host[%s]\n", host));

    return(rc);
} 
/* nexusl_pr_udp_get_my_mi_proto() */


/*
 * nexusl_pr_udp_construct_from_mi_proto()
 *
 * From the passed machine independent protocol list ('mi_proto'), plus
 * the udp specific entry from that list ('proto_array' and 'size'),
 * see if I can use the information to create a nexus_proto_t object
 * that can be used to connect to the context:
 *	- If I cannot use this protocol to attach to the context, then
 *		return NEXUS_FALSE.  (This option is useful if two contexts
 *		both speak a particular protocol, but they cannot
 *		talk to each other via that protocol.  For example,
 *		on two MPP's, the contexts within a single MPP can
 *		talk to each other via the native messaging protocol,
 *		but cannot talk to the contexts on the other MPP
 *		using that native protocol.)
 *	- If this udp protocol points to myself, and thus the local
 *		protocol module should be used, then set
 *		*proto=NULL, and return NEXUS_TRUE.
 *	- Otherwise, construct a udp protocol object for this mi_proto
 *		and put it in *proto.  Then return NEXUS_TRUE.
 *
 * The 'proto_array' should contain:
 *	version (1 byte)
 *      file descriptor ID (4 bytes, big endian integer)
 *	port (2 byte, big endian unsigned short)
 *	tty (1 byte)
 *      loopback flag (1 byte)
 *      multi-sender flag (1 byte)
 *	hostname (null terminated string)
 *
 * If the hostname is an empty string, then use the hostname
 * from the mi_proto.
 */
static nexus_bool_t
nexusl_pr_udp_construct_from_mi_proto(nexus_proto_t **proto,
				      nexus_mi_proto_t *mi_proto,
				      nexus_byte_t *proto_array,
				      int size)
{
    char *				host;
    globus_nexus_proto_info_udp_t	pi;
    int					i;
    int					tmp_int;
    nexus_bool_t			rc;
    int					version;

    rc = NEXUS_TRUE;

    NexusAssert2((size > 10),
		 ("udp_construct_from_mi_proto(): "
		  "Invalid udp information in mi_proto\n"));
		 
    i = 0;
    
    /* Check the udp mi_proto version */
    version = (int) proto_array[i++];
    if (version != GLOBUS_L_NEXUS_UDP_MI_PROTO_VERSION)
    {
	_nx_fault_detected(GLOBUS_NEXUS_ERROR_VERSION_MISMATCH);
	return(NEXUS_FALSE);
    }
    
    /* Extract information from the MI proto */
    pi.pi_id = 0x0BAD0BAD;
    UnpackInt4(proto_array, i, tmp_int);
    UnpackInt2(proto_array, i, tmp_int);
    pi.port = (unsigned short) tmp_int;
    UnpackInt1(proto_array, i, tmp_int);
    pi.ttl = (globus_byte_t) tmp_int;
    UnpackInt1(proto_array, i, tmp_int);
    pi.loopback = (globus_byte_t) tmp_int;
    UnpackInt1(proto_array, i, tmp_int);
    pi.multi_sender = (globus_byte_t) tmp_int;

    host = (char *) &(proto_array[i]);
    if (host[0] == '\0')
    {
	int context_differentiator;
	i = 0;
	UnpackMIProtoHeader(mi_proto->array, i,
			    context_differentiator,
			    host,
			    version);
	NexusAssert2((strlen(host) > 0),
		     ("udp_construct_from_mi_proto(): "
		      "Invalid hostname field in mi_proto\n"));
    }
    
    pi.host = nexus_rdb_lookup(host, "udp_interface");
    if (pi.host == GLOBUS_NULL)
    {
	pi.host = _nx_copy_string(host);
    }

    /*
     * TODO: how do we know if this proto points to itself anymore,
     * the test used to look at the udp_local_port information
     * but that's not used anymore.
     *
     * (old comment below, leave as a reminder)
     * Test to see if this proto points to myself.
     * If it does, then return the _nx_local_proto.
     */
/* 
    if(globus_l_nexus_udp_incoming_find(pi.host, pi.port) == GLOBUS_NULL)
 */   {
        udp_enter();
        if (nexusl_pr_udp_outgoing_construct(&pi, (udp_outgoing_t **) proto) != 0)
        {
	rc = NEXUS_FALSE;
        }
        udp_exit();
    }
  /*  else
    {
        *proto = _nx_local_proto;
    }
*/
    return (rc);
}
/* nexusl_pr_udp_construct_from_mi_proto() */


/*
 * nexusl_pr_udp_destroy_my_mi_proto()
 */
static int
nexusl_pr_udp_destroy_my_mi_proto(nexus_endpoint_t * endpoint,
				  nexus_byte_t *proto_array,
				  int size)
{
    int					version;
    int					fd;
    int					tmp_i;
    int					rc;
    udp_incoming_t *			incoming;
    udp_incoming_t *			found = GLOBUS_NULL;
    globus_list_t *                     list; 

    udp_enter();

    rc = 0;

    tmp_i = 0;
    
    /* Check the tcp mi_proto version */
    version = (int) proto_array[tmp_i++];
    if (version != GLOBUS_L_NEXUS_UDP_MI_PROTO_VERSION)
    {
	return(GLOBUS_NEXUS_ERROR_VERSION_MISMATCH);
    }
    
    UnpackInt4(proto_array, tmp_i, fd);

    globus_mutex_lock(&incoming_mutex);
    {
        list = globus_l_nexus_udp_incoming_list;

	while(!globus_list_empty(list) && found == GLOBUS_NULL)
	{
            incoming = (udp_incoming_t *) globus_list_first(list);
            list = globus_list_rest(list);

	    if(incoming->handle.fd == fd)
            {
		found = incoming;
	    }
	}
    }
    globus_mutex_unlock(&incoming_mutex);

    if(found != GLOBUS_NULL)
    {
        nexusl_pr_udp_incoming_close(incoming);
    }
    else
    {
         if (_nx_fault_tolerant)
	 {
	     nexus_debug_printf(1, ("nexusl_pr_udp_destroy_my_mi_proto(): "
				"fd_table[%d].type != INCOMING\n", fd));
             rc = -1;
  	 }
         else
         {
             globus_fatal("nexusl_pr_udp_destroy_my_mi_proto(): "
	                  "fd_table[%d].type != INCOMING\n", fd);
         }
    }

    udp_exit();

    return(rc);
}
/* nexusl_pr_udp_destroy_my_mi_proto() */


/*
 * nexusl_pr_udp_test_proto()
 */
static int
nexusl_pr_udp_test_proto(nexus_proto_t *proto)
{
    udp_outgoing_t *outgoing = (udp_outgoing_t *) proto;
    return(outgoing->fault_code);
}
/* nexusl_pr_udp_test_proto() */


/*
 * nexusl_pr_udp_direct_info_size()
 */
static int
nexusl_pr_udp_direct_info_size(void)
{
    /* TODO: This needs to be filled in */
    return(0);
}
/* nexusl_pr_udp_direct_info_size() */


/*
 * nexusl_pr_udp_incoming_construct()
 *
 * Construct a udp_incoming_t for the given file descriptor, 'fd'.
 */
static udp_incoming_t *
nexusl_pr_udp_incoming_construct(
				 nexus_endpoint_t * endpoint)
{
    udp_incoming_t *incoming;

    nexus_debug_printf(1, ("nexusl_pr_udp_incoming_construct(), enter, "));
			   

    NexusMalloc(incoming_construct(),
		incoming,
		udp_incoming_t *,
		sizeof(udp_incoming_t));
    NexusMalloc(incoming_construct(),
		incoming->storage,
		nexus_byte_t *,
		UDP_MAX_TRANSMISSION_UNIT);

    
    incoming->expected_seq_num		= 0;
    incoming->nbytes_read		= 0;
    incoming->nbytes_parsed		= 0;
    incoming->nbytes_unparsed		= 0;
    incoming->storage_size		= UDP_MAX_TRANSMISSION_UNIT;
    incoming->current			= incoming->storage;
    incoming->save_storage_size		= 0;
    incoming->dispatch_in_progress	= NEXUS_FALSE;
    incoming->dispatch_q_head		= (struct globus_nexus_buffer_s *) NULL;
    incoming->dispatch_q_tail		= (struct globus_nexus_buffer_s *) NULL;
    incoming->endpoint			= endpoint;

    globus_mutex_lock(&outgoing_mutex);
    {
        globus_list_insert(&globus_l_nexus_udp_incoming_list,
                           (void *) incoming);
    }
    globus_mutex_unlock(&outgoing_mutex);

    /*
     * socket creation notification
     */
#ifdef BUILD_RESOURCE
	{
	  nexus_rusage_socket current_socket;
	  globus_netlen_t len;

	  nexus_resource_name(&incoming->resource_name, RESOURCE_SOCKET);

	  current_socket.who = &incoming->resource_name;
	  current_socket.op = RUSAGE_CREATE;
	  current_socket.pid = globus_libc_getpid();
	  current_socket.fd = incoming->handle.fd;
	  current_socket.mode = RESOURCE_SOCKET_READ;
	  len = sizeof(current_socket.addr);
	  getsockname(incoming->handle.fd, 
		      (struct sockaddr *)&current_socket.addr, &len);
	  {
	    struct in_addr *addr = &current_socket.addr.sin_addr;
	    struct in_addr any_addr = {INADDR_ANY};
	    if (memcmp (addr, &any_addr, sizeof(addr)) == 0) {
	      addr->s_addr = udp_lookup_local_addr();
	    }
	  }
	  current_socket.protocol = IPPROTO_UDP;

	  nexus_overseer_rusage((nexus_rusage_obj*)&current_socket);
	}
#endif /* BUILD_RESOURCE */
    
    nexus_debug_printf(1, ("nexusl_pr_udp_incoming_construct(), exit\n"));

    return incoming;
}
/* nexusl_pr_udp_incoming_construct() */


/*
 * nexusl_pr_udp_incoming_close()
 *
 * Close an incoming connection:
 *   1) Remove the udp_incoming_t from the fd_table table
 *   2) Close the fd
 *   3) Put the udp_incoming_t back on the free list.
 */
static void
nexusl_pr_udp_incoming_close(udp_incoming_t *incoming)
{
#ifdef BUILD_DEBUG
    if (NexusDebug(2))
    {
        struct sockaddr_in local, remote;
        globus_netlen_t len;
	len = sizeof(local);
        getsockname(incoming->handle.fd, (struct sockaddr *) &local, &len);

	nexus_printf("incoming_close(): closing %x fd=%d local=%hu\n",
		     incoming,
		     incoming->handle.fd,
		     htons(local.sin_port));
    }
#endif

    /*
     * socket destruction notification
     */
#ifdef BUILD_RESOURCE
    {
      nexus_rusage_socket current_socket;
      globus_netlen_t len;

      current_socket.who = &incoming->resource_name;
      current_socket.op = RUSAGE_DESTROY;
      current_socket.pid = globus_libc_getpid();
      current_socket.fd = incoming->handle->fd;
      current_socket.mode = RESOURCE_SOCKET_READ;
      len = sizeof(current_socket.addr);
      getsockname(incoming->fd, (struct sockaddr *)&current_socket.addr, &len);
      {
	   struct in_addr *addr = &current_socket.addr.sin_addr;
	   struct in_addr any_addr = {INADDR_ANY};
	   if(memcmp (addr, &any_addr, sizeof(addr)) == 0)  
	   {
	       addr->s_addr = udp_lookup_local_addr();
	   }
      }
      current_socket.protocol = IPPROTO_UDP;

      nexus_overseer_rusage((nexus_rusage_obj*)&current_socket);
    }
#endif /* BUILD_RESOURCE */

    globus_io_register_close(
              	  &incoming->handle,
		  globus_l_nexus_udp_incoming_close_callback,
		  (void *) incoming);
}

static void
globus_l_nexus_udp_incoming_close_callback(
    void *                                 arg,
    globus_io_handle_t *                   handle,
    globus_result_t                        result)
{
    udp_incoming_t *                     incoming;
    globus_list_t *                      list;

    incoming = (udp_incoming_t *)arg;

    globus_mutex_lock(&incoming_mutex);
    {
        list = globus_list_search(globus_l_nexus_udp_incoming_list, 
				  (void *) incoming);
        globus_list_remove(&globus_l_nexus_udp_incoming_list,
			   list);
        globus_l_nexus_udp_n_incoming_handles_open--;

	if(globus_l_nexus_udp_n_incoming_handles_open <= 0)
	{
            globus_cond_signal(&incoming_cond);
	}
    }
    globus_mutex_unlock(&incoming_mutex);
  
    nexusl_pr_udp_incoming_free(incoming);
}


/*
 * nexusl_pr_udp_incoming_free()
 */
static void
nexusl_pr_udp_incoming_free(udp_incoming_t *incoming)
{
    NexusFree(incoming->storage);
    NexusFree(incoming);
}
/* nexusl_pr_udp_incoming_free() */


#define MAX_SOCKET_SEND_BUFFER		9000
#define MAX_SOCKET_RECEIVE_BUFFER	18032

/*
 * nexusl_pr_udp_outgoing_address_setup()
 */
static int
nexusl_pr_udp_outgoing_address_setup(udp_outgoing_t * outgoing)
{
    struct hostent			he;
    struct hostent *			hp;
    char hp_tsdbuffer[500];
    int hp_errnop;

    hp = globus_libc_gethostbyname_r(outgoing->host,
				     &he,
				     hp_tsdbuffer,
				     500,
				     &hp_errnop);
    
    if (hp == NULL)
    {
	return(-1);
    }

    memset(&outgoing->addr, '\0', sizeof(outgoing->addr));
    outgoing->addr.sin_family = hp->h_addrtype;
    memcpy(&outgoing->addr.sin_addr, hp->h_addr, hp->h_length);
    outgoing->addr.sin_port = htons(outgoing->port);

    return(0);
}
/* nexusl_pr_udp_outgoing_address_setup() */


/*
 * nexusl_pr_udp_read_callback()
 */
static void
nexusl_pr_udp_read_callback(
    void *                           arg,
    globus_io_handle_t *             handle,
    globus_result_t                  result,
    globus_byte_t *                  buf,
    globus_size_t                    nbytes_recvd,
    char *                           host,
    unsigned short                   port)
{
    udp_incoming_t *              incoming;
    long                          n_read;
    int                           save_errno;
    globus_bool_t                 done;

    nexus_debug_printf(1,("udp_read_callback(): entering\n"));

    incoming = (udp_incoming_t *) arg;
    udp_enter();

    done = NEXUS_FALSE;
    if(result == GLOBUS_SUCCESS)
    {
	nexus_debug_printf(1,
			   ("udp_read_callback(): "
			    "calling recvfrom, fd=%i, current=%lx, size=%i\n",
			    handle->fd,
			    (incoming->current),
			    (incoming->storage_size - incoming->nbytes_read)));

	incoming->from_len = sizeof(incoming->from_addr);
	    
        n_read = nbytes_recvd;

        result = globus_io_udp_register_recvfrom(
  	              &incoming->handle,
	              incoming->current,
	              UDP_MAX_TRANSMISSION_UNIT,
                      0,
    	              nexusl_pr_udp_read_callback,
	              (void *) incoming);
	/*
	 * n_read: is > 0 if it successfully read some bytes
	 *         is < 0 on error -- need to check errno
	 */
	if (n_read > UDP_HDR_SIZE)
	{
	    nexusl_pr_udp_process_fragment(incoming, n_read);
	}
	else /* n_read < UDP_HDR_SIZE */
	{
	    nexusl_pr_udp_incoming_close(incoming);
	    udp_exit();
	    if (_nx_fault_detected(NEXUS_ERROR_READ_FAILED) != 0)
	    {
		globus_fatal("udp_read_callback(): "
			    "Internal error: Read failed with n_read=%d\n",
			    n_read);
	    }
	    else
	    {
		nexus_debug_printf(1, ("nexusl_pr_udp_read_callback(): read "
				       "read failed, n_read=%d\n", n_read));
	    }
	    udp_enter();
	}
    }
    else
    {
	globus_object_t *err;

	err = globus_error_get(result);

        if(globus_object_type_match(globus_object_get_type(err),
              	    GLOBUS_IO_ERROR_TYPE_IO_CANCELLED))
	{
            /* just end */
	}
        else
	{
            save_errno =  globus_i_nexus_get_errno(&result);
            if (save_errno == EINTR)
            {
                result = globus_io_udp_register_recvfrom(
  	                      &incoming->handle,
	                      incoming->current,
	                      UDP_MAX_TRANSMISSION_UNIT,
                              0,
    	                      nexusl_pr_udp_read_callback,
	                      (void *) incoming);
                assert (result == GLOBUS_SUCCESS);
            }
            else if (save_errno == EAGAIN || save_errno == EWOULDBLOCK)
            {
  	        done = NEXUS_TRUE;
            }
            else
            {
  	        nexusl_pr_udp_incoming_close(incoming);
	        udp_exit();
	        if (_nx_fault_detected(NEXUS_ERROR_READ_FAILED) != 0)
	        {
	            globus_fatal("udp_read_callback(): "
	    		     "Internal error: Read failed with errno=%i\n",
			     save_errno);
	        }
	        else
	        {
	           nexus_debug_printf(1,
			       ("nexusl_pr_udp_read_callback(): read "
				"failed, errno=%d\n", save_errno));
	        }
	        udp_enter();
            }
        }
    }
    
    udp_exit();
	
    nexus_debug_printf(1,("udp_read_callback(): exiting\n"));
}
/* nexusl_pr_udp_read_callback */


/*
 * nexusl_pr_udp_process_fragment()
 *
 * Assume udp_enter() has already been called before entering this function.
 */
static
void
nexusl_pr_udp_process_fragment(
    void *					arg,
    long					n_read)
{
    udp_incoming_t *				incoming;
    nexus_bool_t				done;
    struct globus_nexus_buffer_s *		buffer;
    nexus_bool_t				message_enqueued;
    unsigned					seq_num;
    unsigned					frag0;
    int						version;

    incoming = (udp_incoming_t *) arg;
    message_enqueued = NEXUS_FALSE;
    
    if (udp_done)
    {
	/* just return here, udp_close takes care of
	 * freeing the incomings and closing the fds
	 */
	return;
    }
    
    /* Check the udp header version */
    version = (int) (incoming->current[0]);

    if (version != GLOBUS_L_NEXUS_UDP_PROTOCOL_VERSION)
    {
	/*
	 * Got a udp protocol version mismatch.
	 * So kick out a fault, and then reset things so that
	 * we can possibly pick up future good packets.
	 */
	udp_exit();
	if (_nx_fault_detected(GLOBUS_NEXUS_ERROR_VERSION_MISMATCH) != 0)
	{
	    globus_fatal("nexusl_pr_udp_process_fragment(): "
			 "UDP protocol version mismatch\n");
	}
	udp_enter();
	incoming->current = incoming->storage;
	incoming->nbytes_read = 0;
	incoming->nbytes_parsed = 0;
	incoming->nbytes_unparsed = 0;
	incoming->expected_seq_num = 0;
	return;
    }

    /* Get the message number and sequence number from the header */
    seq_num =
	( ( ((unsigned) incoming->current[1]) & 0x7f) << 16) |
	( ( ((unsigned) incoming->current[2]) & 0xff) << 8) |
	( ( ((unsigned) incoming->current[3]) & 0xff ));

    frag0 = ((unsigned) incoming->current[1]) & 0x80;
    
    nexus_debug_printf(1,("process_fragment(): frag0[%u] seq_num[%u]\n",
			  frag0, seq_num));

    if(frag0) 
    {
	/*
	 * If this is the first packet of the message, then process the message
	 * header and prepare to receive any additional packets which are part
	 * of this message
	 */
	
	int					sizeof_u_long;
	unsigned long				tmp_u_long;
 
	/*
	 * It's possible that current is poiting somewhere in the middle of
	 * incoming->storage since we could have been processing an older
	 * message and lost some packets, so we must move the contents to the
	 * beginning of storage.  Since we overlap the header with data in the
	 * last packet, there may be a small amount of overlap.  Use memmove to
	 * safely move overlapped region.
	 */
	if (incoming->current != incoming->storage)
	{
	    memmove(incoming->storage, incoming->current, n_read);
	    incoming->current = incoming->storage;
	}
	
	incoming->expected_seq_num = seq_num++;
	incoming->nbytes_read = n_read - UDP_HDR_SIZE;
	incoming->nbytes_parsed = 0;
	incoming->nbytes_unparsed = n_read - UDP_HDR_SIZE;
	incoming->current += UDP_HDR_SIZE;

	/*
	 * get the buffer version and check it
	 */
	version = (int) *(incoming->current);
	incoming->current++;
	incoming->nbytes_parsed++;
	incoming->nbytes_unparsed--;
	if (version != GLOBUS_L_NEXUS_UDP_PROTOCOL_VERSION)
	{
	    /*
	     * Got a buffer version mismatch.
	     * So kick out a fault, and then reset things so that
	     * we can possibly pick up future good packets.
	     */
	    udp_exit();
	    if (_nx_fault_detected(GLOBUS_NEXUS_ERROR_VERSION_MISMATCH) != 0)
	    {
		globus_fatal("nexusl_pr_udp_process_fragment(): "
			    "Buffer version mismatch\n");
	    }
	    udp_enter();
	    incoming->expected_seq_num = seq_num;
	    incoming->current = incoming->storage;
	    incoming->nbytes_read = 0;
	    incoming->nbytes_parsed = 0;
	    incoming->nbytes_unparsed = 0;
	    return;
	}
	
	/* get the format */
	incoming->format = (int) *(incoming->current);
	incoming->current++;
	incoming->nbytes_parsed++;
	incoming->nbytes_unparsed--;

	/* Get the message size */
	sizeof_u_long = nexus_dc_sizeof_remote_u_long(1, incoming->format);
	/* Get the message size */
	nexus_dc_get_u_long(&(incoming->current),
			    &(incoming->msg_size),
			    1,
			    incoming->format);

	/* don't increment current, it was moved in nexus_dc_get_u_long */
	incoming->nbytes_parsed += sizeof_u_long;
	incoming->nbytes_unparsed -= sizeof_u_long;

	/*
	 * skip past the data in preparation for receiving any additional
	 * fragments
	 */
	incoming->current += incoming->nbytes_unparsed;
	incoming->nbytes_parsed += incoming->nbytes_unparsed;
	incoming->nbytes_unparsed = 0;
	
	if (incoming->msg_size > incoming->storage_size)
	{
	    /*
	     * The message is too big to fit in this storage.
	     * So allocate new storage that is big enough,
	     * copy the message into the new storage.
	     */
	    nexus_debug_printf(4, ("process_fragment(): msg too big\n"));
	    incoming->save_storage_size = incoming->storage_size;
	    incoming->save_storage = incoming->storage;

	    /* Allocate new storage */
	    incoming->storage_size = incoming->msg_size + UDP_HDR_SIZE +
		UDP_MAX_TRANSMISSION_UNIT - 1;
	    incoming->storage_size =
		(incoming->storage_size / UDP_MAX_TRANSMISSION_UNIT) *
		UDP_MAX_TRANSMISSION_UNIT;

	    NexusMalloc(process_fragment(),
			incoming->storage,
			nexus_byte_t *,
			incoming->storage_size);

	    /* Copy message to new storage */
	    tmp_u_long = incoming->nbytes_parsed + UDP_HDR_SIZE;
	    memcpy(incoming->storage,
		   incoming->current - tmp_u_long,
		   tmp_u_long);
	    incoming->current = incoming->storage + tmp_u_long;
	}
    }
    else
    {
	/*
	 * this packet is part of a multi-packet message...
	 */
	
	if( seq_num != incoming->expected_seq_num)
	{
	    /*
	     * but it wasn't the packet we were expecting
	     */
	    nexus_debug_printf(1, ("nexusl_pr_udp_process_fragment(): "
				   "out-of-sequence packet/msg received\n"));
	    nexus_debug_printf(1, ("expected_seq_num[%d] seq_num[%d]\n",
				   incoming->expected_seq_num, seq_num));

	    /*
	     * drop packets until we get the start of the next message.  this
	     * is accomplished by setting the expected sequence number to the
	     * sequence number of the packet we just received.
	     */
	    incoming->expected_seq_num = seq_num;
	    incoming->current = incoming->storage;    
	    incoming->nbytes_read = 0;    
	    incoming->nbytes_parsed = 0;
	    incoming->nbytes_unparsed = 0;
	    return;
	}
	else
	{
	    /*
	     * and it's the packet we expected, so copy the saved data that was
	     * stomped by the header back into the appropriate location
	     */
	    memcpy(incoming->current, incoming->tmp_storage, UDP_HDR_SIZE);
	    incoming->current += n_read;
	    incoming->expected_seq_num = seq_num++;
	    incoming->nbytes_read += n_read - UDP_HDR_SIZE;
	    incoming->nbytes_parsed += n_read - UDP_HDR_SIZE;
	    incoming->nbytes_unparsed = 0;
	}
	
    }

    nexus_debug_printf(1, ("process_fragment(): seq[%d] np[%d] ms[%d]\n",
		 seq_num,
		 incoming->nbytes_parsed,
		 incoming->msg_size));
    

    if ((incoming->nbytes_parsed) < incoming->msg_size)
    {
	/*
	 * We do not yet have all the fragments for this message.
	 * So save off the last UDP_HDR_SIZE bytes of the buffer,
	 * and setup incoming to expect another fragment and return.
	 */
	incoming->current -= UDP_HDR_SIZE;
	memcpy(incoming->tmp_storage, incoming->current, UDP_HDR_SIZE);
    }
    else
    {
	/*
	 * We have a complete message.
	 * So dispatch it, and reset incoming for next message.
	 */
	nexus_debug_printf(5, ("process_fragment():got all the fragments\n"));
	
	/*
	 * Enqueue the message for dispatch, there is a header
	 * at the beginning of incoming->storage, so be careful
	 * to pass the correct address when creating the buffer
	 */
	_nx_buffer_create_from_raw(incoming->storage,
				   incoming->storage_size,
				   UDP_HDR_SIZE,
				   incoming->msg_size,
				   incoming->endpoint,
				   &buffer);
	Enqueue(incoming->dispatch_q_head,
		incoming->dispatch_q_tail,
		buffer);
	message_enqueued = NEXUS_TRUE;
		
	if (incoming->save_storage_size > 0)
	{
	    /* There is a buffer in the save_storage to use */
	    incoming->storage_size = incoming->save_storage_size;
	    incoming->storage = incoming->save_storage;
	    incoming->save_storage_size = 0;
	}
	else
	{
	    /* Must allocate new storage */
	    incoming->storage_size = UDP_MAX_TRANSMISSION_UNIT;
	    NexusMalloc(process_fragment(),
			incoming->storage,
			nexus_byte_t *,
			UDP_MAX_TRANSMISSION_UNIT);
	}

	incoming->current = incoming->storage;
	incoming->nbytes_read = 0;
	incoming->nbytes_parsed = 0;
	incoming->nbytes_unparsed = 0;
	incoming->expected_seq_num = 0;
    }
	    
    if (message_enqueued && !incoming->dispatch_in_progress)
    {
	incoming->dispatch_in_progress = NEXUS_TRUE;
	while (QueueNotEmpty(incoming->dispatch_q_head))
	{
	    Dequeue(incoming->dispatch_q_head,
		    incoming->dispatch_q_tail,
		    buffer);
	    udp_exit();
	    _nx_buffer_dispatch(buffer);
	    udp_enter();
	}
	incoming->dispatch_in_progress = NEXUS_FALSE;
    }
}
/* nexusl_pr_udp_process_fragment() */



#ifdef BUILD_RESOURCE

/*
 * udp_get_resource_name_sp()
 *
 * Return a pointer to the resource_name of the socket with the passed
 * startpoint.
 */

nexus_resource_name_t *udp_get_resource_name_sp(nexus_startpoint_t *sp) {
  nexus_resource_name_t * result;

  if (sp == NULL || sp->mi_proto == NULL || sp->mi_proto->proto == NULL ) {
    result = NULL;
  } else {
    udp_outgoing_t *outgoing = (udp_outgoing_t*)sp->mi_proto->proto;
    result = &outgoing->resource_name;
  }  

  return (result);

} /* udp_get_resource_name_sp() */

/*
 * udp_get_resource_name_ep()
 *
 * Return a pointer to the resource_name of the socket with the passed 
 * endpoint.
 */

nexus_resource_name_t *udp_get_resource_name_ep(nexus_endpoint_t *ep) {
  nexus_resource_name_t * result = NULL;

  if (ep == NULL ) {
    result = NULL;
  } else {
    int i;
    for (i = 0; (i < fd_tablesize) && (result == NULL); i++) {
      if (fd_table[i].type == FD_TABLE_TYPE_INCOMING) {
	udp_incoming_t *incoming = fd_table[i].value;
	if (incoming->endpoint == ep) {
	  result = &incoming->resource_name;
	}
      }
    }
  }
  return (result);

} /* udp_get_resource_name_ep() */


/*
 * udp_lookup_local_addr()
 * 
 * Returns the local sockaddr bound to udp connections.
 */

static unsigned int udp_lookup_local_addr() {

  char *interface;
  struct hostent *hp;
  struct hostent he;
  char buffer[500];
  int err;

  interface = nexus_rdb_lookup(udp_local_host, "udp_interface");
  if (interface == GLOBUS_NULL)
  {
      interface = _nx_copy_string(udp_local_host);
  }
  
  hp = globus_libc_gethostbyname_r(interface,
				   &he,
				   buffer,
				   500,
				   &err);

  NexusFree(interface);
  
  if (hp == 0) return (0);
  return *((unsigned int **)hp->h_addr_list)[0];

} /* udp_lookup_local_addr() */


#endif /* BUILD_RESOURCE */


#endif /* HAVE_UDP_PROTO */
