/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * context.c
 *
 * Code for maintaining contexts.
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-packages/nexus/source/nexus/context.c,v 1.93 2006/01/19 05:57:05 mlink Exp $";

#include "internal.h"

static int		n_contexts;
static int		next_context_id;
static nexus_mutex_t	context_mutex;

static void finish_destroy_current_context(int called_from_non_threaded_handler,
					   nexus_context_t *context);


/*
 * _nx_context_new_process_params()
 */
int _nx_context_new_process_params(char *buf, int size)
{
    return (0);
} /* _nx_context_new_process_params() */


/*
 * _nx_context_init()
 *
 * Initialize the context module.
 */
void _nx_context_init(void)
{
    n_contexts = 0;
    next_context_id = 0;
    _nx_context_list = NULL;
    nexus_mutex_init(&context_mutex, (nexus_mutexattr_t *) NULL);
    globus_thread_key_create(&_nx_context_key, NULL);

} /* _nx_context_init() */


/*
 * _nx_context_alloc()
 *
 * Initialize my context data structures.
 *
 * Note: When assigning a context id for this context, this routine
 * assumes once context per process.  If that assumption is not valid,
 * then the assignment of context ids must become more sophisticated.
 */
nexus_context_t *_nx_context_alloc()
{
    nexus_context_t *context;
    nexus_segment_t *dummy_segment;
    
    NexusMalloc(_nx_context_alloc(),
		context,
		nexus_context_t *,
		sizeof(nexus_context_t) );
    
    nexus_mutex_init(&(context->mutex), (nexus_mutexattr_t *) NULL);

    nexus_mutex_lock(&context_mutex);
    
    context->context_list_next = _nx_context_list;
    _nx_context_list = context;

    n_contexts++;
    context->id = next_context_id++;

    nexus_mutex_unlock(&context_mutex);
    
#ifdef NEXUS_CONTEXT_CREATE_LOCAL
    context->handle = NULL;
    context->switchingfunc = NULL;
    context->destructofunc = NULL;
    context->pfnNexusBoot = NULL;
#endif /* NEXUS_CONTEXT_CREATE_LOCAL */

    /* First segment of segment list is a dummy */
    NexusMalloc(_nx_context_alloc(),
		dummy_segment,
		struct globus_nexus_segment_s *,
		sizeof(struct globus_nexus_segment_s));
    dummy_segment->data = (void *) NULL;
    dummy_segment->size = 0;
    dummy_segment->context = context;
    context->segment_list = dummy_segment;

    context->n_segments = 0;
    dummy_segment->next = dummy_segment;
    dummy_segment->prev = dummy_segment;
    
    /*
     * Stow away the context pointer in thread specific storage
     * so that we can get it easily later.
     */
    _nx_set_context(context);
    
    return (context);
} /* _nx_context_alloc() */


#ifdef NEXUS_CONTEXT_CREATE_LOCAL
/*
 * nexus_context_create_local()
 */
void nexus_context_create_local()
{
    nexus_startpoint_t reply_sp;
    int checkin_number;
    int executable_path_length;
    int default_executable_path_length;
    char executable_path[NEXUS_MAX_EXECUTABLE_PATH_LENGTH + 1];
    char default_executable_path[NEXUS_MAX_EXECUTABLE_PATH_LENGTH + 1];
    char *path;
    new_context_reply_info_t *new_context_reply_info;
    nexus_startpoint_t new_context_reply_info_sp;
    nexus_context_t *this_context;
    nexus_context_t *new_local_context;
    int failure_rc;

    /*
     * Extract the contexts of the buffer
     */
    nexus_get_int(buffer, &executable_path_length, 1);
    if (executable_path_length >= 0)
    {
	nexus_get_char(buffer, executable_path, executable_path_length);
	executable_path[executable_path_length] = '\0';
    }
    nexus_get_int(buffer, &default_executable_path_length, 1);
    nexus_get_char(buffer,
		   default_executable_path,
		   default_executable_path_length);
    default_executable_path[default_executable_path_length] = '\0';

    nexus_get_startpoint(buffer, &reply_sp, 1);
    nexus_get_int(buffer, &checkin_number, 1);

    if (executable_path_length == -1)
    {
	/* Use default executable path */
	strcpy(executable_path, default_executable_path);
    }
    else if (   (strncmp(executable_path,
			 NEXUS_DATABASE_PREFIX,
			 NEXUS_DATABASE_PREFIX_SIZE) == 0)
	     && (executable_path[NEXUS_DATABASE_PREFIX_SIZE] == ':') )
    {
	/* Lookup the path in the database */
	char *key = executable_path + NEXUS_DATABASE_PREFIX_SIZE + 1;

	if (   (*key == '\0')
	    || ((path = nexus_rdb_lookup(_nx_my_hostname,
					 key)) == (char *) NULL) )
	{
	    /* Empty key or the lookup did not find anything */
	    strcpy(executable_path, default_executable_path);
	}
	else
	{
	    /* Use the path returned from the database */
	    NexusAssert2((strlen(path) <= NEXUS_MAX_EXECUTABLE_PATH_LENGTH),
			 ("_nx_context_create_handler(): executable path from database, %s, for host %s and key %s, is too long. Maximum length is %d\n",
			  path, _nx_my_hostname, key,
			  NEXUS_MAX_EXECUTABLE_PATH_LENGTH) );
	    strcpy(executable_path, path);
	    nexus_rdb_free(path);
	}
    }
    /* else use the executable_path as it is */
    
    nexus_debug_printf(1, ("_nx_context_create_handler(): Creating:[%s] with checkin number:%d\n", executable_path, checkin_number));
    
    _nx_context(&this_context);

    /*
     * Startup the new context
     */
    new_local_context = NULL;
    nexus_bool_t done = NEXUS_FALSE;
    int return_code;
    int access_rc;
    startup_module_list_t *startup_module;
    
    /*
     * Test that the executable_path is executable by this user
     */
#if !defined(HAVE_X_ACC) && defined(HAVE_X_OK)
#define X_ACC X_OK
#endif
    if ((access_rc = access(executable_path, X_ACC)) != 0)
    {
#ifdef BUILD_DEBUG
	int access_error;
	access_error = errno;
	nexus_debug_printf(1, ("_nx_startup_context(): Cannot execute %s, access() failed, rc=%d, errno=%d\n", executable_path, access_rc, access_error) );
#endif
	/* Error code: Cannot access the executable_path */
	return(-1);
    }

    done = _nx_context_create_local(executable_path,
				    &return_code,
				    new_local_context);
    if (!done)
    {
	/* Error code: No startup module can create a context */
	nexus_debug_printf(1, ("_nx_startup_context(): no startup module can create a context\n"));
	return_code = -2;
    }
    
    failure_rc = _nx_startup_context(executable_path,
				     &new_local_context);
    if (failure_rc != 0)
    {
	return;
    }

    if (new_local_context)
    {
	/*
	 * _nx_startup_context() created a context that
	 * lives in this same process.  This requires
	 * a little extra work.
	 */
	int nexus_boot_rc;
	nexus_startpoint_t context_sp;
	
	_nx_set_context(new_local_context);
	
	/*
	 * Call NexusBoot(), since there is no new process
	 * in which to call nexus_start().
	 */
	nexus_startpoint_set_null(&context_sp);
	nexus_debug_printf(1, ("create_context_locally(): calling NexusBoot()=%lu\n", (unsigned long) new_local_context->pfnNexusBoot));
	nexus_boot_rc = (*new_local_context->pfnNexusBoot)(&context_sp);
	nexus_debug_printf(1, ("create_context_locally(): NexusBoot() returned %d\n", nexus_boot_rc));

	_nx_set_context(this_context);
    }

} /* nexus_context_create_local() */


/*
 * nexus_context_destroy()
 *
 * Terminate the current context
 */
void nexus_context_destroy(int called_from_non_threaded_handler)
{
    nexus_context_t *context;
    nexus_segment_t *dummy_segment, *segment, *next_segment;

    _nx_context(&context);

    nexus_debug_printf(1,("nexus_context_destroy(): initiating context destruction, called_from_non_threaded_handler=%d\n", called_from_non_threaded_handler));

    nexus_mutex_destroy(&(context->mutex));

    /* The first segment is a dummy.  The segment_list is circular. */
    dummy_segment = context->segment_list;
    segment = dummy_segment->next;
    while (segment != dummy_segment)
    {
	next_segment = segment->next;
	NexusFree(segment);
	segment = next_segment;
    }
    NexusFree(dummy_segment);
    
    if (context->destructofunc != (int (*)(nexus_context_t*))(NULL) )
    { 
	int rc;
	rc = (*context->destructofunc)(context);
    }

    NexusFree(context);
    
    nexus_mutex_lock(&context_mutex);
    n_contexts--;
    nexus_mutex_unlock(&context_mutex);
    
    if (n_contexts <= 0)
    {
	/* There are no more contexts in this process, so shut it down. */
	nexus_shutdown();
	exit(0);
    }

    nexus_debug_printf(2, ("nexus_context_destroy(): returning\n"));
    
} /* nexus_context_destroy() */
#endif /* NEXUS_CONTEXT_CREATE_LOCAL */
