/* giFTui
 * Copyright (C) 2003 the giFTui team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "main.h"

#include <stdio.h>
#include <string.h>
#include <libgift/libgift.h>
#include <gtk/gtk.h>

#include "event.h"
#include "configure.h"
#include "util.h"

#include "ui.h"
#include "ui_util.h"
#include "ui_icon.h"
#include "ui_search_cb.h"

/*extern gchar **search_network;*/

static gboolean
_check_if_present (GtkTreeModel *model, GtkTreeIter *parent, const gchar *user,
		   const gchar *url)
{
	gboolean ret = FALSE;
	gchar *stored_user;
	GtkTreeIter iter;
	
	if (!url)
		return ret;
	
	ret = gtk_tree_model_iter_children (model, &iter, parent);
	
	while (ret)
	{
		gtk_tree_model_get (model, &iter, SEARCH_FILENAME, &stored_user, -1);
		if (!strcmp (stored_user, user))
		{
			gchar *stored_url;
			
			gtk_tree_model_get (model, &iter, SEARCH_URL, &stored_url, -1);
			
			if (stored_url)
			{
				if (!strcmp (stored_url, url))
				{
					g_free (stored_user);
					g_free (stored_url);
					
					return TRUE;
				}
				g_free (stored_url);
			}
		}
		g_free (stored_user);
		ret = gtk_tree_model_iter_next (model, &iter);
	}
	
	return FALSE;
}

static gboolean
_search_file (GtkTreeModel *model, GtkTreeIter *parent, GtkTreeIter *master,
	      gulong filesize, const gchar *hash)
{
	gboolean ret = FALSE;
	gulong stored_size;
	
	if (!hash)
		return ret;
	
	ret = gtk_tree_model_iter_children (model, parent, master);
	
	while (ret)
	{
		gtk_tree_model_get (model, parent, SEARCH_FILESIZE, &stored_size, -1);
		if (stored_size == filesize)
		{
			gchar *stored_hash;
			
			gtk_tree_model_get (model, parent, SEARCH_HASH, &stored_hash, -1);
			
			if (stored_hash)
			{
				if (!strcmp (stored_hash, hash))
				{
					g_free (stored_hash);
					
					return ret;
				}
				g_free (stored_hash);
			}
		}
		ret = gtk_tree_model_iter_next (model, parent);
	}
	
	return ret;
}

static gboolean
_search_master (GtkTreeModel *model, GtkTreeIter *iter, guint id)
{
	gboolean ret;
	guint stored_id;
	
	ret = gtk_tree_model_get_iter_first (model, iter);
	
	while (ret)
	{
		gtk_tree_model_get (model, iter, SEARCH_ID, &stored_id, -1);
		if (stored_id == id)
			return TRUE;
		ret = gtk_tree_model_iter_next (model, iter);
	}
	
	return FALSE;
}

static void
_sources_inc (GtkTreeModel *model, GtkTreeIter *iter)
{
	guint sources;
	
	gtk_tree_model_get (model, iter, SEARCH_SOURCES, &sources, -1);
	gtk_tree_store_set (GTK_TREE_STORE (model), iter, SEARCH_SOURCES, sources + 1, -1);
	
	return;
}

/**/

static void
meta_print (Interface *iface, InterfaceNode *inode, void *data)
{
	if (inode->key && inode->value)
		printf ("%s %s\n", inode->key, inode->value);
}

/**/

void
giftui_search_add_result (GiftuiSearch *se, GiftuiEvent_t *in)
{
	/* Displayed strings & variables */
	gboolean availability;
	gulong filesize;
	gchar *user, *filename, *size;
	/* Others */
	gchar *hash, *url, *dir, *tmp, *icon;
      	GtkTreeModel *model;
	GtkTreeIter master, parent, child;
	
	g_return_if_fail (se != NULL);
	g_return_if_fail (in != NULL);
	
	/* search id */
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (se->list));
	if (!_search_master (model, &master, in->id))
		return;
	
	/* search is ended ? */
	if ((user = interface_get (in->iface, "user")) == NULL)
	{
		gchar search_text[1024];
		
		gtk_tree_model_get (model, &master, SEARCH_FILENAME, &tmp, -1);
		strcpy (search_text, tmp);
		g_free (tmp);
		
		if ((tmp = strrchr (search_text, '/')))
		{
			*tmp = '\0';
			strncat (search_text, _("/ Completed"), 1023);
		}
		
		gtk_tree_store_set (GTK_TREE_STORE (model), &master,
				    SEARCH_FILENAME, search_text, SEARCH_ID, 0, -1);
		giftui_event_unregister (se, EVENT_ITEM, in->id);
		
		return;
	}
	availability = (INTERFACE_GETI (in->iface, "availability") > 0) ? TRUE : FALSE;
	filesize = INTERFACE_GETLU (in->iface, "size");
	size = size_str_human (filesize);
	dir = interface_get (in->iface, "file");
	
	if ((tmp = strrchr (dir, '/')) == NULL)
		filename = g_strdup (dir);
	else
		filename = g_strdup (tmp + 1);
	url = interface_get (in->iface, "url");
	hash = interface_get (in->iface, "hash");
	
	if ((tmp = network_name_from_url (url)))
	{
		icon = giftui_icon_stock (ICON_NETWORK, tmp);
		g_free (tmp);
	}
	else
		icon = NULL;
	
	/*interface_foreach (in->iface, "META", meta_print, se);*/
	
	/* search file */
	if (_search_file (model, &parent, &master, filesize, hash))
	{
		/* already added ? */
		if (!_check_if_present (model, &parent, user, url))
		{
			_sources_inc (model, &parent);
			gtk_tree_store_append (GTK_TREE_STORE (model), &child, &parent);
			gtk_tree_store_set (GTK_TREE_STORE (model), &child,
					    SEARCH_FILENAME, user, SEARCH_URL, url,
					    SEARCH_AVAILABILITY, availability,
					    SEARCH_SOURCES, 1, SEARCH_ICON, icon, -1);
			giftui_child_set_highlight (GIFTUI_CHILD (se), TRUE);
		}
	}
	else
	{
		_sources_inc (model, &master);
		gtk_tree_store_append (GTK_TREE_STORE (model), &parent, &master);
		gtk_tree_store_set (GTK_TREE_STORE (model), &parent, SEARCH_FILENAME, filename,
				    SEARCH_HASH, hash, SEARCH_FILESIZE, filesize, SEARCH_SOURCES, 1,
				    SEARCH_SIZE, size, SEARCH_ICON, GTK_STOCK_NEW, -1);
		gtk_tree_store_append (GTK_TREE_STORE (model), &child, &parent);
		gtk_tree_store_set (GTK_TREE_STORE (model), &child, SEARCH_SOURCES, 1,
				    SEARCH_FILENAME, user, SEARCH_AVAILABILITY, availability,
				    SEARCH_URL, url, SEARCH_SOURCES, 1,
				    SEARCH_ICON, icon, -1);
		giftui_child_set_highlight (GIFTUI_CHILD (se), TRUE);
	}
	
	if (availability)
		gtk_tree_store_set (GTK_TREE_STORE (model), &parent,
				    SEARCH_AVAILABILITY, availability, -1);
	
	if (icon)
		g_free (icon);
	g_free (size);
	g_free (filename);
	
	return;
}

/**/

void
giftui_search_update_networks (GiftuiSearch *se)
{
	gchar **str;
	GArray *array;
	GtkWidget *menu;
	
	g_return_if_fail (se != NULL);
	
	menu = widget_menu_add (NULL);
	widget_menu_item_add (GTK_MENU (menu), _("all networks"));
	
	if ((array = giftui_data_get (PREFS_SEARCH_NETWORKS)))
	{
		str = (gchar **) array->data;
		
		while (*str)
		{
			widget_menu_item_add (GTK_MENU (menu), *str);
			str++;
		}
	}
	
	if (gtk_option_menu_get_menu (GTK_OPTION_MENU (se->network)))
		gtk_option_menu_remove_menu (GTK_OPTION_MENU (se->network));
	gtk_option_menu_set_menu (GTK_OPTION_MENU (se->network), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (se->network), 0);
	
	return;
}
