/**********************************************************************
 * $Id: WKBReader.h 2131 2008-07-15 22:04:51Z mloskot $
 *
 * GEOS - Geometry Engine Open Source
 * http://geos.refractions.net
 *
 * Copyright (C) 2005-2006 Refractions Research Inc.
 * Copyright (C) 2001-2002 Vivid Solutions Inc.
 *
 * This is free software; you can redistribute and/or modify it under
 * the terms of the GNU Lesser General Public Licence as published
 * by the Free Software Foundation. 
 * See the COPYING file for more information.
 *
 **********************************************************************/

#ifndef GEOS_IO_WKBREADER_H
#define GEOS_IO_WKBREADER_H

#include <geos/geom/GeometryFactory.h>
#include <geos/io/ByteOrderDataInStream.h> // for composition

#include <iosfwd> // ostream, istream
#include <vector>
#include <string>

// Forward declarations
namespace geos {
	namespace geom {

		//class GeometryFactory;
		class Coordinate;
		class Geometry;
		class GeometryCollection;
		class Point;
		class LineString;
		class LinearRing;
		class Polygon;
		class MultiPoint;
		class MultiLineString;
		class MultiPolygon;
		class PrecisionModel;

	} // namespace geom
} // namespace geos


namespace geos {
namespace io {

/**
 * \class WKBReader io.h geos.h
 *
 * \brief Reads a Geometry from Well-Known Binary format.
 *
 * This class is designed to support reuse of a single instance to read
 * multiple geometries. This class is not thread-safe; each thread should
 * create its own instance.
 *
 * The Well-known Binary format is defined in the <A
 * HREF="http://www.opengis.org/techno/specs.htm">OpenGIS Simple Features
 * Specification for SQL</A>.
 * This implementation supports the extended WKB standard which allows
 * representing 3-dimensional coordinates.
 *
 */
class WKBReader {

public:

	WKBReader(geom::GeometryFactory const& f): factory(f) {};

	/// Inizialize parser with default GeometryFactory.
	WKBReader();

	/**
	 * \brief Reads a Geometry from an istream.
	 *
	 * @param is the stream to read from 
	 * @return the Geometry read
	 * @throws IOException
	 * @throws ParseException
	 */
	geom::Geometry* read(std::istream &is);
		// throws IOException, ParseException

	/**
	 * \brief Reads a Geometry from an istream in hex format.
	 *
	 * @param is the stream to read from 
	 * @return the Geometry read
	 * @throws IOException
	 * @throws ParseException
	 */
	geom::Geometry *readHEX(std::istream &is);
		// throws IOException, ParseException

	/**
	 * \brief Print WKB in HEX form to out stream
	 *
	 * @param is is the stream to read from
	 * @param os is the stream to write to
	 */
	static std::ostream &printHEX(std::istream &is, std::ostream &os);
 
private:

	static std::string BAD_GEOM_TYPE_MSG;

	const geom::GeometryFactory &factory;

	// for now support the WKB standard only - may be generalized later
	unsigned int inputDimension;

	ByteOrderDataInStream dis;

	std::vector<double> ordValues;

	geom::Geometry *readGeometry();
		// throws IOException, ParseException

	geom::Point *readPoint();
		// throws IOException

	geom::LineString *readLineString();
		// throws IOException

	geom::LinearRing *readLinearRing();
		// throws IOException

	geom::Polygon *readPolygon();
		// throws IOException

	geom::MultiPoint *readMultiPoint();
		// throws IOException, ParseException

	geom::MultiLineString *readMultiLineString();
		// throws IOException, ParseException

	geom::MultiPolygon *readMultiPolygon();
		// throws IOException, ParseException

	geom::GeometryCollection *readGeometryCollection();
		// throws IOException, ParseException

	geom::CoordinateSequence *readCoordinateSequence(int); // throws IOException

	void readCoordinate(); // throws IOException


};

} // namespace io
} // namespace geos


#endif // #ifndef GEOS_IO_WKBREADER_H

/**********************************************************************
 * $Log$
 * Revision 1.3  2006/06/01 11:49:36  strk
 * Reduced installed headers form geomgraph namespace
 *
 * Revision 1.2  2006/04/12 10:57:19  strk
 * Added WKBReader default ctor using default GeometryFactory instance
 *
 * Revision 1.1  2006/03/20 18:18:14  strk
 * io.h header split
 *
 **********************************************************************/
