/*
** 1998-05-30 -	Implementation of the built-in MOVE command, which is used to, er, move
**		files and directories. It might become incredibly involved. We'll see
**		about that.
** 1998-09-12 -	Did changes in command arguments to ease implementation of MOVEAS, also
**		exported all the move_XXX() functions, as the cmd_copy module does.
** 1999-03-06 -	Adapted for new selection/dirrow representations.
*/

#include "gentoo.h"

#include "cmd_copy.h"
#include "cmd_delete.h"
#include "convutil.h"
#include "dialog.h"
#include "dirpane.h"
#include "errors.h"
#include "fileutil.h"
#include "overwrite.h"
#include "progress.h"

#include "cmd_move.h"

#define	CMD_ID	"move"

/* ----------------------------------------------------------------------------------------- */

/* 2003-11-24 -	Move a "file", i.e. anything that is not a directory. */
gint move_file(MainInfo *min, const gchar *from, const gchar *to, const struct stat *sstat)
{
	/* Check overwrite status. */
	switch(ovw_overwrite_file(min, to, from))
	{
	case OVW_SKIP:
		return TRUE;
	case OVW_CANCEL:
		return FALSE;
	case OVW_PROCEED:
		break;
	case OVW_PROCEED_FILE:
		del_delete_file(min, to);
		break;
	case OVW_PROCEED_DIR:
		del_delete_dir(min, to, FALSE);
		break;
	}
	if(errno != 0)
		return FALSE;
	/* Try a quick (for same-device moves) rename(), otherwise copy. */
	if(rename(from, to) == 0)
		return TRUE;
	else if(copy_file(min, from, to, sstat))	/* If no rename, copy and unlink. */
	{
		unlink(from);
		return errno == 0;
	}
	return FALSE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-06-07 -	Move a directory full of stuff. Does not simply call copy_dir from the cmd_copy module,
**		since that would require also calling del_delete_dir() (from cmd_delete). I feel that
**		is just too inefficient, having to traverse twice.
** 1998-09-12 -	Modified: now <to> is the complete destination name, since this is the new deal.
*/
gboolean move_dir(MainInfo *min, const gchar *from, const gchar *to)
{
	struct stat	stat;
	gchar		old_dir[PATH_MAX];
	gchar		dest[PATH_MAX];
	Conv		conv;

	/* Do safety checking before removing existing destination directory. Bad if nested. */
	if(fut_is_parent_of(from, to))
	{
		gchar	ebuf[2 * PATH_MAX + 128], *fromd, *tod;

		fromd = conv_begin_reverse(&conv, from);
		tod = conv_add_reverse(&conv, to);
		g_snprintf(ebuf, sizeof ebuf, _("Can't move directory \"%s\"\nto \"%s\":\nthe source contains the destination."), fromd, tod);
		dlg_dialog_async_new_error(ebuf);
		conv_end(&conv);
		return FALSE;
	}
	if(fut_is_parent_of(to, from))
	{
		gchar	ebuf[2 * PATH_MAX + 128], *fromd, *tod;

		fromd = conv_begin_reverse(&conv, from);
		tod = conv_add_reverse(&conv, to);
		g_snprintf(ebuf, sizeof ebuf, _("Can't move directory \"%s\"\nto \"%s\":\nthe destination contains the source."), fromd, tod);
		dlg_dialog_async_new_error(ebuf);
		conv_end(&conv);
		return FALSE;
	}

	/* Investigate toplevel overwrite status. */
	switch(ovw_overwrite_file(min, to, from))
	{
	case OVW_SKIP:
		return TRUE;
	case OVW_CANCEL:
		return FALSE;
	case OVW_PROCEED:
		break;
	case OVW_PROCEED_FILE:
		del_delete_file(min, to);
		break;
	case OVW_PROCEED_DIR:
		del_delete_dir(min, to, FALSE);
		break;
	}
	
	/* Try a quick rename. */
	if(rename(from, to) == 0)
		return TRUE;
	else if(errno != EXDEV)
	{
		err_set(min, errno, CMD_ID, from);
		return FALSE;
	}

	/* Get stat, enter and move all items, recursively. */
	if(lstat(from, &stat) != 0)
	{
		err_set(min, errno, CMD_ID, from);
		return 0;
	}
	if(fut_cd(from, old_dir, sizeof old_dir))
	{
		if(mkdir(to, stat.st_mode | S_IWUSR) == 0)
		{
			DIR	*dir;

			if((dir = opendir(".")) != NULL)
			{
				struct dirent	*de;

				errno = 0;
				while(!errno && (de = readdir(dir)) != NULL)
				{
					if(!min->cfg.dir_filter(de->d_name))
						continue;
					if(lstat(de->d_name, &stat) == 0)
					{
						g_snprintf(dest, sizeof dest, "%s/%s", to, de->d_name);
						if(S_ISDIR(stat.st_mode))
						{
							if(!move_dir(min, de->d_name, dest))
								break;
						}
						else
						{
							if(!move_file(min, de->d_name, dest, &stat))
								break;
						}
					}
				}
				closedir(dir);
			}
		}
		if(!errno && fut_cd(old_dir, NULL, 0))
			rmdir(from);
	}
	if(errno)
		err_set(min, errno, CMD_ID, from);
	return errno == 0;
}

/* ----------------------------------------------------------------------------------------- */

/* 2003-11-24 -	Move selected files and/or directories to destination pane's directory. */
gint cmd_move(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	gchar	old_path[PATH_MAX], dest[PATH_MAX];
	GSList	*slist, *iter;

	if((src == NULL) || (dst == NULL))
		return 1;
	if(!fut_cd(src->dir.path, old_path, sizeof old_path))
		return 0;

	if((slist = dp_get_selection(src)) == NULL)
		return 1;

	err_clear(min);
	ovw_overwrite_begin(min, _("\"%s\" Already Exists - Continue With Move?"), 0UL);
	pgs_progress_begin(min, _("Moving..."), PFLG_COUNT_RECURSIVE | PFLG_ITEM_VISIBLE | PFLG_BYTE_VISIBLE);
	for(iter = slist; !errno && (iter != NULL); iter = g_slist_next(iter))
	{
		g_snprintf(dest, sizeof dest, "%s%c%s", dst->dir.path, G_DIR_SEPARATOR, DP_SEL_NAME(iter));
		if(S_ISDIR(DP_SEL_LSTAT(iter).st_mode))
		{
			if(!move_dir(min, DP_SEL_NAME(iter), dest))
				break;
		}
		else
		{
			if(!move_file(min, DP_SEL_NAME(iter), dest, &DP_SEL_LSTAT(iter)))
				break;
		}
	}
	dp_rescan_post_cmd(src);
	dp_rescan_post_cmd(dst);
	if(errno && iter != NULL)
		err_set(min, errno, CMD_ID, dp_full_name(src, DP_SEL_INDEX(src, iter)));
	pgs_progress_end(min);
	ovw_overwrite_end(min);
	fut_cd(old_path, NULL, 0);
	dp_free_selection(slist);
	err_show(min);

	return errno == 0;
}
